<?php

namespace App\Http\Livewire\Sales\Pricing;

use App\Models\Price;
use App\Models\Sizes;
use App\Models\Styles;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Helper\Functions;
use App\Models\Suppliers;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use App\Models\style_versions;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;
use App\Models\CustomerOrderLineQuantities;

class PricingSheet extends FilterableComponent
{
	use WithPagination;
    use Filterable;

    public $hideFilters = false, $selected;

    #[Validate([
        'season' => ['required', 'exists:seasons,id'], // Assuming seasons start from 1
        'department' => ['nullable', 'exists:departments,id'], // Adjust table name if needed
        'customer' => ['nullable', 'exists:customers,id'],
        'factory' => ['nullable', 'exists:suppliers,id'],
        'coo' => ['nullable', 'exists:countries,id'], // Assuming 'coo' refers to country of origin
        'category' => ['nullable', 'in:ladies,mens,childrens,accessories'], // Adjust table name if needed
        'search' => ['nullable', 'string', 'max:30'],
        'cancelled' => ['boolean'], // Assuming it's a true/false flag
        'not_quoted' => ['boolean'], // Assuming it's a true/false flag
    ])]
    public
        $season = 1,
        $department,
        $customer = 1,
        $factory,
        $coo,
        $category,
        $search,
        $cancelled = 0,
        $not_quoted = 0;
    public
        $goTo;

    protected function filterKey(): string
    {
        return 'whpricingfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.1;
    }

    public $columns = [
        'Style' => TRUE,
        'Style Version' => TRUE,
        'Image' => TRUE,
        'Yarns' => TRUE,
        'Price' => TRUE,
        'Apply To' => TRUE,
        'Colour Type' => TRUE,
        'Colourway' => TRUE,
        'Yarn/Kg' => TRUE,
        'Yarn Transport' => TRUE,
        'Total Yarn Cost' => TRUE,
        'Sample Weight' => TRUE,
        'Gross Weight' => TRUE,
        'Yarn Value' => TRUE,
        'CMT' => TRUE,
        'CMT History' => TRUE,
        'Valid Till' => TRUE,
        'AOC (%)' => TRUE,
        'Accessories' => TRUE,
        'Labels' => TRUE,
        'Gmt Transport' => TRUE,
        'Testing' => TRUE,
        'Testing Courier' => TRUE,
        'Embroidery' => TRUE,
        'Duty' => TRUE,
        'Subtotal' => TRUE,
        'With Customs' => TRUE,
        'Subtotal £' => TRUE,
        'Quote' => TRUE,
        'Quotes Sent' => TRUE,
        'Quote History' => TRUE,
        'Target' => TRUE,
        'MOQ' => TRUE,
        'Internal Notes' => TRUE,
        'Customer Notes' => TRUE,
        'Share' => TRUE,
        'Last Update' => TRUE,
    ];

    protected function filters(): array
    {
        return ['season', 'department', 'customer', 'factory', 'coo', 'category', 'search', 'cancelled', 'not_quoted', 'columns'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['season', 'department', 'customer', 'factory', 'coo', 'category', 'search', 'cancelled', 'not_quoted'])) {
            }

            if (in_array($propertyName, ['cancelled', 'not_quoted'])) {
                $this->dispatch('update-filters', cancelled: $this->cancelled, not_quoted: $this->not_quoted);
            }


            // Save filter settings since validation passed
            $this->saveFilterSettings();
            $this->refreshAll();
        }

        if($propertyName == 'not_quoted'){
            $this->dispatch('toggleNotQuoted');
        }
    }

    public $chunks = [];

    protected $queryString = [
        'goTo' => ['except' => ''], // Automatically maps `goTo` from the URL
    ];

    public function mount()
    {
        $this->loadFilterSettings();

        $this->not_quoted = (bool) ($this->not_quoted ?? 0);
        $this->cancelled = (bool) ($this->cancelled ?? 0);

        $this->chunks = $this->loadPrices();
    }

    public function render()
    {
        // dd($this->not_quoted);
        return view('livewire.sales.pricing.pricing-sheet');
    }


    public $refreshKey = 0;

    #[On('refreshFullPage')]
    public function refreshAll()
    {
        $this->chunks = $this->loadPrices(); // Update chunks
        $this->refreshKey = now()->timestamp; // Generate a new unique key
    }

    public function getPriceInStyleCount(Styles $style){
        return $style->style_versions->flatMap(function ($version) {
            return $version->prices;
        })->count();
    }
    public function getPriceInVersionCount(style_versions $version){
        return $version->prices->count();
    }

    #[Computed(cache: true, key: 'factories-with-countries')]
    public function factories(){
        return Suppliers::with('countries')->where('type', 'factory')->get();
    }
    #[Computed]
    public function sizes(){
        return Sizes::get();
    }

    #[Computed]
    public function seasons(){
        return Seasons::orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function customers(){
        return Customer::all();
    }

    #[Computed]
    public function departments(){
        return Departments::all();
    }

    #[On('toggle-selected-price')]
    public function toggleSelected($id){
        if(empty($this->selected[$id])){
            $this->selected[$id] = TRUE;
        }
        else{
            unset($this->selected[$id]);
        }
    }

    public function customerPrint(){
        if(empty($this->selected)){
            session()->flash('message', 'Please select at least one price to quote.');
            session()->flash('alert-class', 'alert-warning');
            return;
        }
        $list = array_keys($this->selected);
        return redirect()->route('pricingcustomerprint', json_encode($list));
    }

    public function loadPrices()
    {
        // Set default values if not set
        if (empty($this->season)) {
            $this->season = 1;
        }
        if (empty($this->customer)) {
            $this->customer = 1;
        }

        $query = Styles::query();

        if (!empty($this->goTo)) {
            $query->where('styles.id', $this->goTo);
        } else {
            $query->where('customers_id', $this->customer)
                ->where('seasons_id', $this->season);
            if (!empty(!$this->cancelled)) {
                $query->where('styles.cancelled', 0);
            }
            if (!empty($this->factory)) {
                $query->whereHas('style_versions.factories', function ($q) {
                    $q->where('id', $this->factory);
                });
            }
            if (!empty($this->department)) {
                $query->where('departments_id', $this->department);
            }
            if (!empty($this->coo)) {
                $query->whereHas('style_versions.factories', function ($q) {
                    $q->where('countries_id', $this->coo);
                });
            }
            if (!empty($this->category)) {
                $query->where('category', $this->category);
            }
            if (!empty($this->search)) {
                $query->search($this->search, true);
            }
        }

        $styles = $query
            ->leftJoin('style_versions', function ($join) {
                $join->on('styles.id', '=', 'style_versions.styles_id')
                    ->whereNull('style_versions.deleted_at');       // Ignore deleted
                })
            ->orderByDesc('style_versions.factory_id')
            ->distinct()
            ->pluck('styles.id')
            ->chunk(10)
            ->toArray();

        return $styles;
    }

    public int $page = 1;
    public function hasMorePages()
    {
        return $this->page < count($this->chunks);
    }
    public function loadMore()
    {
        if (!$this->hasMorePages()) {
            return;
        }

        $this->page = $this->page + 1;
    }

}
