<?php

namespace App\Http\Livewire\Sales\Pricing;

use App\Models\Price;
use App\Models\Sizes;
use App\Models\Styles;
use Livewire\Component;
use App\Models\Suppliers;
use Livewire\Attributes\On;
use App\Models\StyleVersions;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\Gate;

class PriceRow extends Component
{
    /**
     * The Style model with nested relationships loaded.
     *
     * Expected relationships:
     * - styleVersions (hasMany StyleVersion)
     *      - prices (hasMany Price)
     *      - factories (or a singular relation, e.g. factory)
     *      - colourways (hasMany Colourway)
     * - seasons (belongsTo Season) on the Style model (or via styleVersions->style->seasons)
     * - designs, customers, etc.
     */
    public $styles;
    public $chunk;

    public $cancelled, $not_quoted;
    public $printList = [];

    #[Computed(cache: true, key: 'factories-with-countries')]
    public function factories(){
        return Suppliers::with('countries')->where('type', 'factory')->get();
    }
    #[Computed]
    public function sizes(){
        return Sizes::get();
    }

    public function canEdit(array $style){
        if(Gate::allows('price:update') && !$style['seasons']['locked']){
            return TRUE;
        }
    }

    /**
     * Mount the component.
     *
     * @param  \App\Models\Style  $style       The Style model with relationships loaded.
     * @param  mixed              $not_quoted  Filter flag (if any)
     */
    public function mount(array $chunk, bool $cancelled = FALSE, bool $not_quoted = FALSE)
    {
        $this->chunk = $chunk;

        $this->cancelled = $cancelled;
        $this->not_quoted = $not_quoted;

        $this->refreshStyles();
    }

    #[On('update-filters')]
    public function updateFilters(bool $cancelled = FALSE, bool $not_quoted = FALSE){
        $this->cancelled = $cancelled;
        $this->not_quoted = $not_quoted;
        $this->refreshStyles();
    }

    #[On('close-modal')]
    public function closeModal(){
        $this->refreshStyles();
    }

    /**
     * Re-query all the Styles so that $this->styles is up-to-date.
     */
    private function refreshStyles()
    {
        $query = Styles::with([
            'seasons:id,usd_rate,gbp_rate,euro_rate,locked',
            'designs:id,description',
            'customers:id,name,currency',
            'style_versions' => function ($versionQuery) {
                // 1) Also retrieve a total_prices_count ignoring filters:
                $versionQuery->withCount('prices as total_prices_count');

                // 2) Then eagerly load the "filtered" prices
                $versionQuery->with([
                    'colourways:id,name,image,style_versions_id' => [
                        'colourway_yarns' => [
                            'yarn_colours.yarn'
                        ]
                    ],
                    'factories:id,currency',
                    'prices' => function ($priceQuery) {
                        // If $this->not_quoted is set, exclude 'confirmed' prices:
                        if ($this->not_quoted) {
                            $priceQuery->where('quote_status', '!=', 'confirmed');
                        }

                        // Count linked orders (priority pricing)
                        $priceQuery->withCount([
                            'customerOrderLineQuantities as links_count' => function($q) {
                                $q->whereNotNull('prices_id');
                            }
                        ]);
                        
                        // Eager-load nested relationships
                        $priceQuery->with([
                            'sizes:id,name',
                            'style_versions:id,name,factory_id,styles_id' => [
                                'colourways:id,name,image,style_versions_id',
                                'factories:id,currency',
                                'styles:id,seasons_id,customers_id,designs_id' => [
                                    'seasons:id,usd_rate,gbp_rate,euro_rate,locked',
                                    'designs:id,description',
                                    'customers:id,name,currency' => [
                                        'customer_payment_terms',
                                    ],
                                ],
                            ],
                        ]);
                    },
                ]);
            },
        ])
        ->whereIn('id', $this->chunk)
        ->orderByRaw("FIELD(id, " . implode(',', $this->chunk) . ")");

        $results = $query->get();

        $results->each(function ($style) {
            $style->style_versions->each->append('image');
            $style->style_versions->each(function ($sv) {
                $sv->prices->each->append('historic');
            });
        });

        // 3) For each Style, for each StyleVersion, check if the filtered prices are empty
        //    AND if total_prices_count == 0 => means truly no DB record
        //    => create a blank Price placeholder.
        foreach ($results as $style) {
            foreach ($style->style_versions as $sv) {
                if ($sv->prices->isEmpty() && $sv->total_prices_count == 0) {
                    // There are zero actual Price records in the DB for this StyleVersion
                    // so let's add a blank "price" (only in-memory for your array).
                    $blankPrice = new Price([
                        // populate any default fields you want
                        'style_versions_id' => $sv->id,
                    ]);
                    $blankPrice->save();
                    $blankPrice->load([
                        'sizes:id,name',
                        'style_versions:id,name,factory_id,styles_id' => [
                            'colourways:id,name,image,style_versions_id',
                            'factories:id,currency',
                            'styles:id,seasons_id,customers_id,designs_id' => [
                                'seasons:id,usd_rate,gbp_rate,euro_rate,locked',
                                'designs:id,description',
                                'customers:id,name,currency' => [
                                    'customer_payment_terms',
                                ],
                            ],
                        ],
                    ]);
                    // Push it into the relationship
                    $sv->prices->push($blankPrice);
                }
            }

            // If you want to mark the entire Style as "empty" if it has no displayed prices:
            $style->empty = $style
                ->style_versions
                ->pluck('prices')
                ->flatten()
                ->isEmpty();
        }
        // $p = new Price([
        //     // populate any default fields you want
        //     'style_versions_id' => $sv->id,
        //     'id' => null,
        // ]);
        // dd($p->toArray());
        // 4) Convert to array for your Livewire property
        $this->styles = $results->toArray();
        // dd($this->styles);
    }



    /**
     * Update a Price field.
     *
     * If $priceId is null, create a new Price record for the given $styleVersionId.
     *
     * @param  int|null  $priceId         The ID of the Price to update, or null to create one.
     * @param  int       $styleVersionId  The ID of the StyleVersion (used if $priceId is null).
     * @param  string    $field           The field name being updated.
     * @param  mixed     $newValue        The new value.
     */
    public function updatePriceField($priceId, $styleVersionId, $field, $newValue)
    {
        // 1) If $priceId is null, we're creating a new Price.
        if (is_null($priceId)) {
            // Load the StyleVersion, with enough relationship data to check canEdit()
            $sv = StyleVersions::with('styles.seasons')->find($styleVersionId);
            if (! $sv) {
                return; // Invalid style version, bail out
            }

            // Convert the Style to array so we can pass it to canEdit()
            $styleArray = $sv->styles->toArray();
            if (! $this->canEdit($styleArray)) {
                return; // Not allowed to edit
            }

            // Create a new Price model in memory
            $price = new Price();
            $price->style_versions_id = $styleVersionId;
        }
        else {
            // 2) Otherwise, we're updating an existing Price
            $price = Price::with('style_versions.styles.seasons')->find($priceId);
            if (! $price) {
                return; // Price not found
            }

            $styleArray = $price->style_versions->styles->toArray();
            if (! $this->canEdit($styleArray)) {
                return; // Not allowed to edit
            }
        }

        // 3) Set the attribute on the model
        if (in_array($field, ['colour_type', 'colourways_id'])) {
            $price->$field = empty($newValue) ? null : $newValue;
        } else {
            $price->$field = empty($newValue) ? 0 : $newValue;
        }

        // 4) Save to database (new or existing record)
        $price->save();

        // 5) Re-query local data so the UI displays the latest
        $this->refreshStyles();
    }


    /**
     * Update a StyleVersion field.
     *
     * @param  int    $versionId  The ID of the StyleVersion.
     * @param  string $field      The field name being updated.
     * @param  mixed  $newValue   The new value.
     */
    public function updateStyleVersionField($versionId, $field, $newValue)
    {
        // 1) Fetch the StyleVersion with needed relationship(s).
        $sv = StyleVersions::with('styles.seasons')->find($versionId);
        if (!$sv) {
            return; // Not found, do nothing
        }

        // 2) Convert the Style to array for canEdit() checks
        $styleArray = $sv->styles->toArray();

        if ($this->canEdit($styleArray)) {
            // 3) Update and save
            $sv->$field = $newValue;
            $sv->save();

            // 4) Refresh
            $this->refreshStyles();
        }
    }

    /**
     * Delete a Price record.
     *
     * @param  int  $priceId The Price record ID.
     */
    public function deletePrice($style, $priceId)
    {
        if($this->canEdit($this->styles[$style])){
            $price = Price::find($priceId);
            if ($price) {
                $price->delete();
                $this->refreshStyles();
            }
        }
    }

    public function pullOldPrice($newPriceId, $oldPriceId){
        $source = Price::find($oldPriceId);
        $destination = Price::find($newPriceId);

        $fieldsToKeep = ['id', 'created_at', 'style_versions_id', 'colourways_id', 'sizes_id', 'colour_type', 'name'];

        $data = collect($source->getAttributes())
        ->except($fieldsToKeep)
        ->toArray();

        $destination->fill($data);
        $destination->save();
    }

    public function updated($propertyName, $propertyValue)
    {
        // 1) If the updated property is in 'styles.X.style_versions.Y.prices.Z.FIELD'
        if (preg_match('/^styles\.(\d+)\.style_versions\.(\d+)\.prices\.(\d+)\.(\w+)$/', $propertyName, $matches)) {
            // $matches[1] = s index, $matches[2] = sv index, $matches[3] = p index, $matches[4] = field
            [$whole, $s, $sv, $p, $field] = $matches;

            // 2) Get the Price record's ID from our $this->styles array:
            $priceId = $this->styles[$s]['style_versions'][$sv]['prices'][$p]['id'];

            // 3) Get the new value (could also directly use $propertyValue)
            $newValue = $this->styles[$s]['style_versions'][$sv]['prices'][$p][$field]
                        ?? $propertyValue;
            $svId = $this->styles[$s]['style_versions'][$sv]['id'];
            // 4) Call the existing method that updates the Price
            $this->updatePriceField($priceId, $svId, $field, $newValue);
        }

        // If the updated property is in 'styles.X.style_versions.Y.FIELD' (i.e., style version fields):
        elseif (preg_match('/^styles\.(\d+)\.style_versions\.(\d+)\.(\w+)$/', $propertyName, $matches)) {
            [$whole, $s, $sv, $field] = $matches;

            // The StyleVersion ID:
            $versionId = $this->styles[$s]['style_versions'][$sv]['id'];

            // New value:
            $newValue = $this->styles[$s]['style_versions'][$sv][$field] ?? $propertyValue;

            // Call your existing update method
            $this->updateStyleVersionField($versionId, $field, $newValue);
        }

        // You can continue similarly for other nested property patterns...
    }


    /**
     * Count the total number of prices in a given grouped style.
     *
     * @param mixed $style A grouped collection (grouped by style version)
     * @return int
     */
    public function countPricesInStyle($style)
    {
        $count = 0;
        foreach($style['style_versions'] as $sv){
                $count = $count + count($sv['prices']);
        }
        return $count;
    }

    /**
     * Count the number of prices in a specific style version.
     *
     * @param mixed $version A collection of prices for a style version.
     * @return int
     */
    public function countPricesInVersion($version)
    {
        if (is_array($version)) {
            return count($version);
        } elseif ($version instanceof \Illuminate\Support\Collection) {
            return $version->count();
        }
        return 0;
    }

    public function clonePrice($price){
        // dd($price);
        Gate::authorize('price:update');
        $price = Price::find($price)->duplicate();
        $price->cmt_status = FALSE;
        $price->quote_status = FALSE;
        $price->cmt_status = 'not_confirmed';
        $price->quote_status = 'not_confirmed';
        $price->save();
        $this->refreshStyles();
    }

    public function togglePriceFlag($priceId)
    {
        if (!$priceId) {
            return;
        }

        $price = Price::with('style_versions.styles.seasons')->find($priceId);
        if (! $price) {
            return;
        }

        $styleArray = $price->style_versions->styles->toArray();
        if (! $this->canEdit($styleArray)) {
            return;
        }

        $price->flag_issue = ! (bool) $price->flag_issue;
        $price->save();

        $this->refreshStyles();
    }

    public function getYarns($styleVersion){
        return collect($styleVersion['colourways'])
            ->flatMap(fn ($cw) => $cw['colourway_yarns'])
            ->pluck('yarn_colours.yarn.description')
            ->unique();
    }

    /**
     * Get total ordered quantity for a style version
     *
     * @param array $styleVersion
     * @return int
     */
    public function getTotalOrderedQty($styleVersion)
    {
        $styleVersionId = $styleVersion['id'] ?? null;
        
        if (!$styleVersionId) {
            return 0;
        }

        // Query to sum all quantities ordered for this style version
        return \DB::table('customer_order_line_quantities')
            ->join('customer_order_lines', 'customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
            ->join('colourways', 'customer_order_lines.colourways_id', '=', 'colourways.id')
            ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
            ->where('colourways.style_versions_id', $styleVersionId)
            ->whereNull('customer_order_line_quantities.deleted_at')
            ->whereNull('customer_order_lines.deleted_at')
            ->whereNull('customer_orders.deleted_at')
            ->where('customer_order_lines.cancelled', false)
            ->where('customer_orders.cancelled', false)
            ->sum('customer_order_line_quantities.qty');
    }

    /**
     * Navigate to orders page filtered by style version
     *
     * @param int $styleVersionId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function viewOrders($styleVersionId)
    {
        // Update the user's order list filter settings
        auth()->user()->settings([
            'orderslistfilters' => [
                'type' => 'wholesale',
                'style_version' => $styleVersionId,
                'season' => null,
                'department' => null,
                'category' => null,
                'customer' => null,
                'factory' => null,
                'search' => null,
            ]
        ]);
        
        // Redirect to orders page
        return redirect()->route('orders');
    }

    #[On('select-all')]
    public function selectAll(){
        foreach($this->styles as $style){
            foreach($style['style_versions'] as $sv){
                foreach($sv['prices'] as $price){
                    $this->printList[$price['id']] = TRUE;
                    $this->dispatch('toggle-selected-price', id: $price['id']);
                }
            }
        }
    }

    /**
     * Render the component.
     *
     * @return \Illuminate\View\View
     */
    public function render()
    {
        return view('livewire.sales.pricing.price-row');
    }
}
