<?php

namespace App\Http\Livewire\Sales\Pricing;

use App\Models\Price;
use App\Models\CustomerOrders;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrderLineQuantities;
use App\Services\PriceResolutionService;
use Livewire\Component;
use Livewire\Attributes\On;
use Illuminate\Support\Facades\DB;

class ApplyPriceToOrders extends Component
{
    public $showModal = false;
    public $priceId = null;
    public $price = null;
    public $compatibleOrders = [];
    public $selectedOrders = [];
    public $applyToAllSizes = [];
    public $selectedSizes = [];
    public $processing = false;
    public $successMessage = '';
    public $errorMessage = '';

    public function render()
    {
        return view('livewire.sales.pricing.apply-price-to-orders');
    }

    #[On('open-apply-price-modal')]
    public function openModal($priceId)
    {
        $this->reset(['selectedOrders', 'applyToAllSizes', 'selectedSizes', 'successMessage', 'errorMessage']);
        
        $this->priceId = $priceId;
        $this->price = Price::with(['style_versions.styles', 'colourways', 'sizes'])->find($priceId);
        
        if (!$this->price) {
            $this->errorMessage = 'Price not found.';
            return;
        }

        $this->loadCompatibleOrders();
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->reset();
    }

    /**
     * Load orders that are compatible with this price
     */
    protected function loadCompatibleOrders()
    {
        $styleVersionId = $this->price->style_versions_id;
        $colourwayId = $this->price->colourways_id;
        $sizeId = $this->price->sizes_id;
        $phaseId = $this->price->phase_id;

        // Build query for compatible orders
        $query = CustomerOrderLines::with([
            'customer_orders' => function($q) {
                $q->select('id', 'customers_id', 'customer_po', 'order_date', 'seasons_id', 'phase_id', 'order_type');
            },
            'customer_orders.customers:id,name',
            'customer_order_line_quantities' => function($q) {
                $q->select('id', 'customer_order_lines_id', 'sizes_id', 'qty', 'prices_id');
            },
            'customer_order_line_quantities.sizes:id,name',
            'colourways:id,name,style_versions_id'
        ])
        ->whereHas('customer_orders', function($q) {
            $q->where('order_type', 'wholesale')
              ->where('cancelled', 0);
        })
        ->where('cancelled', 0);

        // Match style version
        $query->whereHas('colourways', function($q) use ($styleVersionId) {
            $q->where('style_versions_id', $styleVersionId);
        });

        // If price is colourway-specific, match that colourway
        if ($colourwayId) {
            $query->where('colourways_id', $colourwayId);
        }

        // If price has a phase, match orders with that phase (or NULL phase on order)
        if ($phaseId) {
            $query->whereHas('customer_orders', function($q) use ($phaseId) {
                $q->where(function($subQ) use ($phaseId) {
                    $subQ->where('phase_id', $phaseId)
                         ->orWhereNull('phase_id');
                });
            });
        }

        $orderLines = $query->get();

        // Transform into a structure for the UI
        $this->compatibleOrders = $orderLines->map(function($orderLine) use ($sizeId) {
            $order = $orderLine->customer_orders;
            $quantities = $orderLine->customer_order_line_quantities;

            // Filter quantities by size if price is size-specific
            if ($sizeId && $sizeId != 1) { // 1 is typically the "general" size
                $quantities = $quantities->where('sizes_id', $sizeId);
            }

            return [
                'order_line_id' => $orderLine->id,
                'order_id' => $order->id,
                'customer_name' => $order->customers->name ?? 'Unknown',
                'customer_po' => $order->customer_po,
                'order_date' => $order->order_date?->format('Y-m-d'),
                'colourway_name' => $orderLine->colourways->name ?? 'Unknown',
                'quantities' => $quantities->map(function($qty) {
                    return [
                        'id' => $qty->id,
                        'size_name' => $qty->sizes->name ?? 'Unknown',
                        'size_id' => $qty->sizes_id,
                        'qty' => $qty->qty,
                        'has_priority_price' => !empty($qty->prices_id),
                        'current_price_id' => $qty->prices_id,
                    ];
                })->toArray(),
            ];
        })->toArray();
    }

    /**
     * Toggle selection of an entire order (all sizes)
     */
    public function toggleOrder($orderLineId)
    {
        if (isset($this->selectedOrders[$orderLineId])) {
            unset($this->selectedOrders[$orderLineId]);
            unset($this->applyToAllSizes[$orderLineId]);
            if (isset($this->selectedSizes[$orderLineId])) {
                unset($this->selectedSizes[$orderLineId]);
            }
        } else {
            $this->selectedOrders[$orderLineId] = true;
            $this->applyToAllSizes[$orderLineId] = true; // Default to all sizes
        }
    }

    /**
     * Toggle selection of a specific size within an order
     */
    public function toggleSize($orderLineId, $quantityId)
    {
        if (!isset($this->selectedOrders[$orderLineId])) {
            $this->selectedOrders[$orderLineId] = true;
        }

        // Disable "all sizes" if manually selecting individual sizes
        $this->applyToAllSizes[$orderLineId] = false;

        if (!isset($this->selectedSizes[$orderLineId])) {
            $this->selectedSizes[$orderLineId] = [];
        }

        $key = array_search($quantityId, $this->selectedSizes[$orderLineId]);
        if ($key !== false) {
            unset($this->selectedSizes[$orderLineId][$key]);
            $this->selectedSizes[$orderLineId] = array_values($this->selectedSizes[$orderLineId]);
            
            // If no sizes left, unselect the order
            if (empty($this->selectedSizes[$orderLineId])) {
                unset($this->selectedOrders[$orderLineId]);
                unset($this->selectedSizes[$orderLineId]);
            }
        } else {
            $this->selectedSizes[$orderLineId][] = $quantityId;
        }
    }

    /**
     * Apply the price to selected orders
     */
    public function applyPrice()
    {
        if (empty($this->selectedOrders)) {
            $this->errorMessage = 'Please select at least one order.';
            return;
        }

        $this->processing = true;
        $this->errorMessage = '';
        $this->successMessage = '';

        try {
            DB::beginTransaction();

            $appliedCount = 0;

            foreach ($this->selectedOrders as $orderLineId => $selected) {
                if (!$selected) continue;

                $order = collect($this->compatibleOrders)->firstWhere('order_line_id', $orderLineId);
                if (!$order) continue;

                // Determine which quantities to update
                $quantityIds = [];
                
                if (!empty($this->applyToAllSizes[$orderLineId])) {
                    // Apply to all sizes
                    $quantityIds = collect($order['quantities'])->pluck('id')->toArray();
                } else {
                    // Apply to selected sizes only
                    $quantityIds = $this->selectedSizes[$orderLineId] ?? [];
                }

                if (empty($quantityIds)) continue;

                // Update the quantities
                $updated = CustomerOrderLineQuantities::whereIn('id', $quantityIds)
                    ->update(['prices_id' => $this->priceId]);

                $appliedCount += $updated;

                // Recompute and cache price resolutions for these quantities
                $service = app(PriceResolutionService::class);
                $totalCacheService = app(\App\Services\TotalCacheService::class);
                
                $quantities = CustomerOrderLineQuantities::with([
                    'customer_order_lines.colourways',
                    'customer_order_lines.customer_orders',
                    'customer_order_lines.shipment_lines'
                ])->whereIn('id', $quantityIds)->get();
                
                foreach ($quantities as $qty) {
                    // Clear old cached resolution
                    $qty->setCachedPriceResolution(null);
                    
                    // Trigger recomputation which will save to price_resolutions table
                    $colourway = $qty->customer_order_lines->colourways;
                    $seasonId = $qty->customer_order_lines->customer_orders->seasons_id;
                    
                    // This will use the new prices_id and compute/cache the resolution
                    $service->resolveWithSpecificPrice(
                        $this->priceId,
                        $colourway->style_versions_id,
                        $colourway->id,
                        $qty->sizes_id,
                        $qty->customer_order_lines->phase_id,
                        $seasonId
                    );
                    
                    // Invalidate total cache for affected entities
                    $totalCacheService->invalidateEntity('customer_order_line', $qty->customer_order_lines_id);
                    $totalCacheService->invalidateEntity('customer_order', $qty->customer_order_lines->customer_orders_id);
                    
                    // Invalidate shipment line and shipment caches
                    foreach ($qty->customer_order_lines->shipment_lines as $shipmentLine) {
                        $totalCacheService->invalidateEntity('shipment_line', $shipmentLine->id);
                        if ($shipmentLine->shipment_id) {
                            $totalCacheService->invalidateEntity('shipment', $shipmentLine->shipment_id);
                        }
                    }
                }
            }

            DB::commit();

            $this->successMessage = "Successfully applied priority price to {$appliedCount} order line quantities.";
            $this->selectedOrders = [];
            $this->selectedSizes = [];
            $this->applyToAllSizes = [];
            
            // Reload compatible orders to show updated state
            $this->loadCompatibleOrders();

        } catch (\Exception $e) {
            DB::rollBack();
            $this->errorMessage = 'Error applying price: ' . $e->getMessage();
            \Log::error('Error applying price to orders', [
                'price_id' => $this->priceId,
                'error' => $e->getMessage()
            ]);
        } finally {
            $this->processing = false;
        }
    }

    /**
     * Remove priority price from selected orders
     */
    public function removePriorityPrice()
    {
        if (empty($this->selectedOrders)) {
            $this->errorMessage = 'Please select at least one order.';
            return;
        }

        $this->processing = true;
        $this->errorMessage = '';
        $this->successMessage = '';

        try {
            DB::beginTransaction();

            $removedCount = 0;

            foreach ($this->selectedOrders as $orderLineId => $selected) {
                if (!$selected) continue;

                $order = collect($this->compatibleOrders)->firstWhere('order_line_id', $orderLineId);
                if (!$order) continue;

                $quantityIds = [];
                
                if (!empty($this->applyToAllSizes[$orderLineId])) {
                    $quantityIds = collect($order['quantities'])->pluck('id')->toArray();
                } else {
                    $quantityIds = $this->selectedSizes[$orderLineId] ?? [];
                }

                if (empty($quantityIds)) continue;

                $updated = CustomerOrderLineQuantities::whereIn('id', $quantityIds)
                    ->update(['prices_id' => null]);

                $removedCount += $updated;
                
                // Recompute price resolutions (will use dynamic pricing now)
                $service = app(PriceResolutionService::class);
                $totalCacheService = app(\App\Services\TotalCacheService::class);
                
                $quantities = CustomerOrderLineQuantities::with([
                    'customer_order_lines.colourways',
                    'customer_order_lines.customer_orders',
                    'customer_order_lines.shipment_lines'
                ])->whereIn('id', $quantityIds)->get();
                
                foreach ($quantities as $qty) {
                    $qty->setCachedPriceResolution(null);
                    
                    // Trigger recomputation with dynamic pricing
                    $colourway = $qty->customer_order_lines->colourways;
                    $seasonId = $qty->customer_order_lines->customer_orders->seasons_id;
                    
                    $service->resolve(
                        $colourway->style_versions_id,
                        $colourway->id,
                        $qty->sizes_id,
                        $qty->customer_order_lines->phase_id,
                        $seasonId
                    );
                    
                    // Invalidate total cache for affected entities
                    $totalCacheService->invalidateEntity('customer_order_line', $qty->customer_order_lines_id);
                    $totalCacheService->invalidateEntity('customer_order', $qty->customer_order_lines->customer_orders_id);
                    
                    // Invalidate shipment line and shipment caches
                    foreach ($qty->customer_order_lines->shipment_lines as $shipmentLine) {
                        $totalCacheService->invalidateEntity('shipment_line', $shipmentLine->id);
                        if ($shipmentLine->shipment_id) {
                            $totalCacheService->invalidateEntity('shipment', $shipmentLine->shipment_id);
                        }
                    }
                }
            }

            DB::commit();

            $this->successMessage = "Successfully removed priority price from {$removedCount} order line quantities.";
            $this->selectedOrders = [];
            $this->selectedSizes = [];
            $this->applyToAllSizes = [];
            
            $this->loadCompatibleOrders();

        } catch (\Exception $e) {
            DB::rollBack();
            $this->errorMessage = 'Error removing price: ' . $e->getMessage();
        } finally {
            $this->processing = false;
        }
    }
}

