<?php

namespace App\Http\Livewire\Sales\Orders;

use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use App\Models\CustomerOrders;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

class OrdersUpdateChangeSeasonCust extends Component
{
    public  $order;

    #[Rule('required|exists:seasons,id')]
    public $season;

    #[Rule('required|exists:customers,id')]
    public$customer;

    #[Rule('required|exists:departments,id')]
    public $department;

    #[Computed]
    public function customers(){
        return Customer::allCached();
    }
    #[Computed]
    public function seasons(){
        return Seasons::allCached();
    }

    #[Computed]
    public function departments(){
        return Departments::allCached();
    }

    public function render()
    {
        return view('livewire.sales.orders.orders-update-change-season-cust');
    }

    #[On('edit-season-cust')]
    public function loadOrder($order){
        $this->order = CustomerOrders::with([
            'customer_order_lines.colourways.style_versions.styles',
            'customer_order_lines.colourways.customer_order_lines' // Needed for canChangeSeasonCustomer check
        ])->find($order);
        if ($this->order) {
            $this->season = $this->order->seasons_id;
            $this->customer = $this->order->customers_id;
            $this->department = $this->order->departments_id;
        }
    }

    public function save()
    {
        if (!$this->order) {
            session()->flash('error', 'No order found.');
            return;
        }

        $this->validate(); // Ensure season and customer values are valid

        if (!$this->canChangeSeasonCustomer()) {
            session()->flash('error', 'Order styles belong to multiple orders.');
            return;
        }

        DB::beginTransaction(); // Start a transaction

        try {
            // Update the order
            $this->order->update([
                'customers_id' => $this->customer,
                'seasons_id' => $this->season,
                'departments_id' => $this->department,
            ]);
            // Collect unique styles to prevent duplicate updates
            $stylesToUpdate = collect();

            foreach ($this->order->customer_order_lines as $line) {
                // Skip lines with missing relationships
                $style = $line->colourways?->style_versions?->styles;
                if ($style) {
                    $stylesToUpdate->push($style);
                }
            }

            // Get unique styles by ID
            $stylesToUpdate = $stylesToUpdate->unique('id');

            // Update only the unique styles
            foreach ($stylesToUpdate as $style) {
                $style->update([
                    'customers_id' => $this->customer,
                    'seasons_id' => $this->season,
                    'departments_id' => $this->department,
                ]);
            }

            DB::commit(); // Commit the transaction if everything is successful

            session()->flash('success', 'Order and styles updated successfully.');
            $this->dispatch('order-updated'); // Optional event dispatch
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback changes if an error occurs
            session()->flash('error', 'Failed to update order: ' . $e->getMessage());
        }
        $this->reset();
        $this->dispatch('close-modal');
    }

    public function canChangeSeasonCustomer(){
        if (!isset($this->order)) {
            return false;
        }

        if (!Gate::check('style:update')) {
            return false;
        }

        // Check if any colourway is used by multiple order lines (across different orders)
        foreach ($this->order->customer_order_lines as $line) {
            // Skip if colourway relationship is null
            if (!$line->colourways) {
                continue;
            }
            
            // Check if this colourway has customer order lines from OTHER orders
            $otherOrderLinesCount = $line->colourways->customer_order_lines
                ->where('customer_orders_id', '!=', $this->order->id)
                ->count();
            
            if ($otherOrderLinesCount > 0) {
                return false;
            }
        }

        return true;
    }
}
