<?php

namespace App\Http\Livewire\Sales\Orders;

use App\Models\Gauge;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Departments;
use App\Models\StyleVersions;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use App\Models\CustomerOrders;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use Illuminate\Database\Query\JoinClause;
use App\Http\Livewire\FilterableComponent;

class OrdersList extends FilterableComponent
{
	use WithPagination;
	use Filterable;
    public $hideFilters = false;

	#[Validate([
		'season' => ['nullable', 'exists:seasons,id'],
		'department' => ['nullable', 'exists:departments,id'],
		'category' => ['nullable', 'exists:categories,id'],
		'customer' => ['nullable', 'exists:customers,id'],
		'factory' => ['nullable', 'exists:suppliers,id'],
		'style_version' => ['nullable', 'exists:style_versions,id'],
		'search' => ['nullable', 'string', 'max:30'],
		'paginate' => ['required', 'numeric', 'min:10', 'max:1000'],
		'editID' => ['nullable', 'numeric', 'min:0']
	])]
	public $season, $department, $category, $customer, $factory, $style_version, $search, $paginate = 50, $editID;

    protected function filterKey(): string
    {
        return 'orderslistfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.2;
    }

    protected function filters(): array
    {
        return ['season', 'department', 'category', 'customer', 'factory', 'style_version', 'search', 'columns', 'paginate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
    }

    #[On('refreshData')]
    public function refreshData()
    {
        // Refresh the page to show updated order data
        $this->resetPage();
    }

	public $columns = [
		'Order Date' => TRUE,
		'Season' => TRUE,
		'PO' => TRUE,
		'Customer' => TRUE,
		'Style' => TRUE,
		'Colourway' => TRUE,
		'Qty' => TRUE,
		'Total Qty' => TRUE,
	];

	public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
                if (in_array($propertyName, ['season', 'department', 'category', 'customer', 'factory', 'style_version', 'search', 'paginate'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }


	#[Computed()]
	public function customers(){
		return Customer::allCached();
	}
	#[Computed()]
	public function seasons(){
		return Seasons::allCached();
	}
	#[Computed()]
	public function departments(){
		return Departments::allCached();
	}
    #[Computed(cache: true, key: 'factories-wherehas-style_versions')]
	public function factories(){
		return Suppliers::has('style_versions')->get();
	}

    #[Computed()]
    public function styleVersions(){
        // Get style versions that have orders, with style and design info for display
        return StyleVersions::select('style_versions.id', 'style_versions.name', 'styles.customer_ref', 'designs.id as design_id')
            ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->join('designs', 'designs.id', '=', 'styles.designs_id')
            ->whereHas('colourways.customer_order_lines')
            ->orderBy('designs.id')
            ->orderBy('styles.customer_ref')
            ->orderBy('style_versions.name')
            ->get();
    }

    /**
     * Clear all filters including hidden ones like style_version
     */
    public function clearAllFilters()
    {
        $this->reset(['season', 'department', 'category', 'customer', 'factory', 'style_version', 'search']);
        $this->resetPage();
        $this->saveFilterSettings();
    }


	#[On('close-modal')]
    public function render()
    {
	Gate::authorize('order:read');

		$orders = CustomerOrders::select(
			'customer_orders.id',
			'order_date',
			'seasons.description as season',
			'customer_po as po',
			'customers.name as customer',
			'customer_orders.order_type',
			)
		->selectRaw("(SELECT CONCAT('[',GROUP_CONCAT(JSON_OBJECT('id', col.id, 'deleted', col.deleted_at, 'design', d.id, 'description', d.description, 'style', s.customer_ref, 'version', sv.id, 'version_name', sv.name, 'colourway', c.name, 'qty', (SELECT SUM(qty) FROM customer_order_line_quantities colq WHERE colq.customer_order_lines_id=col.id and colq.deleted_at is null GROUP BY colq.customer_order_lines_id))),']')
			FROM customer_order_lines col
			JOIN colourways c on c.id=col.colourways_id
			JOIN style_versions sv on sv.id=c.style_versions_id
			JOIN styles s on s.id=sv.styles_id
			JOIN designs d on d.id=s.designs_id
			WHERE col.customer_orders_id=customer_orders.id
			) as styles")
		->search($this->search)
        // ->whereIn('customer_orders.departments_id', Auth::user()->user_departments->pluck('departments_id')->toArray())
		->join('customers', function (JoinClause $join) {
			$join->on('customers.id', '=', 'customer_orders.customers_id');
			if(!empty($this->customer))
				$join->where('customers.id', $this->customer);
		})
		->join('seasons', function (JoinClause $join) {
			$join->on('seasons.id', '=', 'customer_orders.seasons_id');
			if(!empty($this->season))
				$join->where('seasons.id', $this->season);
		})
		->where('order_type', 'wholesale')
		->when(!empty($this->department),function($query){
			$query->where('departments_id', $this->department);
		})
        ->when(!empty($this->factory),function($query){
			$query->whereRelation('customer_order_lines.colourways.style_versions', 'factory_id', $this->factory);
		})
		->when(!empty($this->style_version),function($query){
			$query->whereRelation('customer_order_lines.colourways', 'style_versions_id', $this->style_version);
		})
		->paginate($this->paginate);
        // dd($orders);

		foreach($orders as $order){
			$order->styles = collect(json_decode($order->styles));
			$order->total_qty = $order->styles->where('deleted', NULL)->sum('qty');
			$order->styles_qty = $order->styles->count();
		}
        return view('livewire.sales.orders.orders-list', ['orders' => $orders]);
    }
    // #[On('edit-order')]
    // public function view($id){
    //     $this->dispatch('edit-order', order: $id);
    //     $this->editID = $id;
    // }

    public function edit($order){
        // wire:click="$dispatch('edit-order', { order: {{$order->id}} })"
    }

	public function go(){}

    public function restore(int $id){
        $line = CustomerOrderLines::withTrashed()->find($id);
        $line->restore();
    }

	public function allCols(){
		$this->reset('columns');
		$this->updatedColumns();
	}

	public function updatedColumns(){
			auth()->user()->settings(['orders_list_columns' => $this->columns]);
	}

    public function delete($order){
        $order=CustomerOrders::find($order);
        if(!empty($order) && $order->delete()){
            session()->flash('message', 'Order Deleted');
		    session()->flash('alert-class', 'alert-success');
        }
        else{
            session()->flash('message', 'Error Deleting');
            session()->flash('alert-class', 'alert-danger');
        }
    }

    public function mount()
    {
        $this->loadFilterSettings();
    }

}
