<?php

namespace App\Http\Livewire\Sales\Orders;

use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\On;
use App\Models\CustomerOrders;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrderLineQuantities;
use App\Models\Customer;
use App\Models\Seasons;
use App\Models\Departments;
use App\Models\Phase;
use App\Models\Colourways;
use App\Models\Sizes;
use App\Models\CustomerOrderFiles;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Livewire\Forms\OrdersEditV2Form;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Gate;

class OrdersEditV2 extends Component
{
    use WithFileUploads;

    public OrdersEditV2Form $form;
    
    public ?CustomerOrders $order = null;
    public $activeLineId = null;
    public $mode = 'single'; // 'single' or 'quick'
    public $viewMode = 'edit'; // 'edit' or 'view'
    public $itemPicker = false;
    public $itemPickerMode = 'add'; // 'add' or 'change'
    public $itemPickerLine = null;
    public $searchTerm = '';
    public $isNew = false;
    
    // Auto-save state
    public $hasUnsavedChanges = false;
    public $isSaving = false;
    public $lastSaved = null;
    
    // Track pending drops (to be created on save)
    public $pendingDrops = [];
    public $pendingDropData = []; // Store data for pending drops
    
    // Track pending sizes (to be created on save)
    public $pendingSizes = []; // Store pending sizes per line
    public $pendingSizeRemovals = []; // Track sizes marked for deletion
    
    // Computed properties
    public function getReadOnlyProperty()
    {
        if (!$this->order) return false;
        return $this->order->seasons->locked ?? false;
    }
    
    public function getDepartmentsProperty()
    {
        return Departments::where('type', 'wholesale')
            ->where('hidden', false)
            ->orderBy('name')
            ->get();
    }
    
    public function getSeasonsProperty()
    {
        return Seasons::where('locked', false)
            ->orderBy('start_date', 'desc')
            ->get();
    }
    
    public function getCustomersProperty()
    {
        return Customer::with('customer_addresses')
            ->orderBy('name')
            ->get();
    }
    
    public function getPhasesProperty()
    {
        return Phase::orderBy('name')->get();
    }
    
    public function getCustomerAddressesProperty()
    {
        if (!isset($this->form) || !$this->form->customers_id) return collect();
        
        $customer = Customer::with('customer_addresses')->find($this->form->customers_id);
        return $customer ? $customer->customer_addresses : collect();
    }
    
    public function getOrderLinesProperty()
    {
        if (!$this->order) return collect();
        
        return CustomerOrderLines::with([
            'colourways:id,style_versions_id,name' => [
                'style_versions:id,styles_id,factory_id,name' => [
                    'styles:id,designs_id,customer_ref' => [
                        'designs:id,description'
                    ],
                    'factories:id,name'
                ]
            ],
            'customer_order_line_quantities' => [
                'sizes:id,name,order'
            ],
            'shipment_lines' => [
                'shipment_line_sizes' => [
                    'sizes:id,name,order'
                ]
            ]
        ])
        ->where('customer_orders_id', $this->order->id)
        ->orderBy('created_at')
        ->get();
    }
    
    public function getActiveLineProperty()
    {
        if (!$this->activeLineId) return null;
        
        return CustomerOrderLines::with([
            'colourways:id,style_versions_id,name' => [
                'style_versions:id,styles_id,factory_id,name' => [
                    'styles:id,designs_id,customer_ref' => [
                        'designs:id,description'
                    ],
                    'factories:id,name',
                    'prices:id,style_versions_id,quote'
                ]
            ],
            'customer_order_line_quantities' => [
                'sizes:id,name,order'
            ],
            'shipment_lines' => [
                'shipment_line_sizes' => [
                    'sizes:id,name,order'
                ]
            ]
        ])->find($this->activeLineId);
    }
    
    public function getAvailableItemsProperty()
    {
        if (!isset($this->form) || !$this->form->customers_id || !$this->form->seasons_id) {
            return collect();
        }
        
        $query = Colourways::with([
            'style_versions:id,styles_id,factory_id,name' => [
                'styles:id,designs_id,customer_ref,customers_id' => [
                    'designs:id,description'
                ],
                'factories:id,name',
                'prices:id,style_versions_id,quote'
            ]
        ])
        ->whereHas('style_versions.styles', function($q) {
            $q->where('customers_id', $this->form->customers_id)
              ->where('seasons_id', $this->form->seasons_id);
        })
        ->where('cancelled', false);
        
        if ($this->searchTerm) {
            $query->where(function($q) {
                $term = $this->searchTerm;
                $q->where('name', 'like', "%{$term}%")
                  ->orWhereHas('style_versions', function($sq) use ($term) {
                      $sq->where('name', 'like', "%{$term}%")
                         ->orWhereHas('styles', function($st) use ($term) {
                             $st->where('customer_ref', 'like', "%{$term}%")
                                ->orWhere('designs_id', 'like', "%{$term}%")
                                ->orWhereHas('designs', function($d) use ($term) {
                                    $d->where('description', 'like', "%{$term}%");
                                });
                         });
                  });
            });
        }
        
        return $query->limit(50)->get();
    }
    
    public function getCanQuickEditProperty()
    {
        if (!$this->order) return false;
        
        // Disable quick edit if any line has multiple drops
        foreach ($this->orderLines as $line) {
            if ($line->shipment_lines->count() > 1) {
                return false;
            }
        }
        
        return true;
    }
    
    public function getOrderFilesProperty()
    {
        if (!$this->order) return collect();
        
        return CustomerOrderFiles::where('customer_orders_id', $this->order->id)->get();
    }
    
    #[On('edit-order-v2')]
    public function editOrder($order)
    {
        $this->isNew = false;
        $this->order = CustomerOrders::with([
            'departments',
            'seasons',
            'customers.customer_addresses',
            'customer_addresses',
            'phase',
            'customer_order_lines'
        ])->findOrFail($order);
        
        $this->form->open($this->order);
        $this->viewMode = $this->readOnly ? 'view' : 'edit';

        // Reset change tracking
        $this->hasUnsavedChanges = false;
        $this->lastSaved = null;
        $this->pendingDrops = [];
        $this->pendingDropData = [];
        $this->pendingSizes = [];
        $this->pendingSizeRemovals = [];
        
        // Auto-select first line if exists
        $firstLine = $this->orderLines->first();
        if ($firstLine) {
            $this->activeLineId = $firstLine->id;
        }
    }
    
    #[On('new-order-v2')]
    public function newOrder()
    {
        $this->isNew = true;
        $this->reset(['order', 'activeLineId', 'mode', 'hasUnsavedChanges', 'lastSaved', 'pendingDrops', 'pendingDropData', 'pendingSizes', 'pendingSizeRemovals']);
        $this->form->open();
        $this->viewMode = 'edit';
    }
    
    #[On('close-modal')]
    public function closeModal()
    {
        if ($this->hasUnsavedChanges) {
            $this->dispatch('confirm-close');
            return;
        }
        
        $this->form->close();
        $this->reset();
    }
    
    public function saveOrder()
    {
        $this->isSaving = true;
        
        try {
            $result = $this->form->save();
            
            if ($result) {
                // If this was a new order, reload it
                if ($this->isNew && $this->form->order) {
                    $this->order = $this->form->order;
                    $this->isNew = false;
                }
                
                // Delete sizes marked for removal
                foreach ($this->pendingSizeRemovals as $lineId => $quantityIds) {
                    foreach ($quantityIds as $quantityId) {
                        $quantity = CustomerOrderLineQuantities::find($quantityId);
                        if ($quantity) {
                            $quantity->delete();
                        }
                    }
                }
                
                // Clear pending removals after deletion
                $this->pendingSizeRemovals = [];
                
                // Create pending sizes (after removals, before drops)
                foreach ($this->pendingSizes as $lineId => $pendingSizesArray) {
                    foreach ($pendingSizesArray as $tempId => $sizeData) {
                        // If size name was changed, create/find the size with new name
                        $size = Sizes::firstOrCreate(
                            ['name' => $sizeData['size_name']],
                            ['order' => 999]
                        );
                        
                        CustomerOrderLineQuantities::create([
                            'customer_order_lines_id' => $lineId,
                            'sizes_id' => $size->id,
                            'qty' => $sizeData['qty'] ?? 0,
                            'price' => $sizeData['price'] ?? 0,
                            'discount' => $sizeData['discount'] ?? 0,
                            'commission' => $sizeData['commission'] ?? 0,
                        ]);
                    }
                }
                
                // Clear pending sizes after creation
                $this->pendingSizes = [];
                
                // Create pending drops
                foreach ($this->pendingDrops as $lineId => $dropCount) {
                    $line = CustomerOrderLines::find($lineId);
                    if ($line) {
                        for ($i = 0; $i < $dropCount; $i++) {
                            // Get pending drop data
                            $dropData = $this->pendingDropData[$lineId][$i] ?? [];
                            
                            // Create new shipment line
                            $shipmentLine = ShipmentLine::create([
                                'customer_order_lines_id' => $lineId,
                                'exfty' => $dropData['exfty'] ?? null,
                            ]);
                            
                            // Reload line to get newly created sizes
                            $line->refresh();
                            
                            // Create shipment line sizes with stored quantities
                            foreach ($line->customer_order_line_quantities as $qty) {
                                ShipmentLineSizes::create([
                                    'shipment_line_id' => $shipmentLine->id,
                                    'sizes_id' => $qty->sizes_id,
                                    'qty' => $dropData['dropping'][$qty->sizes_id] ?? 0,
                                    'shipped_qty' => $dropData['packed'][$qty->sizes_id] ?? 0,
                                ]);
                            }
                        }
                    }
                }
                
                // Clear pending drops after creation
                $this->pendingDrops = [];
                $this->pendingDropData = [];
                
                $this->hasUnsavedChanges = false;
                $this->lastSaved = now()->format('H:i:s');
                
                session()->flash('message', 'Order saved successfully');
                session()->flash('alert-class', 'alert-success');
            }
        } catch (\Exception $e) {
            session()->flash('message', 'Error saving order: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        } finally {
            $this->isSaving = false;
        }
    }
    
    /**
     * Track changes to any property
     * In Livewire 3, we need to use the updated hook without parameters
     * or with specific property paths to track Form object changes
     */
    public function updated($property)
    {
        // Track changes to form properties
        if (str_starts_with($property, 'form.')) {
            $this->hasUnsavedChanges = true;
        }
    }
    
    public function selectLine($lineId)
    {
        $this->activeLineId = $lineId;
    }
    
    public function openItemPicker($mode = 'add', $lineId = null)
    {
        $this->itemPickerMode = $mode;
        $this->itemPickerLine = $lineId;
        $this->itemPicker = true;
        $this->searchTerm = '';
    }
    
    public function closeItemPicker()
    {
        $this->itemPicker = false;
        $this->searchTerm = '';
    }
    
    public function addItem($colourwayId)
    {
        if (!$this->order) {
            // Need to save order first
            $this->saveOrder();
            if (!$this->order) return;
        }
        
        $line = new CustomerOrderLines();
        $line->customer_orders_id = $this->order->id;
        $line->colourways_id = $colourwayId;
        $line->save();
        
        $this->activeLineId = $line->id;
        $this->closeItemPicker();
        
        session()->flash('message', 'Item added');
        session()->flash('alert-class', 'alert-success');
    }
    
    public function changeItem($colourwayId)
    {
        if (!$this->itemPickerLine) return;
        
        $line = CustomerOrderLines::find($this->itemPickerLine);
        if ($line) {
            $line->colourways_id = $colourwayId;
            $line->update();
            
            $this->activeLineId = $line->id;
            
            session()->flash('message', 'Item changed');
            session()->flash('alert-class', 'alert-success');
        }
        
        $this->closeItemPicker();
    }
    
    public function toggleCancelLine($lineId)
    {
        $line = CustomerOrderLines::find($lineId);
        if ($line) {
            $line->cancelled = !$line->cancelled;
            $line->update();
            
            session()->flash('message', $line->cancelled ? 'Item cancelled' : 'Item un-cancelled');
            session()->flash('alert-class', 'alert-success');
        }
    }
    
    public function deleteLine($lineId)
    {
        $line = CustomerOrderLines::find($lineId);
        if ($line) {
            $line->delete();
            
            if ($this->activeLineId == $lineId) {
                $this->activeLineId = null;
            }
            
            session()->flash('message', 'Item deleted');
            session()->flash('alert-class', 'alert-success');
        }
    }
    
    public function toggleMode()
    {
        $this->mode = $this->mode === 'single' ? 'quick' : 'single';
    }
    
    public function deleteFile($fileId)
    {
        $file = CustomerOrderFiles::find($fileId);
        if ($file) {
            Storage::disk('public')->delete($file->file);
            $file->delete();
            
            session()->flash('message', 'File deleted');
            session()->flash('alert-class', 'alert-success');
        }
    }
    
    // Size Management Methods
    public function addSize()
    {
        if (!$this->activeLineId || $this->readOnly) return;
        
        Gate::authorize('order:update');
        
        $line = CustomerOrderLines::find($this->activeLineId);
        if (!$line) return;
        
        // Create a new size with default name
        $size = Sizes::firstOrCreate(
            ['name' => 'NEW'],
            ['order' => 999]
        );
        
        // Add to pending sizes instead of creating immediately
        if (!isset($this->pendingSizes[$line->id])) {
            $this->pendingSizes[$line->id] = [];
        }
        
        // Generate a temporary unique ID for this pending size
        $tempId = 'pending_' . uniqid();
        
        $this->pendingSizes[$line->id][$tempId] = [
            'size_id' => $size->id,
            'size_name' => $size->name,
            'qty' => 0,
            'price' => 0,
            'discount' => 0,
            'commission' => 0,
        ];
        
        $this->hasUnsavedChanges = true;
        
        session()->flash('message', 'Size will be added on save');
        session()->flash('alert-class', 'alert-info');
    }
    
    public function addCustomerSizes()
    {
        if (!$this->activeLineId || $this->readOnly || !$this->order) return;
        
        Gate::authorize('order:update');
        
        $line = CustomerOrderLines::find($this->activeLineId);
        if (!$line) return;
        
        $customer = $this->order->customers;
        $defaultSizes = $customer->setting('size-defaults');
        
        if (empty($defaultSizes)) {
            session()->flash('message', 'No default sizes configured for this customer');
            session()->flash('alert-class', 'alert-warning');
            return;
        }
        
        $sizes = Sizes::whereIn('id', $defaultSizes)->get();
        
        // Add to pending sizes instead of creating immediately
        if (!isset($this->pendingSizes[$line->id])) {
            $this->pendingSizes[$line->id] = [];
        }
        
        foreach ($sizes as $size) {
            // Check if size already exists (either saved or pending)
            $existingSaved = $line->customer_order_line_quantities->where('sizes_id', $size->id)->first();
            $existingPending = collect($this->pendingSizes[$line->id])->where('size_id', $size->id)->first();
            
            if (!$existingSaved && !$existingPending) {
                // Generate a temporary unique ID for this pending size
                $tempId = 'pending_' . uniqid();
                
                $this->pendingSizes[$line->id][$tempId] = [
                    'size_id' => $size->id,
                    'size_name' => $size->name,
                    'qty' => 0,
                    'price' => 0,
                    'discount' => 0,
                    'commission' => 0,
                ];
            }
        }
        
        $this->hasUnsavedChanges = true;
        
        session()->flash('message', 'Customer default sizes will be added on save');
        session()->flash('alert-class', 'alert-info');
    }
    
    public function removeSize($quantityId)
    {
        if ($this->readOnly) return;
        
        Gate::authorize('order:update');
        
        $quantity = CustomerOrderLineQuantities::find($quantityId);
        if ($quantity) {
            // Mark for deletion instead of deleting immediately
            if (!isset($this->pendingSizeRemovals[$quantity->customer_order_lines_id])) {
                $this->pendingSizeRemovals[$quantity->customer_order_lines_id] = [];
            }
            
            $this->pendingSizeRemovals[$quantity->customer_order_lines_id][] = $quantityId;
            
            $this->hasUnsavedChanges = true;
            
            session()->flash('message', 'Size will be removed on save');
            session()->flash('alert-class', 'alert-info');
        }
    }
    
    public function undoRemoveSize($lineId, $quantityId)
    {
        if ($this->readOnly) return;
        
        Gate::authorize('order:update');
        
        if (isset($this->pendingSizeRemovals[$lineId])) {
            $key = array_search($quantityId, $this->pendingSizeRemovals[$lineId]);
            if ($key !== false) {
                unset($this->pendingSizeRemovals[$lineId][$key]);
                $this->pendingSizeRemovals[$lineId] = array_values($this->pendingSizeRemovals[$lineId]);
                
                // Clean up if empty
                if (empty($this->pendingSizeRemovals[$lineId])) {
                    unset($this->pendingSizeRemovals[$lineId]);
                }
                
                session()->flash('message', 'Size removal cancelled');
                session()->flash('alert-class', 'alert-success');
            }
        }
    }
    
    public function removePendingSize($lineId, $tempId)
    {
        if ($this->readOnly) return;
        
        Gate::authorize('order:update');
        
        if (isset($this->pendingSizes[$lineId][$tempId])) {
            unset($this->pendingSizes[$lineId][$tempId]);
            
            // Clean up if empty
            if (empty($this->pendingSizes[$lineId])) {
                unset($this->pendingSizes[$lineId]);
            }
            
            session()->flash('message', 'Pending size removed');
            session()->flash('alert-class', 'alert-success');
        }
    }
    
    public function addDrop()
    {
        if ($this->readOnly) return;
        
        Gate::authorize('order:update');
        
        if (!$this->activeLine) return;
        
        // Add to pending drops instead of creating immediately
        if (!isset($this->pendingDrops[$this->activeLine->id])) {
            $this->pendingDrops[$this->activeLine->id] = 0;
        }
        
        $dropIndex = $this->pendingDrops[$this->activeLine->id];
        $this->pendingDrops[$this->activeLine->id]++;
        
        // Initialize data structure for this pending drop
        if (!isset($this->pendingDropData[$this->activeLine->id])) {
            $this->pendingDropData[$this->activeLine->id] = [];
        }
        
        $this->pendingDropData[$this->activeLine->id][$dropIndex] = [
            'exfty' => null,
            'dropping' => [],
            'packed' => [],
        ];
        
        // Initialize quantities for each size
        foreach ($this->activeLine->customer_order_line_quantities as $qty) {
            $this->pendingDropData[$this->activeLine->id][$dropIndex]['dropping'][$qty->sizes_id] = 0;
            $this->pendingDropData[$this->activeLine->id][$dropIndex]['packed'][$qty->sizes_id] = 0;
        }
        
        $this->hasUnsavedChanges = true;
        
        session()->flash('message', 'Drop will be created on save');
        session()->flash('alert-class', 'alert-info');
    }
    
    public function deleteDrop($shipmentLineId)
    {
        if ($this->readOnly) return;
        
        Gate::authorize('order:update');
        
        $shipmentLine = ShipmentLine::find($shipmentLineId);
        if ($shipmentLine) {
            $shipmentLine->delete();
            
            $this->hasUnsavedChanges = true;
            
            // Reset active line to refresh the view
            $this->activeLineId = $this->activeLineId;
            
            session()->flash('message', 'Drop deleted');
            session()->flash('alert-class', 'alert-success');
        }
    }
    
    public function removePendingDrop($lineId)
    {
        if ($this->readOnly) return;
        
        Gate::authorize('order:update');
        
        if (isset($this->pendingDrops[$lineId]) && $this->pendingDrops[$lineId] > 0) {
            // Remove the last pending drop
            $lastIndex = $this->pendingDrops[$lineId] - 1;
            $this->pendingDrops[$lineId]--;
            
            // Remove the data for this drop
            if (isset($this->pendingDropData[$lineId][$lastIndex])) {
                unset($this->pendingDropData[$lineId][$lastIndex]);
            }
            
            // Remove the key if count reaches 0
            if ($this->pendingDrops[$lineId] === 0) {
                unset($this->pendingDrops[$lineId]);
                unset($this->pendingDropData[$lineId]);
            }
            
            session()->flash('message', 'Pending drop removed');
            session()->flash('alert-class', 'alert-success');
        }
    }
    
    public function updateSizeName($quantityId, $newName)
    {
        if ($this->readOnly || empty($newName)) return;
        
        Gate::authorize('order:update');
        
        $quantity = CustomerOrderLineQuantities::find($quantityId);
        if (!$quantity) return;
        
        $newName = trim($newName);
        if (empty($newName)) return;
        
        // Find or create size with new name
        $size = Sizes::firstOrCreate(
            ['name' => strtoupper($newName)],
            ['order' => 999]
        );
        
        // Update quantity to use new size
        $quantity->sizes_id = $size->id;
        $quantity->save();
        
        $this->hasUnsavedChanges = true;
        
        // Reset active line to refresh the view
        $this->activeLineId = $this->activeLineId;
        
        session()->flash('message', 'Size name updated');
        session()->flash('alert-class', 'alert-success');
    }
    
    public function updateQuantity($quantityId, $qty)
    {
        if ($this->readOnly) return;
        
        Gate::authorize('order:update');
        
        $quantity = CustomerOrderLineQuantities::find($quantityId);
        if ($quantity) {
            $quantity->qty = $qty ?? 0;
            $quantity->save();
            
            $this->hasUnsavedChanges = true;
        }
    }
    
    public function updateLineDate($field, $value)
    {
        if ($this->readOnly || !$this->activeLineId) return;
        
        Gate::authorize('order:update');
        
        $line = CustomerOrderLines::find($this->activeLineId);
        if ($line && in_array($field, ['factory_cust_date', 'wh_cust_date', 'start_knit'])) {
            $line->$field = $value ?: null;
            $line->save();
            
            $this->hasUnsavedChanges = true;
            
            // Reset active line to refresh the view
            $this->activeLineId = $this->activeLineId;
        }
    }
    
    public function updateDropExfty($shipmentLineId, $value)
    {
        if ($this->readOnly) return;
        
        Gate::authorize('order:update');
        
        $shipmentLine = ShipmentLine::find($shipmentLineId);
        if ($shipmentLine) {
            $shipmentLine->exfty = $value ?: null;
            $shipmentLine->save();
            
            $this->hasUnsavedChanges = true;
            
            // Reset active line to refresh the view
            $this->activeLineId = $this->activeLineId;
        }
    }
    
    public function render()
    {
        return view('livewire.sales.orders.orders-edit-v2');
    }
}

