<?php

namespace App\Http\Livewire\Sales\Orders;

use Livewire\Component;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use App\Models\CustomerOrderLines;
use App\Models\Price;
use App\Http\Livewire\BaseComponent;
use App\Livewire\Forms\OrdersEditLineForm;

class OrdersEditLine extends BaseComponent
{
    public OrdersEditLineForm $form;
    public $advanced = FALSE, $customer, $splitDrops = [], $availablePrices = [];

    public function render()
    {
        return view('livewire.sales.orders.orders-edit-line');
    }

    #[On('clear-line')]
    public function clearLine(){
        $this->reset();
    }


    #[On('edit-line')]
    public function mount($id = FALSE, $orderType = 'wholesale'){
        // dd($id);
        if($id){
            $line = CustomerOrderLines::with([
                'customer_order_line_quantities:id,customer_order_lines_id,sizes_id,qty,SKU,barcode,price,commission,discount,prices_id' => [
                    'sizes:id,name'
                ],
                'colourways:id,style_versions_id,name,cancelled,colour_type' => [
                    'style_versions:id,factory_id,styles_id,name' => [
                        'styles:id,designs_id,customer_ref' => [
                            'designs:id,description'
                        ],
                        'factories:id,name',
                        'prices:id,style_versions_id,quote',
                    ],
                ],
                'shipment_lines:id,customer_order_lines_id,exfty' => [
                    'shipment_line_sizes:id,sizes_id,shipment_line_id,qty,shipped_qty' => [
                        'sizes:id,name'
                    ],
                ],
                'customer_orders:id,customers_id' => [
                    'customers:id,settings,currency,default_discount,default_commission'
                ]
            ])->find($id);
            if($line){
                $this->form->open($line, $orderType);
                $this->customer = $line->customer_orders->customers;
                
                // Load available prices for this style version and colourway
                if ($line->colourways) {
                    $this->availablePrices = Price::with('sizes:id,name')
                        ->where('style_versions_id', $line->colourways->style_versions_id)
                        ->where(function($query) use ($line) {
                            $query->where('colourways_id', $line->colourways->id)
                                  ->orWhereNull('colourways_id');
                        })
                        ->orderByRaw("CASE WHEN colourways_id = ? THEN 2 WHEN colourways_id IS NULL THEN 1 ELSE 0 END DESC", [$line->colourways->id])
                        ->orderByRaw("CASE WHEN quote_status = 'confirmed' THEN 1 ELSE 0 END DESC")
                        ->orderBy('created_at', 'desc')
                        ->get()
                        ->map(function($price) {
                            return [
                                'id' => $price->id,
                                'label' => $this->formatPriceLabel($price),
                            ];
                        })
                        ->toArray();
                }
            }
        }
    }

    #[On('toggle-advanced-order')]
    public function toggleAdvanced(){
        $this->advanced = !$this->advanced;
    }

    #[Computed]
    public function viewOnly(){
        return $this->form->viewOnly();
    }


    #[On('save-order-line')]
    #[On('save-order')]
    public function saveOrder(){
        if(!empty($this->form->orderLineEdit))
            $this->form->save();
    }

    public function removeSize($s){
        $this->form->removeSize($s);
    }

    public function addSize(){
        $this->form->addSize();
    }

    public function addCustSizes(){
        $this->form->addCustSizes($this->customer);
    }

    public function addDrop(){
        $this->form->addDrop();
    }
    public function deleteDrop($d, $drop = NULL){
        $this->form->deleteDrop($d, $drop);
    }

    public function splitDropIntoNewOrder($dropIndex){
        $this->form->splitDropIntoNewOrder($dropIndex);
    }

    /**
     * Format price label for dropdown display
     */
    private function formatPriceLabel($price)
    {
        $parts = [];
        
        // Add price name if available
        if ($price->name) {
            $parts[] = $price->name;
        }
        
        // Add quote and CMT
        if ($price->quote) {
            $parts[] = "Quote: {$price->quote}";
        }
        if ($price->cmt) {
            $parts[] = "CMT: {$price->cmt}";
        }
        
        // Add size info if specific
        if ($price->sizes_id) {
            $sizeName = $price->sizes->name ?? "Size ID: {$price->sizes_id}";
            $parts[] = "Size: {$sizeName}";
        }
        
        // Add phase info if specific
        if ($price->phase_id) {
            $parts[] = "Phase: {$price->phase_id}";
        }
        
        // Add status indicators
        $statusParts = [];
        if ($price->quote_status === 'confirmed') {
            $statusParts[] = 'Quote ✓';
        }
        if ($price->cmt_status === 'confirmed') {
            $statusParts[] = 'CMT ✓';
        }
        if (!empty($statusParts)) {
            $parts[] = '(' . implode(', ', $statusParts) . ')';
        }
        
        // Add colourway specific indicator
        if ($price->colourways_id) {
            $parts[] = '[Colourway Specific]';
        } else {
            $parts[] = '[Generic]';
        }
        
        return implode(' | ', $parts) ?: "Price ID: {$price->id}";
    }
}
