<?php

namespace App\Http\Livewire\Production\Yarn;

use Livewire\Component;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\YarnOrder;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Orders\YarnOrders;
use App\Http\Livewire\Traits\Filterable;
use App\Livewire\Forms\YarnOrderEditForm;
use Illuminate\Database\Query\JoinClause;
use App\Http\Livewire\FilterableComponent;


class YarnOrdersView extends FilterableComponent
{
    use WithPagination;
    use Filterable;

	public $hideFilters = false;

	#[Validate([
		'paginate' => ['required', 'numeric', 'min:10', 'max:1000'],
		'customer' => ['nullable', 'exists:customers,id'], // Assuming 'customer' is an ID
		'spinner' => ['nullable', 'boolean'], // Assuming 'spinner' is a boolean flag
		'search' => ['nullable', 'string', 'max:30'],
		])]
	public $paginate = 10, $customer, $spinner, $search;

	protected function filterKey(): string
    {
        return 'yarnpofilters';
    }

    protected function columnsVersion(): float
    {
        return 1;
    }

    protected function filters(): array
    {
        return ['customer', 'spinner', 'search', 'paginate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['customer', 'spinner', 'search', 'paginate'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }


    public function mount()
    {
        $this->loadFilterSettings();
    }

	public $editID, $po;

	public $new = 0;

    // public $orders;


	#[Computed]
	public function customers(){
		return Customer::select('id', 'name')->get();
	}
	#[Computed]
	public function spinners(){
		return Suppliers::select('id', 'name')->has('yarns')->get();
	}

	#[On('close-modal')]
    public function render()
    {
		Gate::authorize('order:read');
		$orders = YarnOrder::select(
			'yarn_orders.id',
			'yarn_orders.date',
			'suppliers.name as spinner',
			'customers.name as customer',
			)
		->selectRaw(
			"(SELECT count(*)
			FROM yarn_order_lines yol
			WHERE yol.yarn_order_id=yarn_orders.id && yol.deleted_at is null) as yarn_count"
		)
		->selectRaw("(SELECT CONCAT('[',GROUP_CONCAT(JSON_OBJECT('id', yol.id, 'price', yol.price, 'qty', yol.qty, 'yarn', y.description, 'count', c.count, 'colour', yc.description, 'colour_id', yc.id, 'colour_ref', yc.reference,
					'styles',
						(SELECT
							GROUP_CONCAT(COALESCE(NULLIF(s.customer_ref, ''), s.designs_id))
							FROM yarn_order_line_styles yols
							JOIN customer_order_lines col ON col.id=yols.customer_order_lines_id
							JOIN colourways cw ON cw.id=col.colourways_id
							JOIN style_versions sv ON sv.id=cw.style_versions_id
							JOIN styles s ON s.id=sv.styles_id
							WHERE yols.yarn_order_lines_id=yol.id
							AND yols.deleted_at IS NULL
						)
			)),']')
			FROM yarn_order_lines yol
			JOIN yarn_colours yc on yc.id=yol.yarn_colours_id
			JOIN yarns y on y.id=yc.yarn_id
			JOIN counts c on c.id=y.counts_id
			WHERE yol.yarn_order_id=yarn_orders.id && yol.deleted_at is null) as yarns")
		->search($this->search)
		->join('customers', function (JoinClause $join) {
			$join->on('customers.id', '=', 'yarn_orders.customers_id');
			if(!empty($this->customer))
				$join->where('customers.id', $this->customer);
		})
		->join('suppliers', function (JoinClause $join) {
			$join->on('suppliers.id', '=', 'yarn_orders.suppliers_id');
			if(!empty($this->spinner))
				$join->where('suppliers.id', $this->spinner);
		})
		->orderByDesc('id')
		->paginate($this->paginate);


        return view('livewire.production.yarn.yarn-orders-view', ['orders' => $orders]);
    }

    public function sumQty($orders){
        $total = 0;
        foreach($orders as $order){
            foreach(json_decode($order->yarns) ?? [] as $yarn){
                $total = $total + $yarn->qty;
            }
        }
        return $total;
    }
}
