<?php

namespace App\Http\Livewire\Production\Shipments;

use Livewire\Component;
use App\Models\Customer;
use App\Models\Shipment;
use App\Models\Suppliers;
use App\Models\Colourways;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use App\Models\StyleVersions;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;

class ShipmentsView extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideFilters = false;

    #[Validate([
        'customer' => ['nullable', 'exists:customers,id'],
        'transporter'  => ['nullable', 'exists:suppliers,id'],
        'search'   => ['nullable', 'string', 'max:30'],
        'paginate' => ['required', 'numeric', 'min:10', 'max:1000'],
    ])]
    public $transporter, $customer, $paginate, $search;

    public $columns = [
        'Shipment No' => true,
        'Created' => true,
        'Organiser' => true,
        'Transporter' => true,
        'Price' => true,
        'Qty' => true,
        'Shipped Qty' => true,
        'Factory' => true,
        'Style' => false,
        'Description' => false,
        'Customer' => true,
    ];

    protected function filterKey(): string
    {
        return 'shipmentsfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.4;
    }

    protected function filters(): array
    {
        return ['customer', 'transporter', 'search', 'paginate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['customer', 'transporter', 'search', 'paginate'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }

    public function mount()
    {
        $this->loadFilterSettings();
    }

    #[Computed]
    public function transporters(){
        return Suppliers::where('type', 'transporter')->get();
    }
    #[Computed]
    public function customers(){
        return Customer::allCached();
    }

    #[On('close-modal')]
    public function render()
    {
        Gate::authorize('shipment:read');
        $shipments = Shipment::query()
        ->select(['id', 'organiser', 'transporter_id', 'cost', 'created_at'])
        ->with([
            'transporters:id,name',
            'shipment_lines' => function ($q) {
                $q->select(['id', 'shipment_id', 'customer_order_lines_id']);
            },
            'shipment_lines.shipment_line_sizes:id,shipment_line_id,qty,shipped_qty',
            'shipment_lines.customer_order_lines' => function ($q) {
                $q->select(['id', 'customer_orders_id', 'colourways_id']);
            },
            'shipment_lines.customer_order_lines.customer_orders:id,customers_id',
            'shipment_lines.customer_order_lines.customer_orders.customers:id,name',
            'shipment_lines.customer_order_lines.colourways:id,style_versions_id',
            'shipment_lines.customer_order_lines.colourways.style_versions:id,factory_id',
            'shipment_lines.customer_order_lines.colourways.style_versions.factories:id,name',
        ])
        ->search($this->search)
        ->when(!empty($this->transporter),function($query){
            $query->where('transporter_id', $this->transporter);
        })
        ->when(!empty($this->customer),function($query){
            $query->whereHas('shipment_lines.customer_order_lines.customer_orders',function($query){
                $query->where('customers_id', $this->customer);
            });
        })->orderByDesc('id')->paginate($this->paginate);
        // dd(CustomerOrderLines::with('colourways')->first());
        // dd($shipments->first());
        return view('livewire.production.shipments.shipments-view', [
            'shipments' => $shipments,
        ]);
    }
}
