<?php

namespace App\Http\Livewire\Production\Reports;

use Livewire\Component;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Models\Shipment;
use Livewire\Attributes\On;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\CustomerOrders;

class ShipmentScheduleSplitDrop extends Component
{
    public $dropId;
    public $drop;
    public $step = 1;
    public $totalSteps = 3;
    
    // Step 1: Order quantities and current drops
    public $orderQtys = [];
    public $currentDropQtys = [];
    public $currentDropShippedQtys = [];
    public $existingDrops = [];
    public $originalExistingDrops = [];
    
    // Step 2: New drop date, quantities and configuration
    public $newDropDates = [];
    public $newDropCount = 1; // Always 1 drop
    public $createNewOrder = false; // New property to track if we should create a new order
    public $newDropQtys = [];
    public $editableDrops = [];
    public $takeFromExisting = true;
    
    // Step 3: Confirmation
    public $overview = [];
    public $existingDropChanges = [];
    public $newOrderOverview = [];
    
    // Error handling
    public $errorMessage = '';
    public $successMessage = '';

    #[On('open-split-drop-modal')]
    public function mount($dropId = null)
    {
        if ($dropId) {
            $this->dropId = $dropId;
            $this->loadDrop();
            $this->initializeStep1();
        }
    }

    #[On('open-split-drop-modal')]
    public function openSplitDrop($dropId)
    {
        $this->dropId = $dropId;
        $this->step = 1;
        $this->errorMessage = '';
        $this->successMessage = '';
        $this->createNewOrder = false; // Reset the new order flag
        $this->loadDrop();
        $this->initializeStep1();
    }

    public function addDrop()
    {
        $this->newDropCount++;
        $this->initializeNewDrops();
    }

    public function removeDrop($index)
    {
        if ($this->newDropCount > 1) {
            $this->newDropCount--;
            unset($this->newDropDates[$index]);
            unset($this->newDropQtys[$index]);
            $this->initializeNewDrops();
        }
    }

    public function initializeNewDrops()
    {
        // Store existing dates to preserve them
        $existingDates = $this->newDropDates;
        $existingQtys = $this->newDropQtys;
        
        $this->newDropDates = [];
        $this->newDropQtys = [];
        
        for ($i = 0; $i < $this->newDropCount; $i++) {
            // Preserve existing date if it exists, otherwise set to empty
            $this->newDropDates[$i] = $existingDates[$i] ?? '';
            $this->newDropQtys[$i] = [];
            
            foreach ($this->orderQtys as $sizeId => $orderData) {
                $this->newDropQtys[$i][$sizeId] = [
                    'size_name' => $orderData['size_name'],
                    'qty' => $existingQtys[$i][$sizeId]['qty'] ?? 0
                ];
            }
        }
    }

    public function updateDropQuantity($dropIndex, $sizeId, $value)
    {
        // Ensure the array structure exists before updating
        if (!isset($this->newDropQtys[$dropIndex])) {
            $this->newDropQtys[$dropIndex] = [];
        }
        
        if (!isset($this->newDropQtys[$dropIndex][$sizeId])) {
            $this->newDropQtys[$dropIndex][$sizeId] = [
                'size_name' => $this->orderQtys[$sizeId]['size_name'] ?? 'Unknown',
                'qty' => 0
            ];
        }
        
        $this->newDropQtys[$dropIndex][$sizeId]['qty'] = (int)$value;
    }

    public function updateExistingDropQuantity($dropId, $sizeId, $value)
    {
        // Find the drop in existing drops and update it
        foreach ($this->existingDrops as &$drop) {
            if ($drop['id'] == $dropId) {
                // Prevent edits to shipped drops
                if (!isset($drop['complete']) || !$drop['complete']) {
                    $drop['quantities'][$sizeId] = (int)$value;
                    $drop['modified'] = true;
                }
                break;
            }
        }
    }

    public function updatedTakeFromExisting()
    {
        if ($this->takeFromExisting && $this->canEditExistingDrops()) {
            $this->updateExistingDropQuantities();
        }
    }

    public function updateExistingDropQuantities()
    {
        if ($this->takeFromExisting && $this->canEditExistingDrops()) {
            $unshippedIndex = $this->getUnshippedDropIndex();
            if ($unshippedIndex === null) {
                return;
            }
            foreach ($this->orderQtys as $sizeId => $orderData) {
                $newDropsQty = 0;
                foreach ($this->newDropQtys as $dropQtys) {
                    $newDropsQty += (int)($dropQtys[$sizeId]['qty'] ?? 0);
                }
                // Base on original unshipped drop qty snapshot
                $originalQty = (int)($this->originalExistingDrops[$unshippedIndex]['quantities'][$sizeId] ?? 0);
                $this->existingDrops[$unshippedIndex]['quantities'][$sizeId] = max(0, $originalQty - $newDropsQty);
            }
        }
    }

    public function updatedNewDropQtys()
    {
        // Ensure the array structure is correct after Livewire updates
        $this->ensureNewDropQtysStructure();
        
        // Update existing drop quantities whenever new drop quantities change
        if ($this->takeFromExisting && $this->canEditExistingDrops()) {
            $this->updateExistingDropQuantities();
        }
    }

    public function updatedCreateNewOrder()
    {
        // Dispatch event for UI updates
        $this->dispatch('new-order-checkbox-changed', [
            'createNewOrder' => $this->createNewOrder
        ]);
        
        // Clear any previous error messages
        $this->errorMessage = '';
        
        // If checking, validate that we can proceed
        if ($this->createNewOrder) {
            if (!$this->canCreateNewOrder()) {
                $this->errorMessage = $this->getNewOrderWarningMessage();
                $this->createNewOrder = false;
                return;
            }
        }
    }

    public function canEditExistingDrops()
    {
        $unshippedCount = 0;
        foreach ($this->existingDrops as $drop) {
            if (!isset($drop['complete']) || !$drop['complete']) {
                $unshippedCount++;
            }
        }
        
        // Only allow editing if there's exactly one unshipped drop
        return $unshippedCount === 1;
    }

    /**
     * Get the index of the single unshipped existing drop; null if zero/multiple.
     */
    protected function getUnshippedDropIndex(): ?int
    {
        $index = null;
        foreach ($this->existingDrops as $i => $drop) {
            if (!isset($drop['complete']) || !$drop['complete']) {
                if ($index !== null) {
                    return null; // multiple unshipped
                }
                $index = $i;
            }
        }
        return $index;
    }

    public function canShowTakeFromExistingCheckbox()
    {
        return $this->canEditExistingDrops();
    }

    public function canCreateNewOrder()
    {
        // Check if user has permission to create orders
        if (!Gate::allows('order:create')) {
            return false;
        }

        // Check if we have a valid drop with order lines
        if (!$this->drop || !$this->drop->customer_order_lines) {
            return false;
        }

        // Check if there are blocking issues
        if ($this->hasBlockingIssues()) {
            return false;
        }

        // Check if we have valid quantities for new order creation
        if (!$this->validateNewOrderCreation()) {
            return false;
        }

        return true;
    }

    public function getNewOrderWarningMessage()
    {
        if (!$this->drop || !$this->drop->customer_order_lines) {
            return 'Unable to create new order: Drop or order line not found.';
        }
        
        if (!Gate::allows('order:create')) {
            return 'Unable to create new order: You do not have permission to create orders.';
        }
        
        $hasQuantities = false;
        foreach ($this->currentDropQtys as $sizeId => $data) {
            if ($data['qty'] > 0) {
                $hasQuantities = true;
                break;
            }
        }
        
        if (!$hasQuantities) {
            return 'Unable to create new order: The current drop has no quantities to split.';
        }
        
        // Check for blocking issues
        $blockingIssues = $this->hasBlockingIssues();
        if (!empty($blockingIssues)) {
            return 'Unable to create new order: ' . implode(' ', $blockingIssues);
        }
        
        return null;
    }

    public function getTotalQuantitiesForNewOrder()
    {
        $total = [];
        
        foreach ($this->orderQtys as $sizeId => $orderData) {
            $total[$sizeId] = [
                'size_name' => $orderData['size_name'],
                'total_qty' => 0
            ];
            
            foreach ($this->newDropQtys as $dropQtys) {
                $total[$sizeId]['total_qty'] += $dropQtys[$sizeId]['qty'] ?? 0;
            }
        }
        
        return $total;
    }

    public function isSplitReasonable()
    {
        if (!$this->createNewOrder) {
            return true;
        }
        
        $totalNewOrderQty = 0;
        $totalOriginalQty = 0;
        
        foreach ($this->orderQtys as $sizeId => $orderData) {
            $totalOriginalQty += $orderData['qty'];
            
            foreach ($this->newDropQtys as $dropQtys) {
                $totalNewOrderQty += $dropQtys[$sizeId]['qty'] ?? 0;
            }
        }
        
        // The split should leave at least 20% of quantities in the original order
        $minimumRemaining = $totalOriginalQty * 0.2;
        $remainingInOriginal = $totalOriginalQty - $totalNewOrderQty;
        
        return $remainingInOriginal >= $minimumRemaining;
    }

    public function hasBlockingIssues()
    {
        $issues = [];
        
        if ($this->createNewOrder) {
            // Check if the drop is already shipped
            if ($this->drop->complete) {
                $issues[] = 'Cannot split a drop that is already shipped.';
            }
            
            // Check if there are any existing drops that might conflict
            if (count($this->existingDrops) > 1) {
                $issues[] = 'Cannot create new order when there are multiple existing drops.';
            }
            
            // Check if the split is reasonable
            if (!$this->isSplitReasonable()) {
                $issues[] = 'The split is not reasonable for business purposes.';
            }
        }
        
        return $issues;
    }

    public function getCurrentStepDescription()
    {
        switch ($this->step) {
            case 1:
                return 'Order Overview';
            case 2:
                return 'Drop Configuration & Quantities';
            case 3:
                return $this->createNewOrder ? 'Order Comparison' : 'Drops Overview';
            default:
                return 'Unknown Step';
        }
    }

    public function getNewOrderSummary()
    {
        if (!$this->createNewOrder) {
            return null;
        }
        
        $summary = [
            'original_order' => [
                'po' => $this->drop->customer_order_lines->customer_orders->customer_po ?? 'N/A',
                'customer' => $this->drop->customer_order_lines->customer_orders->customers->name ?? 'N/A',
                'remaining_quantities' => []
            ],
            'new_order' => [
                'po' => ($this->drop->customer_order_lines->customer_orders->customer_po ?? 'N/A') . '-SPLIT-' . date('Ymd-His'),
                'quantities' => []
            ]
        ];
        
        foreach ($this->orderQtys as $sizeId => $orderData) {
            $totalNewDropQty = 0;
            foreach ($this->newDropQtys as $dropQtys) {
                $totalNewDropQty += $dropQtys[$sizeId]['qty'] ?? 0;
            }
            
            $remainingInOriginal = $orderData['qty'] - $totalNewDropQty;
            
            $summary['original_order']['remaining_quantities'][$sizeId] = [
                'size_name' => $orderData['size_name'],
                'qty' => $remainingInOriginal
            ];
            
            $summary['new_order']['quantities'][$sizeId] = [
                'size_name' => $orderData['size_name'],
                'qty' => $totalNewDropQty
            ];
        }
        
        return $summary;
    }

    public function getFinalConfirmationMessage()
    {
        if (!$this->createNewOrder) {
            return null;
        }
        
        $totalNewOrderQty = 0;
        $totalOriginalQty = 0;
        
        foreach ($this->orderQtys as $sizeId => $orderData) {
            $totalOriginalQty += $orderData['qty'];
            foreach ($this->newDropQtys as $dropQtys) {
                $totalNewOrderQty += $dropQtys[$sizeId]['qty'] ?? 0;
            }
        }
        
        $remainingInOriginal = $totalOriginalQty - $totalNewOrderQty;
        $splitPercentage = round(($totalNewOrderQty / $totalOriginalQty) * 100, 1);
        
        return [
            'total_original_qty' => $totalOriginalQty,
            'total_new_order_qty' => $totalNewOrderQty,
            'remaining_in_original' => $remainingInOriginal,
            'split_percentage' => $splitPercentage,
            'is_reasonable' => $this->isSplitReasonable()
        ];
    }

    public function validateNewOrderCreation()
    {
        if (!$this->createNewOrder) {
            return true;
        }
        
        // Check if we have quantities to split
        $totalNewDropQty = 0;
        foreach ($this->orderQtys as $sizeId => $orderData) {
            foreach ($this->newDropQtys as $dropQtys) {
                $totalNewDropQty += $dropQtys[$sizeId]['qty'] ?? 0;
            }
        }
        
        if ($totalNewDropQty === 0) {
            $this->errorMessage = 'The new drop must have quantities greater than 0 when creating a new order';
            return false;
        }
        
        // Check if the split makes sense (not taking everything from original order)
        $totalOriginalQty = 0;
        foreach ($this->orderQtys as $sizeId => $orderData) {
            $totalOriginalQty += $orderData['qty'];
        }
        
        if ($totalNewDropQty >= $totalOriginalQty) {
            $this->errorMessage = 'When creating a new order, you must leave some quantities in the original order';
            return false;
        }
        
        // Check if the split is reasonable for business purposes
        if (!$this->isSplitReasonable()) {
            $this->errorMessage = 'The split is not reasonable. Please ensure at least 20% of quantities remain in the original order.';
            return false;
        }
        
        return true;
    }

    public function createNewOrder()
    {
        if (!$this->drop || !$this->drop->customer_order_lines) {
            return null;
        }

        // Check permissions
        if (!Gate::allows('order:create')) {
            return null;
        }

        try {
            $originalOrder = $this->drop->customer_order_lines->customer_orders;
            
            // Create a new order with the same details
            $newOrder = $originalOrder->replicate();
            $newOrder->customer_po = $originalOrder->customer_po . '-SPLIT-' . date('Ymd-His');
            $newOrder->order_date = now();
            $newOrder->created_at = now();
            $newOrder->updated_at = now();
            $newOrder->save();

            // Create a new customer order line
            $newOrderLine = $this->drop->customer_order_lines->replicate();
            $newOrderLine->customer_orders_id = $newOrder->id;
            $newOrderLine->created_at = now();
            $newOrderLine->updated_at = now();
            $newOrderLine->save();

            // Create customer order line quantities for the new order
            foreach ($this->orderQtys as $sizeId => $orderData) {
                $originalQty = $this->drop->customer_order_lines->customer_order_line_quantities
                    ->where('sizes_id', $sizeId)
                    ->first();
                
                if ($originalQty) {
                    $newQty = $originalQty->replicate();
                    $newQty->customer_order_lines_id = $newOrderLine->id;
                    $newQty->qty = 0; // Start with 0, will be updated based on new drop quantities
                    $newQty->created_at = now();
                    $newQty->updated_at = now();
                    $newQty->save();
                }
            }

            return [
                'order' => $newOrder,
                'order_line' => $newOrderLine
            ];
        } catch (\Exception $e) {
            \Log::error('Failed to create new order: ' . $e->getMessage());
            return null;
        }
    }

    public function loadDrop()
    {
        $this->drop = ShipmentLine::with([
            'customer_order_lines.customer_order_line_quantities.sizes',
            'shipment_line_sizes.sizes',
            'shipments'
        ])->find($this->dropId);
        
        if (!$this->drop) {
            $this->errorMessage = 'Drop not found';
            return;
        }
    }

    public function initializeStep1()
    {
        // Get order quantities from customer order line quantities
        $this->orderQtys = [];
        $this->currentDropQtys = [];
        $this->currentDropShippedQtys = [];
        $this->existingDrops = [];
        
        foreach ($this->drop->customer_order_lines->customer_order_line_quantities as $qty) {
            $sizeId = $qty->sizes_id;
            $sizeName = $qty->sizes->name;
            
            $this->orderQtys[$sizeId] = [
                'size_name' => $sizeName,
                'qty' => $qty->qty
            ];
            
            // Get current drop quantities and shipped quantities
            $currentQty = $this->drop->shipment_line_sizes
                ->where('sizes_id', $sizeId)
                ->first();
                
            $this->currentDropQtys[$sizeId] = [
                'size_name' => $sizeName,
                'qty' => $currentQty ? $currentQty->qty : 0
            ];
            
            // Initialize shipped qty to 0, will calculate total after loading existing drops
            $this->currentDropShippedQtys[$sizeId] = [
                'size_name' => $sizeName,
                'shipped_qty' => 0
            ];
        }
        
        // Get all existing drops for this order line
        $this->loadExistingDrops();
        
        // Calculate total shipped quantity across all drops for each size
        foreach ($this->orderQtys as $sizeId => $orderData) {
            $totalShipped = 0;
            foreach ($this->existingDrops as $drop) {
                $totalShipped += $drop['shipped_quantities'][$sizeId] ?? 0;
            }
            $this->currentDropShippedQtys[$sizeId]['shipped_qty'] = $totalShipped;
        }
        
        // Initialize new drops with default values
        $this->initializeNewDrops();
    }

    public function loadExistingDrops()
    {
        // Get all drops for the same customer order line
        $this->existingDrops = ShipmentLine::where('customer_order_lines_id', $this->drop->customer_order_lines_id)
            ->with(['shipment_line_sizes.sizes', 'shipments'])
            ->orderBy('exfty')
            ->get()
            ->map(function ($drop) {
                return [
                    'id' => $drop->id,
                    'exfty' => $drop->exfty,
                    'complete' => $drop->complete,
                    'quantities' => $drop->shipment_line_sizes->mapWithKeys(function ($size) {
                        return [$size->sizes_id => $size->qty];
                    })->toArray(),
                    'shipped_quantities' => $drop->shipment_line_sizes->mapWithKeys(function ($size) {
                        return [$size->sizes_id => $size->shipped_qty];
                    })->toArray(),
                    'total_pieces' => $drop->total_pieces
                ];
            })
            ->toArray();

        // Keep an original snapshot to compute diffs for confirmation
        $this->originalExistingDrops = $this->existingDrops;
    }

    public function nextStep()
    {
        if ($this->validateStep()) {
            $this->step++;
            $this->errorMessage = '';
        }
    }

    public function previousStep()
    {
        $this->step--;
        $this->errorMessage = '';
    }

    public function validateStep()
    {
        switch ($this->step) {
            case 1:
                // Step 1 is just display, always valid
                return true;
                
            case 2:
                // Validate that new drop quantities don't exceed order quantities
                $totalExceeds = false;
                $totalAllocated = [];
                
                // Initialize totals for each size
                foreach ($this->orderQtys as $sizeId => $orderData) {
                    $totalAllocated[$sizeId] = 0;
                }
                
                // Sum up all new drop quantities
                foreach ($this->newDropQtys as $dropQtys) {
                    foreach ($dropQtys as $sizeId => $data) {
                        $totalAllocated[$sizeId] += $data['qty'];
                    }
                }
                
                // Check if any size exceeds order quantity
                foreach ($totalAllocated as $sizeId => $total) {
                    if ($total > $this->orderQtys[$sizeId]['qty']) {
                        $totalExceeds = true;
                        break;
                    }
                }
                
                if ($totalExceeds) {
                    $this->errorMessage = 'Drop quantities cannot exceed order quantities';
                    return false;
                }
                
                // If creating a new order, validate that quantities are reasonable
                if ($this->createNewOrder) {
                    if (!$this->validateNewOrderCreation()) {
                        return false;
                    }
                }
                
                // When creating a new order, require at least one new drop to have quantities
                if ($this->createNewOrder) {
                    $hasQuantity = false;
                    foreach ($this->newDropQtys as $dropQtys) {
                        foreach ($dropQtys as $data) {
                            if (($data['qty'] ?? 0) > 0) {
                                $hasQuantity = true;
                                break 2;
                            }
                        }
                    }
                    
                    if (!$hasQuantity) {
                        $this->errorMessage = 'The new drop must have quantities greater than 0 when creating a new order';
                        return false;
                    }
                }
                
                // Validate the new drop date
                if (empty($this->newDropDates[0])) {
                    $this->errorMessage = 'Drop date must be selected';
                    return false;
                }
                
                if (!$this->drop || !$this->drop->exfty) {
                    $this->errorMessage = 'Invalid drop or missing exfty date';
                    return false;
                }
                
                try {
                    $this->validate([
                        "newDropDates.0" => 'required|date|after:' . $this->drop->exfty
                    ]);
                } catch (\Exception $e) {
                    $this->errorMessage = 'Drop date must be after ' . $this->drop->exfty;
                    return false;
                }
                
                return true;
                
            case 3:
                // If creating a new order, validate that quantities are reasonable
                if ($this->createNewOrder) {
                    if (!$this->validateNewOrderCreation()) {
                        return false;
                    }
                }
                return true;
        }
        
        return false;
    }

    public function prepareOverview()
    {
        $this->overview = [];
        $this->existingDropChanges = [];
        
        foreach ($this->orderQtys as $sizeId => $orderData) {
            $totalExistingDropsQty = 0;
            $totalExistingDropsShipped = 0;
            $totalNewDropQty = 0;
            
            // Sum up quantities across all existing drops
            foreach ($this->existingDrops as $existingDrop) {
                $totalExistingDropsQty += $existingDrop['quantities'][$sizeId] ?? 0;
                $totalExistingDropsShipped += $existingDrop['shipped_quantities'][$sizeId] ?? 0;
            }
            
            // Sum up quantities across all new drops
            foreach ($this->newDropQtys as $dropQtys) {
                $totalNewDropQty += $dropQtys[$sizeId]['qty'] ?? 0;
            }
            
            $this->overview[$sizeId] = [
                'size_name' => $orderData['size_name'],
                'order_qty' => $orderData['qty'],
                'total_new_drop_qty' => $totalNewDropQty,
                'existing_drops_qty' => $totalExistingDropsQty,
                'total_shipped' => $totalExistingDropsShipped,
                'remaining' => $orderData['qty'] - $totalExistingDropsShipped
            ];
        }

        // Build explicit change list for existing drops (unshipped only)
        foreach ($this->existingDrops as $idx => $drop) {
            if (!isset($drop['complete']) || $drop['complete']) {
                continue; // shipped or undefined state; skip
            }
            $original = $this->originalExistingDrops[$idx] ?? null;
            if (!$original) {
                $original = $drop; // fallback; no diff
            }

            $sizeChanges = [];
            $totalDelta = 0;
            foreach ($this->orderQtys as $sizeId => $orderData) {
                $oldQty = (int)($original['quantities'][$sizeId] ?? 0);
                // If auto-calc is enabled for a single existing drop, compute the prospective new value
                $unIdx = $this->getUnshippedDropIndex();
                if ($this->takeFromExisting && $this->canEditExistingDrops() && $unIdx !== null && $idx === $unIdx) {
                    $newDropsQty = 0;
                    foreach ($this->newDropQtys as $dq) {
                        $newDropsQty += (int)($dq[$sizeId]['qty'] ?? 0);
                    }
                    $originalQtyForIdx = (int)($this->originalExistingDrops[$idx]['quantities'][$sizeId] ?? 0);
                    $newQty = max(0, $originalQtyForIdx - $newDropsQty);
                } else {
                    $newQty = (int)($drop['quantities'][$sizeId] ?? 0);
                }
                if ($oldQty !== $newQty) {
                    $delta = $newQty - $oldQty;
                    $totalDelta += $delta;
                    $sizeChanges[] = [
                        'size_id' => $sizeId,
                        'size_name' => $orderData['size_name'],
                        'old' => $oldQty,
                        'new' => $newQty,
                        'delta' => $delta,
                    ];
                }
            }

            if (!empty($sizeChanges)) {
                $this->existingDropChanges[] = [
                    'id' => $drop['id'],
                    'exfty' => $drop['exfty'],
                    'size_changes' => $sizeChanges,
                    'total_delta' => $totalDelta,
                ];
            }
        }
        
        // If creating a new order, prepare the new order overview
        if ($this->createNewOrder) {
            $this->prepareNewOrderOverview();
        }
    }
    
    public function prepareNewOrderOverview()
    {
        $this->newOrderOverview = [];
        
        foreach ($this->orderQtys as $sizeId => $orderData) {
            $totalNewDropQty = 0;
            
            // Sum up quantities across all new drops
            foreach ($this->newDropQtys as $dropQtys) {
                $totalNewDropQty += $dropQtys[$sizeId]['qty'] ?? 0;
            }
            
            $this->newOrderOverview[$sizeId] = [
                'size_name' => $orderData['size_name'],
                'new_order_qty' => $totalNewDropQty,
                'original_order_qty' => $orderData['qty']
            ];
        }
    }

    public function goToStep($step)
    {
        if ($step >= 1 && $step <= $this->totalSteps) {
            $this->step = $step;
            $this->errorMessage = '';
        }
    }

    public function validateSizeMapping()
    {
        $errors = [];
        
        foreach ($this->newDropQtys as $dropIndex => $dropQtys) {
            foreach ($dropQtys as $sizeId => $data) {
                if ($data['qty'] > 0) {
                    // Check if this sizeId exists in orderQtys
                    if (!isset($this->orderQtys[$sizeId])) {
                        $errors[] = "Size ID {$sizeId} not found in order quantities for drop {$dropIndex}";
                    } else {
                        // Check if the size name matches
                        $expectedSizeName = $this->orderQtys[$sizeId]['size_name'];
                        $actualSizeName = $data['size_name'];
                        if ($expectedSizeName !== $actualSizeName) {
                            $errors[] = "Size name mismatch for ID {$sizeId}: expected '{$expectedSizeName}', got '{$actualSizeName}' in drop {$dropIndex}";
                        }
                    }
                }
            }
        }
        
        return $errors;
    }

    public function save()
    {
        if (!$this->validateStep()) {
            return;
        }

        // Validate size mapping before saving
        $sizeMappingErrors = $this->validateSizeMapping();
        if (!empty($sizeMappingErrors)) {
            $this->errorMessage = 'Size mapping errors: ' . implode(', ', $sizeMappingErrors);
            return;
        }

        try {
            DB::beginTransaction();
            
            $newOrderData = null;
            
            // Create new order if checkbox is checked
            if ($this->createNewOrder) {
                $newOrderData = $this->createNewOrder();
                if (!$newOrderData) {
                    throw new \Exception('Failed to create new order');
                }
                
                // Validate that the new order was created successfully
                if (!$newOrderData['order'] || !$newOrderData['order_line']) {
                    throw new \Exception('New order creation failed - missing order data');
                }
            }
            
            // Create new shipment lines for each drop
            foreach ($this->newDropQtys as $dropIndex => $dropQtys) {
                // Create a fresh new drop with only essential fields
                // Do NOT clone the original drop as it may have incorrect invoice numbers, notes, etc.
                $newShipmentLine = new ShipmentLine();
                $newShipmentLine->exfty = $this->newDropDates[$dropIndex];
                $newShipmentLine->complete = 0; // New drops should be marked as not complete
                $newShipmentLine->shipment_id = null;
                
                // Link to the appropriate order line
                if ($this->createNewOrder && $newOrderData) {
                    $newShipmentLine->customer_order_lines_id = $newOrderData['order_line']->id;
                } else {
                    // Link to the same order line as the original drop
                    $newShipmentLine->customer_order_lines_id = $this->drop->customer_order_lines_id;
                }
                
                $newShipmentLine->created_at = now();
                $newShipmentLine->updated_at = now();
                $newShipmentLine->save();
                
                // Create shipment line sizes for new drop
                foreach ($dropQtys as $sizeId => $data) {

                        $newSize = ShipmentLineSizes::create([
                            'shipment_line_id' => $newShipmentLine->id,
                            'sizes_id' => $sizeId, // Use the sizeId directly - it's already correct!
                            'qty' => $data['qty'] ?? 0,
                            'shipped_qty' => 0
                        ]);

                }
            }
            
            // Update new order quantities if creating a new order
            if ($this->createNewOrder && $newOrderData) {
                foreach ($this->orderQtys as $sizeId => $orderData) {
                    $totalNewDropQty = 0;
                    foreach ($this->newDropQtys as $dropQtys) {
                        $totalNewDropQty += $dropQtys[$sizeId]['qty'] ?? 0;
                    }
                    
                    // Update the new order line quantity
                    $newOrderQty = $newOrderData['order_line']->customer_order_line_quantities
                        ->where('sizes_id', $sizeId)
                        ->first();
                    
                    if ($newOrderQty) {
                        $newOrderQty->update(['qty' => $totalNewDropQty]);
                    } else {
                        throw new \Exception('Failed to update new order quantities');
                    }
                }
            }
            
            // Update existing drop quantities if taking from existing
            if ($this->takeFromExisting) {
                $totalToRemove = [];
                
                // Calculate total quantities to remove from existing drop
                foreach ($this->newDropQtys as $dropQtys) {
                    foreach ($dropQtys as $sizeId => $data) {
                        if ($data['qty'] > 0) {
                            // Find the correct size ID by matching the size name
                            $correctSizeId = null;
                            foreach ($this->orderQtys as $dbSizeId => $orderData) {
                                if ($orderData['size_name'] === $data['size_name']) {
                                    $correctSizeId = $dbSizeId;
                                    break;
                                }
                            }
                            
                            if ($correctSizeId) {
                                if (!isset($totalToRemove[$correctSizeId])) {
                                    $totalToRemove[$correctSizeId] = 0;
                                }
                                $totalToRemove[$correctSizeId] += $data['qty'];
                            }
                        }
                    }
                }
                
                // Remove quantities from only the unshipped existing drop using correct size IDs
                $unIdx = $this->getUnshippedDropIndex();
                if ($unIdx !== null) {
                    $unshippedDropId = $this->existingDrops[$unIdx]['id'] ?? null;
                    if ($unshippedDropId) {
                        $targetDrop = \App\Models\ShipmentLine::find($unshippedDropId);
                        if ($targetDrop) {
                            foreach ($totalToRemove as $correctSizeId => $totalQty) {
                                $existingSize = $targetDrop->shipment_line_sizes
                                    ->where('sizes_id', $correctSizeId)
                                    ->first();
                                if ($existingSize) {
                                    $newQty = max(0, $existingSize->qty - $totalQty);
                                    $existingSize->update(['qty' => $newQty]);
                                }
                            }
                        }
                    }
                }
            }
            
            // Update existing drops if quantities were modified
            foreach ($this->existingDrops as $dropData) {
                if (isset($dropData['modified']) && $dropData['modified']) {
                    $drop = \App\Models\ShipmentLine::find($dropData['id']);
                    // Do not allow modifying shipped drops here
                    if ($drop && !$drop->complete) {
                        foreach ($dropData['quantities'] as $sizeId => $qty) {
                            $size = $drop->shipment_line_sizes->where('sizes_id', $sizeId)->first();
                            if ($size) {
                                $size->update(['qty' => $qty]);
                            }
                        }
                    }
                }
            }
            
            DB::commit();
            
            $successMsg = 'Drops created and updated successfully!';
            if ($this->createNewOrder && $newOrderData) {
                $successMsg .= ' New order ' . $newOrderData['order']->customer_po . ' has been created.';
            }
            
            $this->successMessage = $successMsg;
            $this->dispatch('refreshFullSchedule');
            
            // Close modal after a short delay
            $this->dispatch('close-modal');
            
        } catch (\Exception $e) {
            DB::rollBack();
            $this->errorMessage = 'Error creating drops: ' . $e->getMessage();
            
            // Log the error for debugging
            \Log::error('ShipmentScheduleSplitDrop save error: ' . $e->getMessage(), [
                'drop_id' => $this->dropId,
                'create_new_order' => $this->createNewOrder,
                'step' => $this->step,
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    public function ensureNewDropQtysStructure()
    {
        // Ensure the newDropQtys array has the proper structure for all sizes
        for ($i = 0; $i < $this->newDropCount; $i++) {
            if (!isset($this->newDropQtys[$i])) {
                $this->newDropQtys[$i] = [];
            }
            
            foreach ($this->orderQtys as $sizeId => $orderData) {
                if (!isset($this->newDropQtys[$i][$sizeId])) {
                    $this->newDropQtys[$i][$sizeId] = [
                        'size_name' => $orderData['size_name'],
                        'qty' => 0
                    ];
                }
            }
        }
    }

    public function render()
    {
        if (!$this->drop) {
            return view('livewire.production.reports.shipment-schedule-split-drop', [
                'drop' => null,
                'step' => 1,
                'totalSteps' => 5
            ]);
        }

        if ($this->step === 2) {
            $this->initializeNewDrops();
        }

        if ($this->step === 3) {
            $this->prepareOverview();
        }

        // Ensure the newDropQtys structure is always properly initialized
        $this->ensureNewDropQtysStructure();

        return view('livewire.production.reports.shipment-schedule-split-drop', [
            'drop' => $this->drop,
            'step' => $this->step,
            'totalSteps' => $this->totalSteps
        ]);
    }
}
