<?php

namespace App\Http\Livewire\Production\Reports;

use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Shipment;
use App\Models\Suppliers;
use App\Helper\Conversions;
use App\Models\Departments;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithPagination;
use App\Models\CustomerOrders;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Reactive;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Renderless;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use Illuminate\Support\Facades\Validator;
use App\Http\Livewire\FilterableComponent;
use App\Models\CustomerOrderLineQuantities;

class ShipmentSchedule extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideMabli = 0;

    // Properties for filters and display options
    public $hideFilters = false;
    public $borders = false;
    public $view = 'unshipped';
    public $fromDate = null;
    public $toDate = null;
    public $department = [];
    public $customer = [];
    public $season = [];
    public $factory = [];
    public $truck = "";
    public $noTruck = false;
    public $coo = [];
    public $category = [];
    public $search = "";
    public $columns = [
        'Season' => true,
        'Factory' => true,
        'Customer' => true,
        'Order No' => true,
        'Style' => true,
        'Description' => true,
        'Colour' => true,
        'Image' => true,
        'Sizes' => true,
        'Sizes - Ordered' => true,
        'Sizes - Price' => true,
        'Total Qty' => true,
        'Total Shipped' => true,
        'Sealer Sample' => true,
        'Shipment Sample' => true,
        'ExFty' => true,
        'Revised ExFty' => true,
        'Customer exFty' => true,
        'Customer Into w/h' => true,
        'Total CMT' => true,
        'Total Sale' => true,
        'Transport Budget' => true,
        'Incoterms' => true,
        'Truck' => true,
        'Collection' => true,
        'Customs ERN' => true,
        'Shipped' => true,
        'Issue' => true,
        'ASOS 3% Discount Price' => false,
        'RT Invoice' => false,
        'Factory Invoice' => false,
        'Customs Invoice' => false,
        'Transporter Invoice' => false,
        'Finance Complete' => false,
    ];

    // Filter dropdown visibility properties
    public $showSeasonDropdown = false;
    public $showCustomerDropdown = false;
    public $showFactoryDropdown = false;
    public $showCooDropdown = false;
    public $showDeptDropdown = false;
    public $showTruckDropdown = false;
    public $showCatDropdown = false;
    public $showEndedSeasons = false;

    // Validation rules
    #[Validate([
        'hideFilters' => ['nullable', 'boolean'],
        'borders' => ['nullable', 'boolean'],
        'view' => ['nullable', 'in:unshipped,financeNC,noRTInvoice,all,cashflow'],
        'fromDate' => ['nullable', 'date', 'before_or_equal:toDate'],
        'toDate' => ['nullable', 'date', 'after_or_equal:fromDate'],
        'department' => ['nullable', 'array', 'exists:departments,id'],
        'customer' => ['nullable', 'array', 'exists:customers,id'],
        'season' => ['nullable', 'array', 'exists:seasons,id'],
        'factory' => ['nullable', 'array', 'exists:suppliers,id'],
        'coo' => ['nullable', 'array', 'exists:countries,id'],
        'category' => ['nullable', 'array'],
        'search' => ['nullable', 'string', 'max:30'],
        'noTruck' => ['nullable', 'boolean'],
        'selected' => ['array'],
        'selected.*' => ['nullable', 'boolean'],
    ])]

    public array $selected = [];

    #[On('updateSelection')]
    public function updateSelection($shipmentLineId, $checked)
    {
        if ($checked) {
                $this->selected[$shipmentLineId] = true;
        } else {
            $this->selected[$shipmentLineId] = false;
        }
    }

    #[On('clearSelection')]
    public function clearSelection()
    {
        $this->selected = [];
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
        $this->refreshAll();
    }


    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['borders', 'view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'truck', 'noTruck', 'hideMabli'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
            // dd();
            $this->refreshAll();
        }
    }

    protected function filterKey(): string
    {
        return 'shipmentschedulefilters';
    }

    #[On('updateShippedQty')]
    public function updateShippedQty($sizeId, $value)
    {
        \Log::debug('updateShippedQty called', ['sizeId' => $sizeId, 'value' => $value]);
        
        try {
            $size = ShipmentLineSizes::find($sizeId);
            if ($size) {
                $size->shipped_qty = $value;
                $size->save();
                \Log::debug('Size updated successfully', ['sizeId' => $sizeId, 'newValue' => $value]);
                
                // Force a refresh of the component to show updated values
                $this->dispatch('$refresh');
            } else {
                \Log::error('Size not found', ['sizeId' => $sizeId]);
            }
        } catch (\Exception $e) {
            \Log::error('Error updating size', ['sizeId' => $sizeId, 'error' => $e->getMessage()]);
        }
    }

    #[On('refreshData')]
    public function refreshData()
    {
        // Refresh the component to reorder shipments
        $this->refreshAll();
    }

    #[On('close-modal')]
    public function handleCloseModal()
    {
        // Handle modal close event - no action needed
    }

    public function mount()
    {
        $this->loadFilterSettings();

        // Set dynamic date defaults if not loaded from settings
        if ($this->fromDate === null) {
            $this->fromDate = now()->subYear()->format('Y-m-d');
        }
        if ($this->toDate === null) {
            $this->toDate = now()->addYears(5)->format('Y-m-d');
        }
    }

    public $refreshKey = 0;

    #[On('refreshFullSchedule')]
    public function refreshAll()
    {
        $this->render(); // Update chunks
        $this->refreshKey = now()->timestamp; // Generate a new unique key
    }

    /**
     * Render the component.
     */
    public function render()
    {
        Gate::authorize('shipment:read');

        $this->chunks = $this->getData();

        return view('livewire.production.reports.shipment-schedule');
    }

    public array $chunks;
    public int $page = 1;
    public function hasMorePages()
    {
        return $this->page < count($this->chunks);
    }
    public function loadMore()
    {
        if (!$this->hasMorePages()) {
            return;
        }

        $this->page = $this->page + 1;
    }




    // Computed properties for filters
    #[Computed]
    public function suppliers()
    {
        return Suppliers::allCached()->sortBy('name');
    }

    #[Computed]
    public function factories()
    {
        return Suppliers::whereHas('style_versions')
            ->where('type', 'factory')
            ->with('countries')
            ->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::whereHas('styles')
            ->get()
            ->sortBy('name');
    }

    #[Computed]
    public function seasons()
    {
        $allSeasons = Seasons::allCached()->sortByDesc('created_at');
        
        if ($this->showEndedSeasons) {
            return $allSeasons;
        }
        
        // Filter out ended seasons (where end_date is in the past)
        return $allSeasons->filter(function($season) {
            if (!$season->end_date) {
                return true; // Show seasons without an end date
            }
            return $season->end_date >= now();
        });
    }
    
    #[Computed]
    public function endedSeasons()
    {
        $allSeasons = Seasons::allCached()->sortByDesc('created_at');
        
        // Get only ended seasons (where end_date is in the past)
        return $allSeasons->filter(function($season) {
            if (!$season->end_date) {
                return false;
            }
            return $season->end_date < now();
        });
    }

    #[Computed]
    public function departments()
    {
        return Departments::allCached()->sortBy('name');
    }

    #[Computed]
    public function trucks()
    {
        return Shipment::whereNotNull('truck_first_collection')
            ->distinct()
            ->orderBy('id', 'desc')
            ->pluck('id')
            ->values();
    }

    #[Computed]
    public function totalQuoteGBP(){
        $query = $this->buildBaseQuery();
        
        return DB::table($query, 'filtered_shipments')
            ->join('total_cache as tc', function($join) {
                $join->on('tc.entity_id', '=', 'filtered_shipments.id')
                     ->where('tc.entity_type', '=', 'shipment_line')
                     ->where('tc.cache_key', '=', 'prices')
                     ->whereNotNull('tc.fresh_at');
            })
            ->sum(DB::raw("JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, '$.quote_base'))"));
    }
    
    #[Computed]
    public function totalCmtGBP(){
        $query = $this->buildBaseQuery();
        
        return DB::table($query, 'filtered_shipments')
            ->join('total_cache as tc', function($join) {
                $join->on('tc.entity_id', '=', 'filtered_shipments.id')
                     ->where('tc.entity_type', '=', 'shipment_line')
                     ->where('tc.cache_key', '=', 'prices')
                     ->whereNotNull('tc.fresh_at');
            })
            ->sum(DB::raw("COALESCE(JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, '$.cmt_base')), 0)"));
    }
    
    #[Computed]
    public function totalCostGBP(){
        $query = $this->buildBaseQuery();
        
        return DB::table($query, 'filtered_shipments')
            ->join('total_cache as tc', function($join) {
                $join->on('tc.entity_id', '=', 'filtered_shipments.id')
                     ->where('tc.entity_type', '=', 'shipment_line')
                     ->where('tc.cache_key', '=', 'prices')
                     ->whereNotNull('tc.fresh_at');
            })
            ->sum(DB::raw("JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, '$.subtotal_base'))"));
    }
    
    #[Computed]
    public function totalYarnEUR(){
        $query = $this->buildBaseQuery();
        
        return DB::table($query, 'filtered_shipments')
            ->join('total_cache as tc', function($join) {
                $join->on('tc.entity_id', '=', 'filtered_shipments.id')
                     ->where('tc.entity_type', '=', 'shipment_line')
                     ->where('tc.cache_key', '=', 'prices')
                     ->whereNotNull('tc.fresh_at');
            })
            ->sum(DB::raw("COALESCE(JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, '$.yarn_value_euro')), 0)"));
    }
    
    #[Computed]
    public function totalYarnGBP(){
        // Convert EUR to GBP using current conversion rates
        $yarnEUR = $this->totalYarnEUR;
        
        // Get a representative season for conversion rates
        // Using the most recent season as a fallback for conversion
        $season = \App\Models\Seasons::orderBy('id', 'desc')->first();
        
        if ($season && $season->euro_rate > 0) {
            return $yarnEUR / $season->euro_rate;
        }
        
        // Fallback conversion rate if no season found
        return $yarnEUR * 0.85;
    }
    
    #[Computed]
    public function totalMargin(){
        $totalQuote = $this->totalQuoteGBP;
        $totalCost = $this->totalCostGBP;
        
        if($totalQuote > 0) {
            return (($totalQuote - $totalCost) / $totalQuote) * 100;
        }
        
        return 0;
    }

    // Methods for filter selection
    public function toggleEndedSeasons()
    {
        $this->showEndedSeasons = !$this->showEndedSeasons;
    }
    
    public function select($filter, $value = 0)
    {
        // Handle truck selection differently since it's a string, not an array
        if ($filter === 'truck') {
            if ($value == 0) {
                $this->truck = "";
            } else {
                $this->truck = $value;
            }
            $this->updated($filter);
            return;
        }

        if (is_string($this->$filter)) {
            $this->$filter = [];
        }
        if ($value == 0) {
            $this->$filter = [];
        } else {
            if (($this->$filter[$value] ?? 0) == 0) {
                $this->$filter[$value] = 1;
            } else {
                unset($this->$filter[$value]);
            }
        }

        $this->updated($filter);
    }

    /**
     * Clear all filters.
     */
    public function clearFilters()
    {
        $this->reset('view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'truck', 'noTruck', 'hideMabli');
    }

    /**
     * Select all drops.
     */
    public function selectAll()
    {
        $this->selected = [];
        $this->dispatch('select-all');
        foreach ($this->chunks as $chunk) {
            foreach($chunk as $sl){
                $this->selected[$sl['id']] = true;
            }
        }
    }

    /**
     * Clear selected drops.
     */
    public function clearSelected()
    {
        $this->selected = [];
        $this->dispatch('clear-selected');
    }

    /**
     * Edit selected drops.
     */
    public function edit()
    {
        // if(Gate::check('ad'))
        //     dd($this->selected);
        $selectedIds = collect($this->selected)
            ->filter() // keeps only truthy values (i.e. true)
            ->keys()   // gets the keys (IDs)
            ->all();   // convert to plain array
        $this->dispatch('edit-ss', selected: json_encode(array_values($selectedIds)));
    }

    /**
     * Export shipment schedule to CSV/Excel
     */
    public function exportToCSV()
    {
        Gate::authorize('shipment:read');
        
        $query = $this->buildExportQuery();
        $drops = $query->get();
        
        $filename = 'shipment-schedule-' . date('Y-m-d-His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            'Pragma' => 'no-cache',
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Expires' => '0',
        ];
        
        $callback = function() use ($drops) {
            $file = fopen('php://output', 'w');
            
            // Add UTF-8 BOM for Excel compatibility
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // First pass: collect all unique sizes in order
            $allSizeNames = [];
            if($this->columns['Sizes']) {
                foreach($drops as $drop) {
                    $sizes = json_decode($drop->shipment_line_sizes ?? '[]', true) ?: [];
                    foreach($sizes as $size) {
                        $sizeName = $size['name'] ?? '';
                        if ($sizeName && !in_array($sizeName, $allSizeNames)) {
                            $allSizeNames[] = $sizeName;
                        }
                    }
                }
            }
            
            // Build header row based on enabled columns
            $headerRow = ['Drop#'];
            if($this->columns['Season']) $headerRow[] = 'Season';
            if($this->columns['Factory']) $headerRow[] = 'Factory';
            if($this->columns['Customer']) $headerRow[] = 'Customer';
            if($this->columns['Order No']) $headerRow[] = 'Order No';
            if($this->columns['Style']) $headerRow[] = 'Style';
            $headerRow[] = 'Customer Ref';
            if($this->columns['Description']) $headerRow[] = 'Description';
            if($this->columns['Colour']) $headerRow[] = 'Colour';
            
            // Add size columns - ordered first, then shipped
            if($this->columns['Sizes']) {
                foreach($allSizeNames as $sizeName) {
                    $headerRow[] = $sizeName . ' (Ordered)';
                }
                foreach($allSizeNames as $sizeName) {
                    $headerRow[] = $sizeName . ' (Shipped)';
                }
            }
            
            if($this->columns['Total Qty']) $headerRow[] = 'Total Qty';
            if($this->columns['Total Shipped']) $headerRow[] = 'Total Shipped';
            if($this->columns['Sealer Sample']) $headerRow[] = 'Sealer Sample';
            if($this->columns['Shipment Sample']) $headerRow[] = 'Shipment Sample';
            if($this->columns['ExFty']) $headerRow[] = 'First ExFty';
            if($this->columns['Revised ExFty']) $headerRow[] = 'Revised ExFty';
            if($this->columns['Customer exFty']) $headerRow[] = 'Customer exFty';
            if($this->columns['Customer Into w/h']) $headerRow[] = 'Customer Into w/h';
            if($this->columns['Total CMT']) $headerRow[] = 'Total Cost';
            if($this->columns['Total Sale']) $headerRow[] = 'Total Sale';
            if($this->columns['Transport Budget']) $headerRow[] = 'Transport Budget';
            if($this->columns['Incoterms']) $headerRow[] = 'Incoterms';
            if($this->columns['Truck']) $headerRow[] = 'Truck';
            $headerRow[] = 'Truck Organiser';
            $headerRow[] = 'Truck Collection Date';
            if($this->columns['Collection']) $headerRow[] = 'Collection Date';
            if($this->columns['Customs ERN']) $headerRow[] = 'Customs ERN';
            if($this->columns['Shipped']) $headerRow[] = 'Shipped';
            if($this->columns['RT Invoice']) $headerRow[] = 'RT Invoice';
            $headerRow[] = 'Notes';
            
            fputcsv($file, $headerRow);
            
            // Add data rows
            foreach($drops as $drop) {
                $row = [$drop->id];
                
                if($this->columns['Season']) $row[] = $drop->season_description ?? '';
                if($this->columns['Factory']) $row[] = $drop->factory_name ?? '';
                if($this->columns['Customer']) $row[] = $drop->customer_name ?? '';
                if($this->columns['Order No']) $row[] = $drop->customer_po ?? '';
                if($this->columns['Style']) $row[] = ($drop->designs_id ?? '') . ' - ' . ($drop->version_name ?? '');
                $row[] = $drop->customer_ref ?? '';
                if($this->columns['Description']) $row[] = $drop->design_description ?? '';
                if($this->columns['Colour']) $row[] = $drop->colourway_name ?? '';
                
                if($this->columns['Sizes']) {
                    $sizes = json_decode($drop->shipment_line_sizes ?? '[]', true) ?: [];
                    
                    // Create a map of size name to quantities
                    $sizeMap = [];
                    foreach($sizes as $size) {
                        $sizeMap[$size['name'] ?? ''] = [
                            'qty' => $size['qty'] ?? 0,
                            'shipped_qty' => $size['shipped_qty'] ?? 0
                        ];
                    }
                    
                    // Add ordered quantities for each size
                    foreach($allSizeNames as $sizeName) {
                        $row[] = $sizeMap[$sizeName]['qty'] ?? 0;
                    }
                    
                    // Add shipped quantities for each size
                    foreach($allSizeNames as $sizeName) {
                        $row[] = $sizeMap[$sizeName]['shipped_qty'] ?? 0;
                    }
                }
                
                if($this->columns['Total Qty']) $row[] = $drop->total_pieces ?? 0;
                if($this->columns['Total Shipped']) $row[] = $drop->total_pieces_shipped ?? 0;
                if($this->columns['Sealer Sample']) $row[] = $drop->sealer_sample_status ?? '';
                if($this->columns['Shipment Sample']) $row[] = $drop->shipment_sample_status ?? '';
                if($this->columns['ExFty']) $row[] = $drop->first_exfty ? date('Y-m-d', strtotime($drop->first_exfty)) : '';
                if($this->columns['Revised ExFty']) $row[] = $drop->exfty ? date('Y-m-d', strtotime($drop->exfty)) : '';
                if($this->columns['Customer exFty']) $row[] = $drop->factory_cust_date ? date('Y-m-d', strtotime($drop->factory_cust_date)) : '';
                if($this->columns['Customer Into w/h']) $row[] = $drop->wh_cust_date ? date('Y-m-d', strtotime($drop->wh_cust_date)) : '';
                if($this->columns['Total CMT']) $row[] = $drop->total_cost ?? 0;
                if($this->columns['Total Sale']) $row[] = $drop->total_sale ?? 0;
                if($this->columns['Transport Budget']) $row[] = $drop->transport_budget ?? 0;
                if($this->columns['Incoterms']) $row[] = $drop->incoterms ?? '';
                if($this->columns['Truck']) $row[] = $drop->shipment_id ?? '';
                $row[] = $drop->truck_organiser ?? '';
                $row[] = $drop->truck_first_collection ? date('Y-m-d', strtotime($drop->truck_first_collection)) : '';
                if($this->columns['Collection']) $row[] = $drop->collection_date ? date('Y-m-d', strtotime($drop->collection_date)) : '';
                if($this->columns['Customs ERN']) $row[] = $drop->customs_submitted_text ?? '';
                if($this->columns['Shipped']) $row[] = $drop->complete ? 'Yes' : 'No';
                if($this->columns['RT Invoice']) $row[] = $drop->rt_invoice ?? '';
                $row[] = $drop->notes ?? '';
                
                fputcsv($file, $row);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    private function buildExportQuery()
    {
        $query = ShipmentLine::query();
        
        $query = $this->applyFilters($query)
            ->select([
                'shipment_lines.id',
                'shipment_lines.exfty',
                'shipment_lines.shipment_id',
                'shipment_lines.notes',
                'shipment_lines.complete',
                'shipment_lines.rt_invoice',
                'shipment_lines.collection_date',
                'customer_orders.customer_po',
                'customer_orders.incoterms',
                'customers.name as customer_name',
                'seasons.description as season_description',
                'factory.name as factory_name',
                'styles.designs_id',
                'styles.customer_ref',
                'colourways.name as colourway_name',
                'style_versions.name as version_name',
                'designs.description as design_description',
                'customer_order_lines.factory_cust_date',
                'customer_order_lines.wh_cust_date',
                'shipments.customs_submitted_text',
                'shipments.organiser as truck_organiser',
                'shipments.truck_first_collection',
            ])
            ->selectRaw('
                (SELECT SUM(sls.qty) 
                 FROM shipment_line_sizes sls 
                 WHERE sls.shipment_line_id = shipment_lines.id) as total_pieces
            ')
            ->selectRaw('
                (SELECT SUM(sls.shipped_qty) 
                 FROM shipment_line_sizes sls 
                 WHERE sls.shipment_line_id = shipment_lines.id) as total_pieces_shipped
            ')
            ->selectRaw("
                COALESCE(
                    (SELECT JSON_UNQUOTE(JSON_EXTRACT(audit.new_values, '$.exfty'))
                     FROM audits AS audit
                     WHERE audit.auditable_id = shipment_lines.id
                     AND audit.auditable_type = 'App\\\Models\\\ShipmentLine'
                     AND JSON_EXTRACT(audit.new_values, '$.exfty') IS NOT NULL
                     ORDER BY audit.created_at ASC
                     LIMIT 1),
                    shipment_lines.exfty
                ) AS first_exfty
            ")
            ->selectRaw("
                (SELECT status 
                 FROM samples 
                 WHERE samples.colourways_id = colourways.id 
                 AND samples.sample_types_id = 3 
                 AND samples.deleted_at IS NULL
                 ORDER BY samples.created_at DESC
                 LIMIT 1) as sealer_sample_status
            ")
            ->selectRaw("
                (SELECT status 
                 FROM samples 
                 WHERE samples.colourways_id = colourways.id 
                 AND samples.sample_types_id = 7 
                 AND samples.deleted_at IS NULL
                 ORDER BY samples.created_at DESC
                 LIMIT 1) as shipment_sample_status
            ")
            ->selectRaw('
                (SELECT JSON_ARRAYAGG(
                    JSON_OBJECT(
                        "name", sizes.name,
                        "qty", sls.qty,
                        "shipped_qty", sls.shipped_qty
                    )
                )
                FROM shipment_line_sizes sls
                JOIN sizes ON sizes.id = sls.sizes_id
                WHERE sls.shipment_line_id = shipment_lines.id
                ORDER BY sizes.order) as shipment_line_sizes
            ')
            ->join('customer_order_lines', function ($join) {
                $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id')
                    ->where('customer_order_lines.cancelled', 0);
            })
            ->join('customer_orders', function($join) {
                $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
                $join->on('customer_orders.order_type', '=', DB::raw("'wholesale'"));
                $join->on('customer_orders.cancelled', '=', DB::raw(0));
            })
            ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
            ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
            ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->join('designs', 'designs.id', '=', 'styles.designs_id')
            ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
            ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
            ->leftJoin('shipments', 'shipments.id', '=', 'shipment_lines.shipment_id')
            ->leftJoin('suppliers as transporters', 'transporters.id', '=', 'shipments.transporter_id')
            ->orderByRaw("
                -(shipment_lines.complete),
                -(shipments.truck_first_collection) desc,
                shipment_lines.shipment_id,
                -(COALESCE(shipments.truck_first_collection, shipment_lines.exfty)) desc,
                factory.countries_id,
                -(factory.name),
                -(customers.name),
                -(customer_orders.customer_po),
                -(styles.id),
                -(colourways.name),
                shipment_lines.id
            ");
        
        return $query;
    }


    /**
     * Get the filter key string.
     */
    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    /**
     * Get the columns version.
     */
    protected function columnsVersion(): float
    {
        return 5;
    }

    /**
     * Get the filters.
     */
    protected function filters(): array
    {
        return ['borders', 'view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'truck', 'noTruck', 'columns', 'hideMabli'];
    }




    /**
     * Update the number of cartons for a drop.
     */
    public function updateFlagIssue($dropId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'required|boolean'] // Assuming flag is a binary field (0 or 1)
        )->validate();

        if (Gate::check('order:update')) {
            $drop = ShipmentLine::find($dropId);
            if ($drop) {
                $drop->flag_issue = empty($validatedData['value']) ? 0 : $validatedData['value'];
                $drop->save();
            }
        }
    }














    private function getData()
    {
        $query = ShipmentLine::query();

        $query = $this->applyFilters($query)
            ->select([
                'shipment_lines.id',
            ])
            ->join('customer_order_lines', function ($join) {
                $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id')
                    ->where('customer_order_lines.cancelled', 0);
            })
            ->Join('customer_orders', function($join)
                {
                    $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
                    $join->on('customer_orders.order_type', '=', DB::raw("'wholesale'"));
                    $join->on('customer_orders.cancelled', '=', DB::raw(0));
                })
            ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
            ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
            ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
            ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
            ->leftJoin('shipments', 'shipments.id', '=', 'shipment_lines.shipment_id')
            ->leftJoin('suppliers as transporters', 'transporters.id', '=', 'shipments.transporter_id')


            // ✅ Mark first row using ROW_NUMBER()
            ->selectRaw("
                CASE
                    WHEN ROW_NUMBER() OVER (
                        PARTITION BY shipment_lines.shipment_id
                        ORDER BY
                            -(shipment_lines.complete),
                            -(shipments.truck_first_collection) desc,
                            shipment_lines.shipment_id,
                            -(COALESCE(shipments.truck_first_collection, shipment_lines.exfty)) desc,
                            factory.countries_id,
                            -(factory.name),
                            -(customers.name),
                            -(customer_orders.customer_po),
                            -(styles.id),
                            -(colourways.name),
                            shipment_lines.id
                    ) = 1
                    THEN 1 ELSE 0
                END AS is_first_in_shipment
            ")

            // ✅ Mark last row using COUNT(*) OVER()
            ->selectRaw("
                CASE
                    WHEN ROW_NUMBER() OVER (
                        PARTITION BY shipment_lines.shipment_id
                        ORDER BY
                            -(shipment_lines.complete),
                            -(shipments.truck_first_collection) desc,
                            shipment_lines.shipment_id,
                            -(COALESCE(shipments.truck_first_collection, shipment_lines.exfty)) desc,
                            factory.countries_id,
                            -(factory.name),
                            -(customers.name),
                            -(customer_orders.customer_po),
                            -(styles.id),
                            -(colourways.name),
                            shipment_lines.id
                    ) = COUNT(*) OVER (
                        PARTITION BY shipment_lines.shipment_id
                    )
                    THEN 1 ELSE 0
                END AS is_last_in_shipment
            ")

        ->orderByRaw("
            -(shipment_lines.complete),
            -(shipments.truck_first_collection) desc,
            shipment_lines.shipment_id,
            -(COALESCE(shipments.truck_first_collection, shipment_lines.exfty)) desc,
            factory.countries_id,
            -(factory.name),
            -(customers.name),
            -(customer_orders.customer_po),
            -(styles.id),
            -(colourways.name),
            shipment_lines.id
        ");


        // ✅ Fetch all columns and chunk the results
        $data = $query->get([
            'shipment_lines.id', 'shipment_id', 'is_first_in_shipment', 'is_last_in_shipment'
        ])->chunk(20);

        return $data->toArray();
    }







    /**
     * Build the base query with all filters applied for totals calculations.
     * Returns a query builder instance that can be used as a subquery.
     */
    private function buildBaseQuery()
    {
        $query = ShipmentLine::query()
            ->select(['shipment_lines.id'])
            ->join('customer_order_lines', function ($join) {
                $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id')
                    ->where('customer_order_lines.cancelled', 0);
            })
            ->join('customer_orders', function($join) {
                $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id')
                     ->where('customer_orders.order_type', '=', 'wholesale')
                     ->where('customer_orders.cancelled', '=', 0);
            })
            ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
            ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
            ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
            ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
            ->leftJoin('shipments', 'shipments.id', '=', 'shipment_lines.shipment_id');

        return $this->applyFilters($query);
    }

    private function applyFilters($query){

        $query->where('customer_orders.cancelled', 0)
        ->where('customer_order_lines.cancelled', 0);


        //VIEW FILTER
        $query->where('customer_orders.order_type', 'wholesale');

        $query->where('customer_order_lines.cancelled', FALSE);

        if($this->hideMabli){
            $query->where('customer_orders.customers_id', '!=', 9);
        }


        if($this->view == "unshipped"){
            $query->where('shipment_lines.complete', FALSE);
        }
        elseif($this->view == "financeNC"){
            $query->where('shipment_lines.invoiced', FALSE);
        }
        elseif($this->view == "cashflow"){
            $query->whereNull('shipment_lines.rt_invoice');
        }
        elseif($this->view == "noRTInvoice"){
            $query->where(function($q) {
                $q->whereNull('shipment_lines.rt_invoice')
                ->orWhere('shipment_lines.rt_invoice', '');
            });
            // $query->where('shipment_lines.complete', TRUE); //DO NOT ADD BACK IN - VALERIE
        }
        else{
            $query->where('shipment_lines.exfty', '>=', $this->fromDate);
        }

        //DATE FILTER
        if (!empty($this->toDate)) {
            $query->where('exfty', '<=', $this->toDate);
        }

        //SEASONS FILTER
        if(!empty($this->season)) {
            $query->whereHas('customer_order_lines.customer_orders', function($q){
                $q->whereIn('seasons_id', array_keys($this->season));
            });
        }

        //CUSTOMERS FILTER
        if (!empty($this->customer)) {
            $query->whereIn('customer_orders.customers_id', array_keys($this->customer));
        }

        //FACTORIES FILTER
        if(!empty($this->factory)) {
            $query->whereHas('customer_order_lines.colourways.style_versions', function($q){
                $q->whereIn('factory_id', array_keys($this->factory));
            });
        }

        //COO FILTER
        if(!empty($this->coo)) {
            $query->whereHas('customer_order_lines.colourways.style_versions.factories', function($q){
                $q->whereIn('countries_id', array_keys($this->coo));
            });
        }

        //DEPARTMENT FILTER
        if(!empty($this->department)) {
            $query->whereHas('customer_order_lines.customer_orders', function($q){
                $q->whereIn('departments_id', array_keys($this->department));
            });
        }

        //CATEGORY FILTER
        if(!empty($this->category)) {
            $query->whereHas('customer_order_lines.colourways.style_versions.styles', function($q){
                $q->whereIn('category', array_keys($this->category));
            });
        }

        //TRUCK FILTER
        if ($this->noTruck) {
            $query->whereNull('shipment_lines.shipment_id');
        } elseif (!empty($this->truck)) {
            $query->where('shipments.id', $this->truck);
        }

        //SEARCH FILTER
        if (! empty($this->search)) {
            $query->search($this->search);
        }
        return $query;
    }

}
