<?php

namespace App\Http\Livewire\Production\Reports;

use Cache;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Shipment;
use App\Helper\Functions;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Helper\Conversions;
use App\Models\Departments;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithPagination;
use App\Models\StyleVersions;
use Illuminate\Support\Number;
use Livewire\Attributes\Computed;

use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use App\Models\ViewShipmentSchedule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Intervention\Image\Facades\Image;

use App\Models\ShipmentLineExftyDates;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\ServiceProvider;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;
use App\Models\CustomerOrderLineCustDates;


class SalesValuesExfty extends FilterableComponent
{
	use WithPagination;
    use Filterable;

    #[Validate([
        'view' => ['required', 'in:unshipped,financeNC,all,noRTInvoice'],
        'fromDate' => ['required', 'date_format:Y-m-d', 'before_or_equal:toDate'],
        'toDate' => ['required', 'date_format:Y-m-d', 'after_or_equal:fromDate'],
    ])]
	public $view = 'unshipped', $fromDate = null, $toDate = null;

    public $hideFilters = false;

    public function mount()
    {
        $this->loadFilterSettings();

        // Set dynamic date defaults if not loaded from settings
        if ($this->fromDate === null) {
            $this->fromDate = now()->subYear()->format('Y-m-d');
        }
        if ($this->toDate === null) {
            $this->toDate = now()->addYears(5)->format('Y-m-d');
        }
    }

    protected function filterKey(): string
    {
        return 'orderslistfilters';
    }

    protected function columnsVersion(): float
    {
        return 1;
    }

    protected function filters(): array
    {
        return ['view', 'fromDate', 'toDate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['view', 'fromDate', 'toDate'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }


	public function render()
	{
		Gate::authorize('shipment:read');

		$drops = $this->efficientGetShipment;

        $pivot = [
            "dataSource" => [
                "dataSourceType" => "json",
                'data' => $drops->toArray(),
            ],
            "slice" => [
                "rows" => [
                    [
                        "uniqueName" => "revised_exfty.Month"
                    ],
                    [
                        "uniqueName" => "coo_name"
                    ],
                    [
                        "uniqueName" => "department"
                    ],
                    [
                        "uniqueName" => "factory"
                    ],
                    [
                        "uniqueName" => "customer"
                    ],
                    [
                        "uniqueName" => "category"
                    ],
                    [
                        "uniqueName" => "customer_ref"
                    ]
                ],
                "columns" => [
                    [
                        "uniqueName" => "revised_exfty.Year"
                    ],
                    [
                        "uniqueName" => "Measures"
                    ]
                ],
                "measures" => [
                    [
                        "uniqueName" => "CMT",
                        "aggregation" => "sum",
                        "format" => "5xyib2vm"
                    ],
                    [
                        "uniqueName" => "Cost",
                        "aggregation" => "sum",
                        "format" => "5xyiaxxn"
                    ],
                    [
                        "uniqueName" => "Quote",
                        "aggregation" => "sum",
                        "format" => "5xyiaxxn"
                    ],
                    [
                        "uniqueName" => "Margin",
                        "aggregation" => "average",
                        "format" => "5yxxehbo"
                    ],
                    [
                        "uniqueName" => "qty",
                        "aggregation" => "sum",
                        "format" => "5y64ej8q",
                        "caption" => "Qty"
                    ],
                    [
                        "uniqueName" => "customer_ref",
                        "aggregation" => "distinctcount",
                        "caption" => "Styles Count"
                    ]
                ]
            ],
            "options" => [
                "showAggregationLabels" => false
            ],
            "formats" => [
                [
                    "name" => "5xyiaxxn",
                    "thousandsSeparator" => ",",
                    "decimalSeparator" => ".",
                    "decimalPlaces" => 0,
                    "currencySymbol" => "£",
                    "currencySymbolAlign" => "left",
                    "nullValue" => "",
                    "textAlign" => "right",
                    "isPercent" => false
                ],
                [
                    "name" => "5xyib2vm",
                    "thousandsSeparator" => ",",
                    "decimalSeparator" => ".",
                    "decimalPlaces" => 0,
                    "currencySymbol" => "£",
                    "currencySymbolAlign" => "left",
                    "nullValue" => "",
                    "textAlign" => "right",
                    "isPercent" => false
                ],
                [
                    "name" => "5y64ej8q",
                    "thousandsSeparator" => ",",
                    "decimalSeparator" => ".",
                    "currencySymbol" => "",
                    "currencySymbolAlign" => "left",
                    "nullValue" => "",
                    "textAlign" => "right",
                    "isPercent" => false
                ],
                [
                    "name" => "5yxxehbo",
                    "thousandsSeparator" => " ",
                    "decimalSeparator" => ".",
                    "decimalPlaces" => 1,
                    "currencySymbol" => "%",
                    "currencySymbolAlign" => "left",
                    "nullValue" => "",
                    "textAlign" => "right",
                    "isPercent" => true
                ]
            ]
        ];

        return view('livewire.production.reports.sales-values-exfty', [
			'drops' => $drops,
			'pivot' => $pivot
        ]);
	}

    public function go(){
        return redirect(request()->header('Referer'));
    }

    #[Computed]
    private function efficientGetShipment(){


        $query = ShipmentLine::select(
            "shipment_lines.id as id",
            "shipment_lines.customer_order_lines_id",
            "seasons.description AS season",
            "factory.name AS factory",
            "departments.description AS department",
            "styles.category AS category",
            "customers.name AS customer",
            "customer_orders.customer_po AS customer_po",

            "shipment_lines.exfty AS revised_exfty",
            "countries.country AS coo_name",
            "styles.customer_ref",
        )
        ->selectRaw("SUM(COALESCE(sls.shipped_qty, sls.qty)) AS qty")
        // ->selectRaw("
        //     (
        //         select
        //             JSON_OBJECT(
        //                 'quote_total', CONVERT_CURRENCY(sum(quote_cache * COALESCE(NULLIF(sls.shipped_qty, 0), sls.qty)), customers.currency, '£', customer_orders.seasons_id),
        //                 'cmt_total', CONVERT_CURRENCY(sum(cmt_cache * COALESCE(NULLIF(sls.shipped_qty, 0), sls.qty)), factory.currency, '£', customer_orders.seasons_id),
        //                 'cost_total', sum(cost_cache * COALESCE(NULLIF(sls.shipped_qty, 0), sls.qty))
        //             ) as totals
        //         from  shipment_line_sizes sls
        //         join sizes on sizes.id = sls.sizes_id and sls.shipment_line_id = shipment_lines.id
        //         join customer_order_line_quantities colq on colq.customer_order_lines_id = customer_order_lines.id and colq.sizes_id = sls.sizes_id
        //         where sls.deleted_at is null
        //         group by sls.shipment_line_id
        //     ) AS totals
        // ")
        ->selectRaw("
            (SELECT JSON_ARRAYAGG(
                        JSON_OBJECT(
                            'size_id',     sls.sizes_id,
                            'size_name',     s.name,
                            'sls_id', sls.id,
                            'ordered_qty', sls.qty,
                            'shipped_qty', sls.shipped_qty
                        )
                    )
                FROM shipment_line_sizes AS sls
                JOIN sizes s on s.id = sls.sizes_id
                WHERE sls.shipment_line_id = shipment_lines.id
            ) AS size_breakdown")
        ->join('customer_order_lines', function ($join) {
            $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id')
                ->where('customer_order_lines.cancelled', 0);
        })
        ->Join('customer_orders', function($join)
        {
            $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
            $join->on('customer_orders.order_type', '=', DB::raw("'wholesale'"));
            $join->on('customer_orders.cancelled', '=', DB::raw(0));
        })
        ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
        ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
        ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->join('departments', 'departments.id', '=', 'styles.departments_id')
        ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
        ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
        ->join('countries', 'countries.id', '=', 'factory.countries_id')
        ->join('designs', 'designs.id', '=', 'styles.designs_id')
        ->leftJoin('shipments', 'shipments.id', '=', 'shipment_lines.shipment_id')
        ->leftJoin('suppliers as transporters', 'transporters.id', '=', 'shipments.transporter_id')
        ->leftJoin('customer_order_line_quantities as colq', 'colq.customer_order_lines_id', '=', 'customer_order_lines.id')
        ->leftJoin('shipment_line_sizes as sls', function($join)
        {
            $join->on('sls.shipment_line_id', '=', 'shipment_lines.id');
            $join->on('sls.sizes_id', '=', 'colq.sizes_id');
        })
        ->groupBy(
            'shipment_lines.id',
            'seasons.description',
            'factory.name',
            'departments.description',
            'styles.category',
            'customers.name',
            'customer_orders.customer_po',
            'shipment_lines.exfty',
            'countries.country',
            'styles.customer_ref',
            'customers.currency',
            'customer_orders.seasons_id',
            'factory.currency',
            'customer_order_lines.id'
        )
        ->whereNull('shipment_lines.deleted_at');

        if($this->view == "unshipped"){
			$query->where('shipment_lines.complete', FALSE);
		}
		elseif($this->view == "financeNC"){
			$query->where('shipment_lines.invoiced', FALSE);
		}
        elseif($this->view == "noRTInvoice"){
            $query->where(function($q) {
                $q->whereNull('shipment_lines.rt_invoice')
                  ->orWhere('shipment_lines.rt_invoice', '');
            });
            // $query->where('shipment_lines.complete', TRUE);
        }
		// else{
        // }
        $query->where('shipment_lines.exfty', '>=', $this->fromDate);

        if(!empty($this->toDate)){
			$query->where('shipment_lines.exfty', '<=', $this->toDate);
		};

        $query->with('customer_order_lines.colourways');

        $results = $query->get();

        // dd($results);

		foreach($results as $l=>$line){
            $sizes = json_decode($line->size_breakdown);
            $totalCost = 0;
            $totalSale = 0;
            $totalCMT = 0;
            foreach($sizes as $size){
                $price = $line->customer_order_lines->colourways->price($size->size_id)->first();
                $totalCMT = $totalCMT + (($size->shipped_qty > 0 ? $size->shipped_qty : $size->ordered_qty) * $price?->cmt_base);
                $totalCost = $totalCost + (($size->shipped_qty > 0 ? $size->shipped_qty : $size->ordered_qty) * $price?->subtotal_base);
                $totalSale = $totalSale + (($size->shipped_qty > 0 ? $size->shipped_qty : $size->ordered_qty) * $price?->quote_base);
            }
            $results[$l]->size_breakdown = null;

            $results[$l]->Quote = $totalSale;
            $results[$l]->CMT = $totalCMT;
            $results[$l]->Cost = $totalCost;


            unset($line->size_breakdown);
            unset($line->customer_order_lines);
		}

		return $results;
	}
}
