<?php

namespace App\Http\Livewire\Production\Reports;

use Cache;
use Carbon\Carbon;
use App\Models\Customer;
use Livewire\Component;
use App\Models\ShipmentLine;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Livewire\Attributes\Validate;
use App\Http\Livewire\FilterableComponent;

class SalesOutlook extends FilterableComponent
{
    #[Validate([
        'includeAll' => ['boolean'],
        'showBaseCurrency' => ['boolean'],
        'fromDate' => ['required', 'date_format:Y-m-d', 'before_or_equal:toDate'],
        'toDate' => ['required', 'date_format:Y-m-d', 'after_or_equal:fromDate'],
        'metric' => ['required', 'in:sales_value,total_qty'],
    ])]
    public $includeAll = false;
    public $showBaseCurrency = false;
    public $fromDate;
    public $toDate;
    public $metric = 'sales_value';
    public $hideFilters = false;
    
    public $metrics = [
        'sales_value' => 'Sales Value (Base Currency)',
        'total_qty' => 'Total Quantity (Garments)'
    ];

    protected function filterKey(): string
    {
        return 'salesoutlookfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.0;
    }

    protected function filters(): array
    {
        return ['includeAll', 'showBaseCurrency', 'fromDate', 'toDate', 'metric'];
    }

    protected function getDefaultColumns(): array
    {
        return []; // No columns for this report, only filters
    }

    public function mount()
    {
        // Load saved filter settings from parent trait
        parent::mount();
        
        // Set defaults if not already set
        if (!$this->fromDate) {
            $this->fromDate = now()->startOfYear()->format('Y-m-d');
        }
        if (!$this->toDate) {
            $this->toDate = now()->endOfYear()->format('Y-m-d');
        }
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Validate date range is within 2 years
            if (in_array($propertyName, ['fromDate', 'toDate']) && $this->fromDate && $this->toDate) {
                $from = Carbon::parse($this->fromDate);
                $to = Carbon::parse($this->toDate);
                
                if ($to->diffInYears($from) > 2) {
                    $this->addError($propertyName, 'Date range cannot exceed 2 years.');
                    $this->reset($propertyName);
                    return;
                }
            }
            
            // Save filter settings if validation passed
            if (in_array($propertyName, $this->filters())) {
                $this->persistFilters();
            }
        }
    }

    public function go()
    {
        $this->validate();
        $this->persistFilters();
    }
    
    public function exportToCSV()
    {
        Gate::authorize('shipment:read');
        
        $data = $this->getData();
        $months = $this->getMonthsWithData($data);
        
        $filename = 'sales-outlook-' . date('Y-m-d-His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            'Pragma' => 'no-cache',
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Expires' => '0',
        ];
        
        $callback = function() use ($data, $months) {
            $file = fopen('php://output', 'w');
            
            // Add UTF-8 BOM for Excel compatibility
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // Build header row
            $headerRow = ['Customer', 'Currency'];
            foreach ($months as $month) {
                $headerRow[] = $month['label'];
            }
            $headerRow[] = 'Total';
            fputcsv($file, $headerRow);
            
            // Add data rows
            foreach ($data as $row) {
                $dataRow = [
                    $row['customer_name'],
                    $this->showBaseCurrency ? 'GBP' : $row['currency']
                ];
                
                foreach ($months as $month) {
                    // Display value (customer or base depending on toggle)
                    $displayValue = $this->showBaseCurrency 
                        ? ($row['months_base'][$month['key']] ?? 0)
                        : ($row['months'][$month['key']] ?? 0);
                    
                    if ($this->metric === 'sales_value') {
                        $dataRow[] = $displayValue > 0 ? number_format($displayValue, 2, '.', '') : '';
                    } else {
                        $dataRow[] = $displayValue > 0 ? $displayValue : '';
                    }
                }
                
                // Display total (customer or base depending on toggle)
                $displayTotal = $this->showBaseCurrency ? $row['total_base'] : $row['total'];
                
                if ($this->metric === 'sales_value') {
                    $dataRow[] = number_format($displayTotal, 2, '.', '');
                } else {
                    $dataRow[] = $displayTotal;
                }
                
                fputcsv($file, $dataRow);
            }
            
            // Add grand totals row (always in base currency)
            $totalsRow = ['GRAND TOTAL', 'GBP'];
            $grandTotal = 0;
            
            foreach ($months as $month) {
                $value = 0;
                foreach ($data as $customerData) {
                    $value += $customerData['months_base'][$month['key']] ?? 0;
                }
                
                if ($this->metric === 'sales_value') {
                    $totalsRow[] = number_format($value, 2, '.', '');
                } else {
                    $totalsRow[] = $value;
                }
                
                $grandTotal += $value;
            }
            
            if ($this->metric === 'sales_value') {
                $totalsRow[] = number_format($grandTotal, 2, '.', '');
            } else {
                $totalsRow[] = $grandTotal;
            }
            
            fputcsv($file, $totalsRow);
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    private function persistFilters()
    {
        $filterData = [];
        foreach ($this->filters() as $property) {
            if (property_exists($this, $property)) {
                $filterData[$property] = $this->$property;
            }
        }
        
        auth()->user()->settings([$this->filterKey() => $filterData]);
    }

    public function render()
    {
        Gate::authorize('shipment:read');
        
        $data = [];
        $months = [];
        
        if ($this->fromDate && $this->toDate) {
            $data = $this->getData();
            $months = $this->getMonthsWithData($data);
        }

        return view('livewire.production.reports.sales-outlook', [
            'data' => $data,
            'months' => $months,
        ]);
    }

    private function getMonthsWithData($data)
    {
        // First, get all possible months in the range
        $allMonths = [];
        $start = Carbon::parse($this->fromDate)->startOfMonth();
        $end = Carbon::parse($this->toDate)->endOfMonth();
        
        while ($start->lte($end)) {
            $allMonths[] = [
                'key' => $start->format('Y-m'),
                'label' => $start->format('M Y')
            ];
            $start->addMonth();
        }
        
        // Find months that actually have data (using base values for consistency)
        $monthsWithData = [];
        foreach ($allMonths as $month) {
            $hasData = false;
            foreach ($data as $customerData) {
                if (isset($customerData['months_base'][$month['key']]) && $customerData['months_base'][$month['key']] > 0) {
                    $hasData = true;
                    break;
                }
            }
            if ($hasData) {
                $monthsWithData[] = $month;
            }
        }
        
        return $monthsWithData;
    }

    private function getData()
    {
        $query = ShipmentLine::query()
            ->select([
                'customers.id as customer_id',
                'customers.name as customer_name',
                'customers.currency as customer_currency',
                DB::raw('DATE_FORMAT(shipment_lines.exfty, "%Y-%m") as month'),
                DB::raw('SUM(COALESCE(
                    CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.quote")) AS DECIMAL(18,2)), 
                    0
                )) as sales_value_customer'),
                DB::raw('SUM(COALESCE(
                    CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.quote_base")) AS DECIMAL(18,2)), 
                    0
                )) as sales_value_base'),
                DB::raw('SUM(COALESCE((
                    SELECT SUM(colq.qty) 
                    FROM customer_order_line_quantities colq 
                    WHERE colq.customer_order_lines_id = customer_order_lines.id
                    AND colq.deleted_at IS NULL
                ), 0)) as total_qty'),
                DB::raw('SUM(COALESCE((
                    SELECT SUM(sls.qty) 
                    FROM shipment_line_sizes sls 
                    WHERE sls.shipment_line_id = shipment_lines.id
                    AND sls.deleted_at IS NULL
                ), 0)) as total_pieces')
            ])
            ->join('customer_order_lines', function ($join) {
                $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id')
                    ->where('customer_order_lines.cancelled', 0);
            })
            ->join('customer_orders', function($join) {
                $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
                $join->on('customer_orders.order_type', '=', DB::raw("'wholesale'"));
                $join->on('customer_orders.cancelled', '=', DB::raw(0));
            })
            ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
            ->leftJoin('total_cache', function($join) {
                $join->on('total_cache.entity_id', '=', 'shipment_lines.id')
                    ->where('total_cache.entity_type', '=', 'shipment_line')
                    ->where('total_cache.cache_key', '=', 'prices')
                    ->whereNotNull('total_cache.fresh_at');
            })
            ->whereNull('shipment_lines.deleted_at')
            ->whereNull('customer_order_lines.deleted_at')
            ->whereNull('customer_orders.deleted_at')
            ->whereNull('customers.deleted_at')
            ->whereBetween('shipment_lines.exfty', [$this->fromDate, $this->toDate]);
        
        // Filter for unshipped or all
        if (!$this->includeAll) {
            $query->where('shipment_lines.complete', false);
        }
        
        $query->groupBy('customers.id', 'customers.name', 'customers.currency', 'month')
            ->orderBy('customers.name')
            ->orderBy('month');
        
        $results = $query->get();
        
        // Transform data into customer => month => value structure
        $data = [];
        foreach ($results as $row) {
            $customerId = $row->customer_id;
            $customerName = $row->customer_name;
            $currency = $row->customer_currency;
            $month = $row->month;
            
            if (!isset($data[$customerId])) {
                $data[$customerId] = [
                    'customer_id' => $customerId,
                    'customer_name' => $customerName,
                    'currency' => $currency,
                    'months' => [],
                    'months_base' => [],
                    'total' => 0,
                    'total_base' => 0
                ];
            }
            
            // Get the metric values (customer currency and base)
            $value = $this->getMetricValue($row, false);
            $valueBase = $this->getMetricValue($row, true);
            
            $data[$customerId]['months'][$month] = $value;
            $data[$customerId]['months_base'][$month] = $valueBase;
            $data[$customerId]['total'] += $value;
            $data[$customerId]['total_base'] += $valueBase;
        }
        
        return array_values($data);
    }
    
    private function getMetricValue($row, $useBase = false)
    {
        switch ($this->metric) {
            case 'sales_value':
                if ($useBase) {
                    return round($row->sales_value_base ?? 0, 2);
                }
                return round($row->sales_value_customer ?? 0, 2);
            case 'total_qty':
                return $row->total_qty ?? 0;
            default:
                return 0;
        }
    }
}

