<?php

namespace App\Http\Livewire\Production\Reports;

use Livewire\Component;
use App\Models\Shipment;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Reactive;
use Livewire\Attributes\Renderless;

class CpRow extends Component
{
    public array $chunk;
    public $columns;
    public $drops;

    public function mount($chunk, $selected){
        $this->chunk = array_values($chunk);
        $this->selected = [];
        foreach($this->chunk as $sl){
            $this->selected[$sl['id']] = $selected[$sl['id']] ?? FALSE;
        }
        // dd($this->selected, $this->chunk);
    }

    #[Renderless]
    #[On('refreshRows')]
    public function refreshIfModified($modifiedIds)
    {
        if (!empty(array_intersect(collect($this->chunk)->pluck('id')->toArray(), json_decode($modifiedIds)))) {
            $this->render();
        }
    }

    public function render()
    {
        $this->drops = $this->getDrops();
        return view('livewire.production.reports.cp-row');
    }

    #[On('updateQty')]
    public function updateQty($sizeId, $value)
    {
        $this->updateShipmentLineSize($sizeId, 'qty', $value);
    }
    #[On('updateShippedQty')]
    public function updateShippedQty($sizeId, $value)
    {
        $this->updateShipmentLineSize($sizeId, 'shipped_qty', $value);
    }

    /**
     * Update shipment line size using Eloquent to ensure events are triggered.
     */
    private function updateShipmentLineSize($sizeId, $field, $value)
    {
        $size = ShipmentLineSizes::find($sizeId);
        if ($size) {
            $size->$field = $value;
            $size->save();
        }
    }

    /**
     * Update shipment line using Eloquent to ensure events are triggered.
     */
    private function updateShipmentLine($lineId, $field, $value)
    {
        $line = ShipmentLine::find($lineId);
        if ($line) {
            $line->$field = $value;
            $line->save();
        }
    }

    public array $selected = [];

    #[On('select-all')]
    public function selectAll(){
        foreach($this->chunk as $sl){
            $this->selected[$sl['id']] = true;
        }
    }
    #[On('clear-selected')]
    public function clearSelected(){
        foreach($this->chunk as $sl){
            $this->selected[$sl['id']] = false;
        }
    }

    public function updateFlagIssue($id){
        $line = ShipmentLine::find($id);
        $line->update(['flag_issue' => !$line->flag_issue]);
    }

    public function updated($property, $value)
    {
        // Extract parts of the property name
        $parts = explode('.', $property);

        if ($parts[0] === 'drops') {
            $dropIndex = $parts[1];
            // Retrieve the actual ShipmentLine ID
            $dropId = $this->drops[$dropIndex]['id'] ?? null;

            // Handle shipment_line_sizes updates
            if (isset($parts[2]) && $parts[2] === 'shipment_line_sizes') {
                $sizeIndex = $parts[3]; // Array index, NOT the ID
                $field = $parts[4] ?? null; // Field being updated

                // Get the actual ID from the array
                $sizeId = $this->drops[$dropIndex]['shipment_line_sizes'][$sizeIndex]['id'] ?? null;

                if ($sizeId && in_array($field, ['qty', 'shipped_qty'])) {
                    $this->updateShipmentLineSize($sizeId, $field, $value);
                    return;
                }
            }

            // Handle shipments updates
            if (isset($parts[2]) && $parts[2] === 'shipments') {
                $shipmentField = $parts[3] ?? null;

                if ($shipmentField === 'transporter_invoice') {
                    Shipment::where('id', $this->drops[$dropIndex]['shipments']['id'] ?? null)
                        ->update(['transporter_invoice' => $value]);
                    return;
                }
            }

            if (in_array($parts[2] ?? null, ['notes', 'fn_notes', 'rt_invoice', 'factory_invoice', 'customs_agency_invoice', 'invoiced', 'net_weight', 'gross_weight', 'no_cartons'])) {
                $this->updateShipmentLine($dropId, $parts[2], $value);
                return;
            }
        }
    }

    public function getDrops(){
        $ids = collect($this->chunk)->pluck('id')->toArray();
        $query = ShipmentLine::with([
            'customer_order_lines:id,customer_orders_id,colourways_id,factory_cust_date,wh_cust_date,cancelled' => [
                'customer_orders:id,customers_id,seasons_id,order_date,customer_po,incoterms' => [
                    'seasons:id,description',
                    'customers:id,name,currency,settings,samples_required'
                ],
                'colourways:id,style_versions_id,name,img_thumb,cancelled' => [
                    'style_versions:id,styles_id,factory_id,name' => [
                        'factories:id,name,currency',
                        'styles:id,designs_id,customer_ref,cancelled' => [
                            'designs:id,description'
                        ]
                    ]
                ],
                'customer_order_line_quantities' => [
                    'sizes',
                    'customer_order_lines:id,customer_orders_id,colourways_id' => [
                        'colourways:id,style_versions_id,name,img_thumb'
                    ]
                ]
            ],
            'shipment_line_sizes.sizes',
            'shipments:id,transporter_id,cost,organiser,customs_submitted_text,transporter_invoice' => [
                'transporters:id,currency,name',
            ],

        ])
        ->whereIn('id', $ids)
        ->orderByRaw("FIELD(id, " . implode(',', $ids) . ")");

        $data = $query->get();

        $data->each(function ($shipmentLine) {
            $shipmentLine->append('total_pieces', 'total_pieces_shipped', 'total_prices', 'first_exfty', 'shipment_sample', 'sealer_sample');
            // $shipmentLine->customer_order_lines->append('total_value');
        });

        return $data->toArray();
    }
}
