<?php

namespace App\Http\Livewire\Production\Reports;

use Livewire\Component;
use App\Models\Shipment;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Reactive;
use Livewire\Attributes\Renderless;
use App\Models\CustomerOrderLineQuantities;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;

class CommissionShipmentScheduleRow extends Component
{
    public array $chunk;
    public $columns;
    public $drops;
    public $editQtys;

    public function mount($chunk, $selected, $editQtys = false){
        // Chunk now contains IDs only (integers), not arrays
        $this->chunk = $chunk;
        $this->editQtys = $editQtys;
        $this->selected = [];
        
        // Map selection for IDs
        foreach($this->chunk as $id){
            $this->selected[$id] = $selected[$id] ?? false;
        }
    }

    /**
     * When selection changes in the row, notify the parent component
     */
    public function updatedSelected($value, $key)
    {
        // Dispatch to parent to sync selection state
        $this->dispatch('row-selection-changed', id: (int)$key, selected: (bool)$value)->to(CommissionShipmentScheduleModern::class);
    }

    #[Renderless]
    #[On('refreshRows')]
    public function refreshIfModified($modifiedIds)
    {
        // Chunk now contains IDs directly
        if (!empty(array_intersect($this->chunk, json_decode($modifiedIds)))) {
            $this->render();
        }
    }

    public function render()
    {
        // Get the actual data for these IDs
        $this->drops = $this->getDrops();
        return view('livewire.production.reports.commission-shipment-schedule-row');
    }

    #[On('updateQty')]
    public function updateQty($sizeId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
        )->validate();

        $this->updateShipmentLineSize($sizeId, 'qty', empty($validatedData['value']) ? 0 : $validatedData['value']);
        // Refresh the component to get updated data
        $this->render();
    }
    
    #[On('updateShippedQty')]
    public function updateShippedQty($sizeId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
        )->validate();

        $processedValue = empty($validatedData['value']) ? 0 : $validatedData['value'];
        $this->updateShipmentLineSize($sizeId, 'shipped_qty', $processedValue);
        
        // Update the local data immediately to prevent reverting to 0
        $this->updateLocalData($sizeId, 'shipped_qty', $processedValue);
    }

    #[On('updatePrice')]
    public function updatePrice($colqId, $value)
    {
        if (Gate::check('order:update')) {
            \DB::table('customer_order_line_quantities')
                ->where('id', $colqId)
                ->update([
                    'price' => $value,
                    'updated_at' => now()
                ]);
        }
    }

    #[On('updateDiscount')]
    public function updateDiscount($colqId, $value)
    {
        if (Gate::check('order:update')) {
            \DB::table('customer_order_line_quantities')
                ->where('id', $colqId)
                ->update([
                    'discount' => $value,
                    'updated_at' => now()
                ]);
        }
    }

    #[On('updateCommission')]
    public function updateCommission($colqId, $value)
    {
        if (Gate::check('order:update')) {
            \DB::table('customer_order_line_quantities')
                ->where('id', $colqId)
                ->update([
                    'commission' => $value,
                    'updated_at' => now()
                ]);
        }
    }

    public function updateShipmentComplete($shipmentLineId, $value)
    {
        if (Gate::check('order:update')) {
            \DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->update([
                    'complete' => $value,
                    'updated_at' => now()
                ]);
        }
    }

    public function updateExftyDate($shipmentLineId, $date)
    {
        if (Gate::check('order:update')) {
            // Get the customer_order_lines_id first
            $customerOrderLineId = \DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->value('customer_order_lines_id');
            
            if ($customerOrderLineId) {
                \DB::table('customer_order_lines')
                    ->where('id', $customerOrderLineId)
                    ->update([
                        'factory_cust_date' => $date ?: null,
                        'updated_at' => now()
                    ]);
            }
        }
    }

    public function updateRevisedExfty($shipmentLineId, $date)
    {
        if (Gate::check('order:update')) {
            \DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->update([
                    'exfty' => $date ?: null,
                    'updated_at' => now()
                ]);
        }
    }

    /**
     * Update shipment line size using direct SQL query for better performance.
     */
    private function updateShipmentLineSize($sizeId, $field, $value)
    {
        if (Gate::check('order:update') || Gate::check('finance:update')) {
            \DB::table('shipment_line_sizes')
                ->where('id', $sizeId)
                ->update([
                    $field => $value,
                    'updated_at' => now()
                ]);
        }
    }

    /**
     * Update local data immediately to prevent UI from reverting to old values
     */
    private function updateLocalData($sizeId, $field, $value)
    {
        if (isset($this->drops)) {
            foreach ($this->drops as &$drop) {
                if (isset($drop['shipment_line_sizes'])) {
                    foreach ($drop['shipment_line_sizes'] as &$size) {
                        // The SQL query uses 'sls_id' as the key for the shipment line size ID
                        if (isset($size['sls_id']) && $size['sls_id'] == $sizeId) {
                            $size[$field] = $value;
                            break 2; // Break out of both loops
                        }
                    }
                }
            }
        }
    }

    /**
     * Update shipment line using direct SQL query for better performance.
     */
    public function updateShipmentLine($lineId, $field, $value)
    {
        if (Gate::check('style:update') || Gate::check('order:update') || Gate::check('finance:update')) {
            // Convert empty strings to appropriate values for integer fields
            if ($field === 'no_invoiced' && $value === '') {
                $value = 0;
            }
            
            // Convert 'invoiced' (finance_complete) to boolean
            if ($field === 'invoiced') {
                $value = (bool) $value;
            }
            
            \DB::table('shipment_lines')
                ->where('id', $lineId)
                ->update([
                    $field => $value,
                    'updated_at' => now()
                ]);
        }
    }

    public array $selected = [];

    #[On('select-all')]
    public function selectAll(){
        foreach($this->chunk as $id){
            $this->selected[$id] = true;
        }
    }
    
    #[On('clear-selected')]
    public function clearSelected(){
        foreach($this->chunk as $id){
            $this->selected[$id] = false;
        }
    }

    public function updateFlagIssue($id){
        if (Gate::check('order:update')) {
            \DB::table('shipment_lines')
                ->where('id', $id)
                ->update([
                    'flag_issue' => \DB::raw('NOT flag_issue'),
                    'updated_at' => now()
                ]);
        }
    }

    public function updated($property, $value)
    {
        // Extract parts of the property name
        $parts = explode('.', $property);

        if ($parts[0] === 'drops') {
            $dropIndex = $parts[1];
            // Retrieve the actual ShipmentLine ID
            $dropId = $this->drops[$dropIndex]['id'] ?? null;

            // Handle shipment_line_sizes updates
            if (isset($parts[2]) && $parts[2] === 'shipment_line_sizes') {
                $sizeIndex = $parts[3]; // Array index, NOT the ID
                $field = $parts[4] ?? null; // Field being updated

                // Get the actual ID from the array
                $sizeId = $this->drops[$dropIndex]['shipment_line_sizes'][$sizeIndex]['id'] ?? null;

                if ($sizeId && in_array($field, ['qty', 'shipped_qty'])) {
                    $this->updateShipmentLineSize($sizeId, $field, $value);
                    return;
                }
            }

            if (in_array($parts[2] ?? null, ['notes', 'fn_notes', 'rt_invoice', 'factory_invoice', 'customs_agency_invoice', 'invoiced', 'net_weight', 'gross_weight', 'no_cartons'])) {
                $this->updateShipmentLine($dropId, $parts[2], $value);
                return;
            }
        }
    }

    public function getDrops(){
        // Get the actual data for these IDs (like wholesale schedule)
        $ids = $this->chunk; // Now contains array of integers
        
        if (empty($ids)) {
            return [];
        }

        // Simplified query - get essential data only
        $results = \DB::table('shipment_lines as sl')
            ->select([
                'sl.id as shipment_lines_id',
                'sl.complete as shipped',
                'sl.invoiced as finance_complete',
                'sl.exfty as revised_exfty',
                'sl.fn_notes',
                'sl.rt_invoice',
                'sl.factory_invoice',
                'sl.no_invoiced',
                's.description as season',
                'f.name as factory',
                'f.id as factory_id',
                'c.name as customer',
                'c.id as customer_id',
                'c.currency as cust_currency',
                'co.order_date as order_issued',
                'co.customer_po',
                'co.id as customer_orders_id',
                'col.factory_cust_date as factory_cust_date',
                'd.description as description',
                'cw.name as colour',
                'cw.img_thumb',
                'st.id as styles_id',
                'st.designs_id',
                'sv.name as style_version_name',
                'st.customer_ref',
            ])
            ->selectRaw("
                COALESCE(
                    (SELECT JSON_UNQUOTE(JSON_EXTRACT(audit.new_values, '$.exfty'))
                     FROM audits AS audit
                     WHERE audit.auditable_id = sl.id
                     AND audit.auditable_type = 'App\\\\Models\\\\ShipmentLine'
                     AND JSON_EXTRACT(audit.new_values, '$.exfty') IS NOT NULL
                     ORDER BY audit.created_at ASC LIMIT 1),
                    sl.exfty
                ) AS first_exfty
            ")
            ->selectRaw("
                (SELECT CONCAT('[',
                    GROUP_CONCAT(
                        JSON_OBJECT(
                            'sls_id', sls.id,
                            'name', sizes.name,
                            'sizes_id', sizes.id,
                            'qty', sls.qty,
                            'shipped_qty', sls.shipped_qty,
                            'sizes', JSON_OBJECT('id', sizes.id, 'name', sizes.name)
                        ) ORDER BY sizes.`order` SEPARATOR ','
                    ), ']')
                 FROM shipment_line_sizes sls
                 JOIN sizes ON sizes.id = sls.sizes_id
                 WHERE sls.shipment_line_id = sl.id
                ) AS sizes
            ")
            ->selectRaw("
                (SELECT CONCAT('[',
                    GROUP_CONCAT(
                        JSON_OBJECT(
                            'id', colq.id,
                            'price', colq.price,
                            'discount', colq.discount,
                            'commission', colq.commission
                        ) ORDER BY sizes.`order` SEPARATOR ','
                    ), ']')
                 FROM customer_order_line_quantities colq
                 JOIN sizes ON sizes.id = colq.sizes_id
                 WHERE colq.customer_order_lines_id = sl.customer_order_lines_id
                ) AS quantities
            ")
            ->selectRaw("
                (SELECT SUM(sls.qty) FROM shipment_line_sizes sls WHERE sls.shipment_line_id = sl.id) AS total_qty
            ")
            ->selectRaw("
                (SELECT SUM(sls.shipped_qty) FROM shipment_line_sizes sls WHERE sls.shipment_line_id = sl.id) AS total_shipped
            ")
            
            ->join('customer_order_lines as col', 'col.id', '=', 'sl.customer_order_lines_id')
            ->join('customer_orders as co', 'co.id', '=', 'col.customer_orders_id')
            ->join('customers as c', 'c.id', '=', 'co.customers_id')
            ->join('colourways as cw', 'cw.id', '=', 'col.colourways_id')
            ->join('style_versions as sv', 'sv.id', '=', 'cw.style_versions_id')
            ->join('styles as st', 'st.id', '=', 'sv.styles_id')
            ->join('seasons as s', 's.id', '=', 'co.seasons_id')
            ->join('suppliers as f', 'f.id', '=', 'sv.factory_id')
            ->join('designs as d', 'd.id', '=', 'st.designs_id')
            ->whereIn('sl.id', $ids)
            ->orderByRaw("FIELD(sl.id, " . implode(',', $ids) . ")")
            ->get();

        // Minimal processing
        $drops = [];
        foreach ($results as $line) {
            $sizes = json_decode($line->sizes, true) ?? [];
            $quantities = json_decode($line->quantities, true) ?? [];
            
            $drop = (array) $line;
            $drop['id'] = $line->shipment_lines_id;
            $drop['complete'] = (bool) $line->shipped;
            $drop['finance_complete'] = (bool) $line->finance_complete;
            
            // Structure for template
            $drop['shipment_line_sizes'] = [];
            $drop['customer_order_lines'] = ['customer_order_line_quantities' => []];

            $drop['shipment_line_sizes'] = $sizes;
            $orderedQuantities = $quantities;
            
            // Set the ordered quantities array
            $drop['customer_order_lines']['customer_order_line_quantities'] = $orderedQuantities;
            
            $drops[] = $drop;
        }
        
        return $drops;
    }

    public function notesColsCount()
    {
        $i = 0;
        if ($this->columns['Season']) $i++;
        if ($this->columns['Factory']) $i++;
        if ($this->columns['Customer']) $i++;
        if ($this->columns['Order No']) $i++;
        if ($this->columns['Style']) $i++;
        if ($this->columns['Description']) $i++;
        if ($this->columns['Colour']) $i++;
        return $i;
    }

    public function fnNotesColsCount()
    {
        $i = 0;
        if ($this->columns['Finance']) $i++;
        return $i;
    }
}
