<?php

namespace App\Http\Livewire\Production\Reports;

use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Shipment;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Models\Departments;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithPagination;
use App\Models\StyleVersions;
use Illuminate\Support\Carbon;
use App\Models\ShipmentLineSize;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Renderless;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Models\CustomerOrderLineQuantity;
use Illuminate\Support\Facades\Validator;
use App\Http\Livewire\FilterableComponent;
use App\Models\CustomerOrderLineQuantities;

class CommissionShipmentScheduleModern extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    // Properties for filters and display options
    public $hideFilters = false;
    public $borders = false;
    public $view = 'unshipped';
    public $fromDate = '2024-01-01';
    public $toDate = '2025-12-31';
    public $department = [];
    public $customer = [];
    public $season = [];
    public $factory = [];
    public $coo = [];
    public $category = [];
    public $search = "";
    public $selected = [];
    public $editQtys = false;

    // Pagination and chunking
    public $page = 1;
    public $refreshKey = 0;
    public $chunks = [];

    public $columns = [
        'Season' => true,
        'Factory' => true,
        'Customer' => true,
        'Order Issued' => true,
        'Order No' => true,
        'Style' => true,
        'Description' => true,
        'Colour' => true,
        'Image' => true,
        'Sizes' => true,
        'Total Qty' => true,
        'Total Shipped' => true,
        'Commission' => true,
        'Value With Discount' => true,
        'Value Without Discount' => true,
        'ExFty' => true,
        'Revised ExFty' => true,
        'Shipped' => true,
        'Notes For Finance' => true,
        'Finance' => false,
    ];

    // Filter dropdown visibility properties
    public $showSeasonDropdown = false;
    public $showCustomerDropdown = false;
    public $showFactoryDropdown = false;
    public $showCooDropdown = false;
    public $showDeptDropdown = false;
    public $showCatDropdown = false;
    public $showEndedSeasons = false;

    // Validation rules
    #[Validate([
        'hideFilters' => ['nullable', 'boolean'],
        'view' => ['nullable', 'in:unshipped,financeNC,noRTInvoice,all'],
        'fromDate' => ['nullable', 'date', 'before_or_equal:toDate'],
        'toDate' => ['nullable', 'date', 'after_or_equal:fromDate'],
        'department' => ['nullable', 'array', 'exists:departments,id'],
        'customer' => ['nullable', 'array', 'exists:customers,id'],
        'season' => ['nullable', 'array', 'exists:seasons,id'],
        'factory' => ['nullable', 'array', 'exists:suppliers,id'],
        'coo' => ['nullable', 'array', 'exists:countries,id'],
        'category' => ['nullable', 'array', 'in:ladies,mens,accessories,childrens'],
        'search' => ['nullable', 'string', 'max:30'],

        'selected' => ['array'],
        'selected.*' => ['nullable', 'boolean'],

        'editQtys' => ['nullable', 'boolean'],
    ])]

    protected function filterKey(): string
    {
        return 'commissionshipmentschedulemodernfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.2;
    }

    protected function filters(): array
    {
        return ['view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'columns'];
    }

    protected function getDefaultColumns(): array
    {
        return $this->columns;
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
        // Force Livewire to recognize the columns property has changed
        $this->columns = $this->columns;
        $this->refreshKey = rand();
    }

    public function mount()
    {
        $this->loadFilterSettings();
        $this->refreshKey = rand();
        $this->getData();
    }

    public function getData()
    {
        // Get only IDs first (like wholesale schedule) - much faster
        $query = \DB::table('shipment_lines as sl')
            ->select('sl.id')
            ->join('customer_order_lines as col', function($join) {
                $join->on('col.id', '=', 'sl.customer_order_lines_id')
                     ->where('col.cancelled', 0)
                     ->whereNull('col.deleted_at');
            })
            ->join('customer_orders as co', function($join) {
                $join->on('co.id', '=', 'col.customer_orders_id')
                     ->where('co.order_type', 'commission')
                     ->where('co.cancelled', 0);
            })
            ->join('customers as c', 'c.id', '=', 'co.customers_id')
            ->join('colourways as cw', 'cw.id', '=', 'col.colourways_id')
            ->join('style_versions as sv', 'sv.id', '=', 'cw.style_versions_id')
            ->join('styles as st', 'st.id', '=', 'sv.styles_id')
            ->join('seasons as s', 's.id', '=', 'co.seasons_id')
            ->join('suppliers as f', 'f.id', '=', 'sv.factory_id')
            ->join('designs as d', 'd.id', '=', 'st.designs_id');

        // Apply filters (simplified)
        if ($this->view === 'unshipped') {
            $query->where('sl.complete', 0);
        } elseif ($this->view === 'financeNC') {
            $query->where('sl.invoiced', 0);
        } elseif ($this->view === 'noRTInvoice') {
            $query->where(function($q) {
                $q->whereNull('sl.rt_invoice')->orWhere('sl.rt_invoice', '');
            });
        }

        if ($this->fromDate) {
            $query->where('sl.exfty', '>=', $this->fromDate);
        }
        if ($this->toDate) {
            $query->where('sl.exfty', '<=', $this->toDate);
        }

        // Apply other filters if needed
        if (!empty($this->customer)) {
            $query->whereIn('c.id', array_keys(array_filter($this->customer)));
        }
        if (!empty($this->season)) {
            $query->whereIn('s.id', array_keys(array_filter($this->season)));
        }
        if (!empty($this->factory)) {
            $query->whereIn('f.id', array_keys(array_filter($this->factory)));
        }

        // Apply department filter
        if (!empty($this->department)) {
            $query->whereIn('d.departments_id', array_keys(array_filter($this->department)));
        }

        // Apply COO (Country of Origin) filter
        if (!empty($this->coo)) {
            $query->whereIn('f.countries_id', array_keys(array_filter($this->coo)));
        }

        // Apply category filter
        if (!empty($this->category)) {
            $query->whereIn('st.category', array_keys(array_filter($this->category)));
        }

        // Apply search filter
        if (!empty($this->search)) {
            $searchTerm = '%' . $this->search . '%';
            $query->where(function($q) use ($searchTerm) {
                $q->where('c.name', 'LIKE', $searchTerm)
                  ->orWhere('st.customer_ref', 'LIKE', $searchTerm)
                  ->orWhere('d.description', 'LIKE', $searchTerm)
                  ->orWhere('cw.name', 'LIKE', $searchTerm)
                  ->orWhere('f.name', 'LIKE', $searchTerm)
                  ->orWhere('co.customer_po', 'LIKE', $searchTerm)
                  ->orWhere('sl.id', 'LIKE', $searchTerm);
            });
        }

        // Get IDs and chunk them (fast!)
        $ids = $query->pluck('sl.id')->toArray();
        $this->chunks = array_chunk($ids, 25);
        $this->refreshKey = rand();
    }

    public function hasMorePages()
    {
        return $this->page < count($this->chunks);
    }

    public function loadMore()
    {
        if ($this->hasMorePages()) {
            $this->page++;
        }
    }

    public function updated($property, $value)
    {
        // Validate individual field if it's in the validation rules
        if (in_array($property, ['view', 'fromDate', 'toDate', 'search'])) {
            $this->validateOnly($property);
        }

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($property)) {
            $this->reset($property);
        } else {
            // Reset pagination when filters change
            if (in_array($property, $this->filters())) {
                $this->page = 1;
                $this->getData();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }

    public function selectAll()
    {
        foreach ($this->chunks as $chunk) {
            foreach ($chunk as $dropId) {
                $this->selected[$dropId] = true;
            }
        }
        $this->dispatch('select-all');
    }

    /**
     * Edit selected drops.
     */
    public function edit()
    {
        $this->dispatch('edit-c-ss', selected: json_encode(array_keys($this->selected, true)));
    }

    /**
     * Reset selected items and refresh data after bulk edit.
     */
    #[On('reset-selected')]
    public function resetSelected()
    {
        $this->selected = [];
        $this->getData();
        $this->refreshKey = rand();
    }

    /**
     * Handle selection changes from row components
     */
    #[On('row-selection-changed')]
    public function handleRowSelectionChanged($id, $selected)
    {
        $this->selected[$id] = $selected;
    }

    public function toggleEndedSeasons()
    {
        $this->showEndedSeasons = !$this->showEndedSeasons;
    }
    
    public function select($filter, $value = 0)
    {
        if (is_string($this->$filter)) {
            $this->$filter = [];
        }
        if ($value == 0) {
            $this->$filter = [];
        } else {
            if (($this->$filter[$value] ?? 0) == 0) {
                $this->$filter[$value] = 1;
            } else {
                unset($this->$filter[$value]);
            }
        }

        auth()->user()->settings(['commission_shipment_schedule_modern_' . $filter . '_new' => $this->$filter]);
        
        // Trigger the updated method to apply the filter and refresh the data
        $this->updated($filter, $this->$filter);
    }
    
    public function clearFilters()
    {
        $this->view = 'unshipped';
        $this->fromDate = '2024-01-01';
        $this->toDate = '2025-12-31';
        $this->department = [];
        $this->customer = [];
        $this->season = [];
        $this->factory = [];
        $this->coo = [];
        $this->category = [];
        $this->search = "";
        $this->selected = [];
        $this->page = 1;
        $this->getData();
        $this->saveFilterSettings();
    }

    #[Computed]
    public function seasons()
    {
        // Use direct SQL for faster loading
        $query = \DB::table('seasons')
            ->whereNull('deleted_at');
        
        // Filter out ended seasons if showEndedSeasons is false
        if (!$this->showEndedSeasons) {
            $query->where(function($q) {
                $q->whereNull('end_date')
                  ->orWhere('end_date', '>=', now());
            });
        }
        
        return $query->orderBy('created_at', 'desc')
            ->get(['id', 'description', 'end_date', 'created_at']);
    }
    
    #[Computed]
    public function endedSeasons()
    {
        // Get only ended seasons (where end_date is in the past)
        return \DB::table('seasons')
            ->whereNull('deleted_at')
            ->whereNotNull('end_date')
            ->where('end_date', '<', now())
            ->orderBy('created_at', 'desc')
            ->get(['id', 'description', 'end_date', 'created_at']);
    }

    #[Computed]
    public function customers()
    {
        // Only show customers that have shipment lines on the schedule
        return \DB::table('customers')
            ->whereNull('customers.deleted_at')
            ->whereExists(function($query) {
                $query->select(\DB::raw(1))
                    ->from('customer_orders as co')
                    ->join('customer_order_lines as col', 'col.customer_orders_id', '=', 'co.id')
                    ->join('shipment_lines as sl', 'sl.customer_order_lines_id', '=', 'col.id')
                    ->whereColumn('co.customers_id', 'customers.id');
            })
            ->orderBy('name')
            ->get(['customers.id', 'customers.name']);
    }

    #[Computed]
    public function factories()
    {
        // Use direct SQL with countries join for faster loading
        return \DB::table('suppliers as s')
            ->leftJoin('countries as c', 'c.id', '=', 's.countries_id')
            ->where('s.type', 'factories')
            ->whereNull('s.deleted_at')
            ->orderBy('s.name')
            ->get(['s.id', 's.name', 's.countries_id', 'c.country']);
    }

    #[Computed]
    public function departments()
    {
        // Use direct SQL for faster loading
        return \DB::table('departments')
            ->whereNull('deleted_at')
            ->orderBy('description')
            ->get(['id', 'description']);
    }

    #[On('close-modal')]
    public function modalClosed()
    {
        $this->render();
    }

    public function render()
    {
        Gate::authorize('shipment:read');

        return view('livewire.production.reports.commission-shipment-schedule-modern', [
            'chunks' => $this->chunks,
            'selectedTotalShipped' => $this->calculateSelectedTotalShipped(),
        ]);
    }

    private function calculateSelectedTotalShipped()
    {
        if (empty($this->selected)) {
            return 0;
        }

        // Quick calculation from selected IDs
        $selectedIds = array_keys(array_filter($this->selected));
        if (empty($selectedIds)) {
            return 0;
        }

        return \DB::table('shipment_line_sizes')
            ->whereIn('shipment_line_id', $selectedIds)
            ->sum('shipped_qty');
    }

    // Methods to handle updates - Using direct SQL for better performance
    #[Renderless]
    public function updateQty($slsId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
            )->validate();

        if (Gate::check('order:update')) {
            \DB::table('shipment_line_sizes')
                ->where('id', $slsId)
                ->update([
                    'qty' => empty($validatedData['value']) ? 0 : $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updateShippedQty($slsId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
        )->validate();

        if (Gate::check('order:update')) {
            \DB::table('shipment_line_sizes')
                ->where('id', $slsId)
                ->update([
                    'shipped_qty' => empty($validatedData['value']) ? 0 : $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updatePrice($colqId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
        )->validate();

        if (Gate::check('order:update')) {
            \DB::table('customer_order_line_quantities')
                ->where('id', $colqId)
                ->update([
                    'price' => $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updateDiscount($colqId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0|max:100']
        )->validate();

        if (Gate::check('order:update')) {
            \DB::table('customer_order_line_quantities')
                ->where('id', $colqId)
                ->update([
                    'discount' => $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updateCommission($colqId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0|max:100']
        )->validate();

        if (Gate::check('order:update')) {
            \DB::table('customer_order_line_quantities')
                ->where('id', $colqId)
                ->update([
                    'commission' => $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updateRevisedExfty($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|date']
        )->validate();

        if (Gate::check('order:update')) {
            \DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->update([
                    'exfty' => $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updateShipped($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'boolean']
        )->validate();

        if (Gate::check('order:update')) {
            \DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->update([
                    'complete' => $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updateFnNotes($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|string']
        )->validate();

        if (Gate::check('style:update') || Gate::check('order:update') || Gate::check('finance:update')) {
            \DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->update([
                    'fn_notes' => $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updateRtInvoice($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|string']
        )->validate();

        if (Gate::check('finance:update')) {
            \DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->update([
                    'rt_invoice' => $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updateNoInvoiced($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|integer']
        )->validate();

        if (Gate::check('finance:update')) {
            \DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->update([
                    'no_invoiced' => empty($validatedData['value']) ? 0 : $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
    
    #[Renderless]
    public function updateFinanceComplete($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'boolean']
        )->validate();

        if (Gate::check('finance:update')) {
            \DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->update([
                    'invoiced' => $validatedData['value'],
                    'updated_at' => now()
                ]);
        }
    }
}