<?php

namespace App\Http\Livewire\Production\Reports;

use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Shipment;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Models\Departments;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithPagination;
use App\Models\StyleVersions;
use Illuminate\Support\Carbon;
use App\Models\ShipmentLineSize;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Renderless;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Models\CustomerOrderLineQuantity;
use Illuminate\Support\Facades\Validator;
use App\Http\Livewire\FilterableComponent;
use App\Models\CustomerOrderLineQuantities;

class CommissionShipmentSchedule extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    #[Validate([
        'hideFilters' => ['nullable', 'boolean'],
        'view' => ['nullable', 'in:unshipped,financeNC,all'],
        'fromDate' => ['nullable', 'date', 'before_or_equal:toDate'],
        'toDate' => ['nullable', 'date', 'after_or_equal:fromDate'],
        'department' => ['nullable', 'array', 'exists:departments,id'],
        'customer' => ['nullable', 'array', 'exists:customers,id'],
        'season' => ['nullable', 'array', 'exists:seasons,id'],
        'factory' => ['nullable', 'array', 'exists:suppliers,id'],
        'coo' => ['nullable', 'array', 'exists:countries,id'],
        'category' => ['nullable', 'array', 'in:ladies,mens,accessories,childrens'],
        'search' => ['nullable', 'string', 'max:30'],
        'paginate' => ['required', 'numeric', 'min:10', 'max:1000'],

        'selected' => ['array'],
        'selected.*' => ['nullable', 'boolean'],

        'editQtys' => ['nullable', 'boolean'],
    ])]
    public $hideFilters;
    public $view = 'unshipped', $fromDate = '2024-01-01', $toDate = '2025-12-31', $department, $customer, $season, $factory, $coo, $category, $search, $paginate = 25;
    public $selected = [];
    public $editQtys = FALSE;

    public $showSeasonDropdown, $showCustomerDropdown, $showFactoryDropdown, $showCooDropdown, $showDeptDropdown, $showCatDropdown, $showEndedSeasons = false;

    public $columns = [
        'Season' => true,
        'Factory' => true,
        'Customer' => true,
        'Order Issued' => true,
        'Order No' => true,
        'Style' => true,
        'Description' => true,
        'Colour' =>  true,
        'Image' => true,
        'Sizes' => true,
        'Total Qty' => true,
        'Commission' => true,
        'Value Without Discount' => true,
        'Value With Discount' => true,
        'Total Shipped' => true,
        'ExFty' => true,
        'Revised ExFty' => true,
        'Shipped' => true,
        'Notes For Finance' => true,
        'Finance' => false,
    ];

    protected function filterKey(): string
    {
        return 'commissionshipmentschedulefilters';
    }

    protected function columnsVersion(): float
    {
        return 1.2;
    }

    protected function filters(): array
    {
        return ['view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'columns', 'paginate'];
    }

        /**
     * Clear all filters.
     */
    public function clearFilters()
    {
        $this->reset('view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'columns', 'paginate');
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
        // Force Livewire to recognize the columns property has changed
        $this->columns = $this->columns;
    }

    #[On('refreshData')]
    public function refreshData()
    {
        // Refresh the page to show updated data
        $this->resetPage();
    }

    public function updated($propertyName, $val)
    {
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'paginate'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }

    public function mount()
    {
        $this->loadFilterSettings();
    }

    public function allCols()
    {
        $this->reset('columns');
        $this->updatedColumns();
    }

    public function toggleEndedSeasons()
    {
        $this->showEndedSeasons = !$this->showEndedSeasons;
    }
    
    public function select($filter, $value = 0)
    {
        if (is_string($this->$filter)) {
            $this->$filter = [];
        }
        if ($value == 0) {
            $this->$filter = [];
        } else {
            if (($this->$filter[$value] ?? 0) == 0) {
                $this->$filter[$value] = 1;
            } else {
                unset($this->$filter[$value]);
            }
        }

        auth()->user()->settings(['commission_shipment_schedule_' . $filter . '_new' => $this->$filter]);
        
        // Trigger the updated method to apply the filter and refresh the page
        $this->updated($filter, $this->$filter);
    }

    #[Computed]
    public function seasons()
    {
        $allSeasons = Seasons::whereHas('styles')->get()->sortByDesc('created_at');
        
        if ($this->showEndedSeasons) {
            return $allSeasons;
        }
        
        // Filter out ended seasons (where end_date is in the past)
        return $allSeasons->filter(function($season) {
            if (!$season->end_date) {
                return true; // Show seasons without an end date
            }
            return $season->end_date >= now();
        });
    }
    
    #[Computed]
    public function endedSeasons()
    {
        $allSeasons = Seasons::whereHas('styles')->get()->sortByDesc('created_at');
        
        // Get only ended seasons (where end_date is in the past)
        return $allSeasons->filter(function($season) {
            if (!$season->end_date) {
                return false;
            }
            return $season->end_date < now();
        });
    }

    #[Computed(cache: true, key: 'factories-with-countries-wherehas-style_versions')]
    public function factories()
    {
        return Suppliers::whereHas('style_versions')->where('type', 'factory')->with('countries')->get();
    }

    #[Computed(cache: true, key: 'customers-with-shipment-lines')]
    public function customers()
    {
        // Only show customers that have shipment lines on the schedule
        return Customer::whereHas('customer_orders.customer_order_lines.shipment_lines')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function departments()
    {
        return Departments::allCached()->sortBy('name');
    }

    #[On('close-modal')]
    public function modalClosed()
    {
        $this->render();
    }

    public function exportToCSV()
    {
        Gate::authorize('shipment:read');
        
        $drops = $this->efficientGetShipment()->get();
        
        $filename = 'commission-shipment-schedule-' . date('Y-m-d-His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            'Pragma' => 'no-cache',
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Expires' => '0',
        ];
        
        $callback = function() use ($drops) {
            $file = fopen('php://output', 'w');
            
            // Add UTF-8 BOM for Excel compatibility
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // Header row
            $headerRow = [
                'Drop#', 'Season', 'Factory', 'Customer', 'Order No', 'Style',
                'Description', 'Colour', 'Total Qty', 'Total Shipped',
                'Commission', 'Value With Discount', 'Value Without Discount',
                'First ExFty', 'Revised ExFty', 'Shipped', 'RT Invoice'
            ];
            
            fputcsv($file, $headerRow);
            
            // Add data rows
            foreach($drops as $drop) {
                $sizes = collect(json_decode($drop->sizes ?? '[]', true) ?: []);
                
                // Use the exact same calculations as the main table (lines 679-692 in this file)
                $totalQty = $sizes->sum('qty');
                $totalShipped = $sizes->sum('shipped_qty');
                
                // Commission calculation: (commission% / 100) * price * (shipped_qty if > 0, else qty)
                $commission = $sizes->map(function ($size) {
                    $commission = $size['commission'] ?? 0;
                    $price = $size['price'] ?? 0;
                    $qty = ($size['shipped_qty'] ?? 0) == 0 ? ($size['qty'] ?? 0) : ($size['shipped_qty'] ?? 0);
                    return $commission == 0 ? 0 : ($commission / 100) * $price * $qty;
                })->sum();
                
                // Value (this is "Value With Discount"): price * (shipped_qty if > 0, else qty)
                $value = $sizes->map(function ($size) {
                    $price = $size['price'] ?? 0;
                    $qty = ($size['shipped_qty'] ?? 0) == 0 ? ($size['qty'] ?? 0) : ($size['shipped_qty'] ?? 0);
                    return $price * $qty;
                })->sum();
                
                // Value Discount (this is displayed as "Value Without Discount"): (discount% / 100) * price * (shipped_qty if > 0, else qty)
                $valueDiscount = $sizes->map(function ($size) {
                    $discount = $size['discount'] ?? 0;
                    $price = $size['price'] ?? 0;
                    $qty = ($size['shipped_qty'] ?? 0) == 0 ? ($size['qty'] ?? 0) : ($size['shipped_qty'] ?? 0);
                    return $discount == 0 ? 0 : ($discount / 100) * $price * $qty;
                })->sum();
                
                $currency = $drop->cust_currency ?? '';
                
                $row = [
                    $drop->shipment_lines_id,
                    $drop->season ?? '',
                    $drop->factory ?? '',
                    $drop->customer ?? '',
                    $drop->customer_po ?? '',
                    $drop->designs_id ?? '',
                    $drop->description ?? '',
                    $drop->colour ?? '',
                    $totalQty,
                    $totalShipped,
                    $currency . number_format($commission, 2),
                    $currency . number_format($value, 2),
                    $currency . number_format($valueDiscount, 2),
                    $drop->first_exfty ?? '',
                    $drop->revised_exfty ?? '',
                    $drop->shipped ? 'Yes' : 'No',
                    $drop->rt_invoice ?? ''
                ];
                
                fputcsv($file, $row);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }

    public function render()
    {
        Gate::authorize('shipment:read');

        $drops = $this->efficientGetShipment();

        return view('livewire.production.reports.commission-shipment-schedule', [
            'drops' => $drops,
            'selectedTotalShipped' => $this->calculateSelectedTotalShipped($drops),
        ]);
    }

    // Methods to handle updates
    #[Renderless]
    public function updateQty($slsId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
            )->validate();

        if (Gate::check('order:update')) {
            $sls = ShipmentLineSizes::find($slsId);
            if ($sls) {
                $sls->qty = empty($validatedData['value']) ? 0 : $validatedData['value'];
                $sls->save();
            }
        }
    }
    #[Renderless]
    public function updateShippedQty($slsId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
        )->validate();

        if (Gate::check('order:update')) {
            $sls = ShipmentLineSizes::find($slsId);
            if ($sls) {
                $sls->shipped_qty = empty($validatedData['value']) ? 0 : $validatedData['value'];
                $sls->save();
            }
        }
    }
    #[Renderless]
    public function updatePrice($colqId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
        )->validate();

        if (Gate::check('order:update')) {
            $colq = CustomerOrderLineQuantities::find($colqId);
            if ($colq) {
                $colq->price = $validatedData['value'];
                $colq->save();
            }
        }
    }
    #[Renderless]
    public function updateDiscount($colqId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0|max:100']
        )->validate();

        if (Gate::check('order:update')) {
            $colq = CustomerOrderLineQuantities::find($colqId);
            if ($colq) {
                $colq->discount = $validatedData['value'];
                $colq->save();
            }
        }
    }
    #[Renderless]
    public function updateCommission($colqId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0|max:100']
        )->validate();

        if (Gate::check('order:update')) {
            $colq = CustomerOrderLineQuantities::find($colqId);
            if ($colq) {
                $colq->commission = $validatedData['value'];
                $colq->save();
            }
        }
    }
    #[Renderless]
    public function updateRevisedExfty($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|date']
        )->validate();

        if (Gate::check('order:update')) {
            $shipmentLine = ShipmentLine::find($shipmentLineId);
            if ($shipmentLine) {
                $shipmentLine->exfty = $validatedData['value'];
                $shipmentLine->save();
            }
        }
    }
    #[Renderless]
    public function updateShipped($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'boolean']
        )->validate();

        if (Gate::check('order:update')) {
            $shipmentLine = ShipmentLine::find($shipmentLineId);
            if ($shipmentLine) {
                $shipmentLine->complete = $validatedData['value'];
                $shipmentLine->save();
            }
        }
    }
    #[Renderless]
    public function updateFnNotes($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|string']
        )->validate();

        if (Gate::check('style:update') || Gate::check('order:update') || Gate::check('finance:update')) {
            $shipmentLine = ShipmentLine::find($shipmentLineId);
            if ($shipmentLine) {
                $shipmentLine->fn_notes = $validatedData['value'];
                $shipmentLine->save();
            }
        }
    }
    #[Renderless]
    public function updateFactoryInvoice($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|string']
        )->validate();

        if (Gate::check('finance:update')) {
            $shipmentLine = ShipmentLine::find($shipmentLineId);
            if ($shipmentLine) {
                $shipmentLine->factory_invoice = $validatedData['value'];
                $shipmentLine->save();
            }
        }
    }
    #[Renderless]
    public function updateRtInvoice($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|string']
        )->validate();

        if (Gate::check('finance:update')) {
            $shipmentLine = ShipmentLine::find($shipmentLineId);
            if ($shipmentLine) {
                $shipmentLine->rt_invoice = $validatedData['value'];
                $shipmentLine->save();
            }
        }
    }
    #[Renderless]
    public function updateNoInvoiced($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|integer']
        )->validate();

        if (Gate::check('finance:update')) {
            $shipmentLine = ShipmentLine::find($shipmentLineId);
            if ($shipmentLine) {
                $shipmentLine->no_invoiced = empty($validatedData['value']) ? 0 : $validatedData['value'];
                $shipmentLine->save();
            }
        }
    }
    #[Renderless]
    public function updateFinanceComplete($shipmentLineId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'boolean']
        )->validate();

        if (Gate::check('finance:update')) {
            $shipmentLine = ShipmentLine::find($shipmentLineId);
            if ($shipmentLine) {
                $shipmentLine->invoiced = $validatedData['value'];
                $shipmentLine->save();
            }
        }
    }

    private function efficientGetShipment()
    {
        // The database query remains unaltered from your original code

        $query = ShipmentLine::select(
            "colourways.cancelled AS colourway_cancelled",
            "styles.cancelled AS style_cancelled",
            "shipment_lines.no_invoiced AS no_invoiced",
            "customer_orders.order_type AS order_type",
            "shipment_lines.id AS shipment_lines_id",
            "seasons.description AS season",
            "factory.name AS factory",
            "factory.id AS factory_id",
            "styles.departments_id AS departments_id",
            "styles.category AS category",
            "customers.name AS customer",
            "customers.id AS customer_id",
            "customer_orders.order_date AS order_issued",
            "customer_orders.customer_po AS customer_po",
            "designs.description AS description",
            "colourways.name AS colour",
            "colourways.img_thumb AS img_thumb",
            "shipment_lines.complete AS shipped",
            "shipment_lines.rt_invoice AS rt_invoice",
            "shipment_lines.factory_invoice AS factory_invoice",
            "shipment_lines.customs_agency_invoice AS customs_invoice",
            "shipment_lines.invoiced AS finance_complete",
            "shipment_lines.complete AS complete",
            "shipment_lines.shipment_id AS shipment_id",
            "shipment_lines.exfty AS revised_exfty",
            "customer_orders.id AS customer_orders_id",
            "styles.id AS styles_id",
            "colourways.id AS colourways_id",
            "colourways.colour_type AS colour_type",
            "style_versions.id AS style_versions_id",
            "customers.currency AS cust_currency",
            "factory.currency AS fact_currency",
            "factory.countries_id AS coo",
            "shipment_lines.fn_notes AS fn_notes",
            "shipment_lines.notes AS notes",
            "customers.factored AS customer_factored",
            "seasons.id AS seasons_id",
            "styles.designs_id",
            "styles.customer_ref",
        )
        ->selectRaw("
        COALESCE(
            (
                SELECT
                    JSON_UNQUOTE(JSON_EXTRACT(audit.new_values, '$.exfty'))
                FROM audits AS audit
                WHERE audit.auditable_id = shipment_lines.id
                AND audit.auditable_type = 'App\\\Models\\\ShipmentLine'
                AND JSON_EXTRACT(audit.new_values, '$.exfty') IS NOT NULL
                ORDER BY audit.created_at ASC
                LIMIT 1
            ),
            shipment_lines.exfty
        ) AS first_exfty
        ")
        ->selectRaw("
            (
                select
                concat(
                    '[',
                    group_concat(
                    JSON_OBJECT(
                        'sls_id', sls.id,
                        'colq_id', colq.id,
                        'name', sizes.name,
                        'size', sizes.name,
                        'sizes_id', sizes.id,
                        'qty', sls.qty,
                        'shipped_qty', sls.shipped_qty,
                        'price', colq.price,
                        'discount', colq.discount,
                        'commission', colq.commission
                    ) ORDER BY sizes.`order` SEPARATOR ','
                    ),
                    ']'
                )
                from  shipment_line_sizes sls
                join sizes on sizes.id = sls.sizes_id and sls.shipment_line_id = shipment_lines.id
                join customer_order_line_quantities colq on colq.customer_order_lines_id = customer_order_lines.id and colq.sizes_id = sls.sizes_id
                group by sls.shipment_line_id
            ) AS sizes
        ")
        ->Join('customer_order_lines', function($join)
        {
            $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id');
            $join->on('customer_order_lines.cancelled', '=', DB::raw(0));
        })
        ->Join('customer_orders', function($join)
        {
            $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
            $join->on('customer_orders.order_type', '=', DB::raw("'commission'"));
            $join->on('customer_orders.cancelled', '=', DB::raw(0));
        })
        ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
        ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
        ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
        ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
        ->join('designs', 'designs.id', '=', 'styles.designs_id')
        ->groupBy(
            'shipment_lines.id',
            'colourways.cancelled',
            'styles.cancelled',
            'shipment_lines.no_invoiced',
            'customer_orders.order_type',
            'shipment_lines.id',
            'seasons.description',
            'factory.name',
            'factory.id',
            'styles.departments_id',
            'styles.category',
            'customers.name',
            'customers.id',
            'customer_orders.order_date',
            'customer_orders.customer_po',
            'designs.description',
            'colourways.name',
            'colourways.img_thumb',
            'shipment_lines.complete',
            'shipment_lines.rt_invoice',
            'shipment_lines.factory_invoice',
            'shipment_lines.customs_agency_invoice',
            'shipment_lines.invoiced',
            'shipment_lines.complete',
            'shipment_lines.shipment_id',
            'first_exfty',
            'shipment_lines.exfty',
            'customer_orders.id',
            'styles.id',
            'colourways.id',
            'colourways.colour_type',
            'style_versions.id',
            'customers.currency',
            'factory.currency',
            'factory.countries_id',
            'shipment_lines.fn_notes',
            'shipment_lines.notes',
            'customers.factored',
            'seasons.id',
            'styles.designs_id',
            'styles.customer_ref',
            'customer_order_lines.id',
            )
        ->whereNull('shipment_lines.deleted_at');

        // Apply view-specific filters
        if($this->view == "unshipped"){
            $query->where('shipment_lines.complete', FALSE);
        }
        if($this->view == "financeNC"){
            $query->where('shipment_lines.invoiced', FALSE);
        }
        
        // Apply date filters independently of view selection
        if($this->fromDate){
            $query->where('shipment_lines.exfty', '>=', $this->fromDate);
        }
        if(!empty($this->toDate)){
            $query->where('shipment_lines.exfty', '<=', $this->toDate);
        }
        if(!empty($this->season) && is_array($this->season)){
            $query->whereIn('customer_orders.seasons_id', array_keys($this->season));
        }
        if(!empty($this->customer) && is_array($this->customer)){
            $query->whereIn('customer_orders.customers_id', array_keys($this->customer));
        }
        if(!empty($this->factory) && is_array($this->factory)){
            $query->whereIn('style_versions.factory_id', array_keys($this->factory));
        }
        if(!empty($this->coo) && is_array($this->coo)){
            $query->whereIn('factory.countries_id', array_keys($this->coo));
        }
        if(!empty($this->department) && is_array($this->department)){
            $query->whereIn('styles.departments_id', array_keys($this->department));
        }
        if(!empty($this->category) && is_array($this->category)){
            $query->whereIn('styles.category', array_keys($this->category));
        }
        if(!empty($this->search)){
            $searchTerm = '%' . $this->search . '%';
            $query->where(function($q) use ($searchTerm) {
                $q->where('customers.name', 'LIKE', $searchTerm)
                  ->orWhere('styles.customer_ref', 'LIKE', $searchTerm)
                  ->orWhere('designs.description', 'LIKE', $searchTerm)
                  ->orWhere('colourways.name', 'LIKE', $searchTerm)
                  ->orWhere('factory.name', 'LIKE', $searchTerm)
                  ->orWhere('customer_orders.customer_po', 'LIKE', $searchTerm)
                  ->orWhere('shipment_lines.id', 'LIKE', $searchTerm)
                  ->orWhere('styles.designs_id', 'LIKE', $searchTerm);
            });
        }

        $query->orderByRaw("
            -(shipment_lines.complete),
            shipment_lines.shipment_id,
            -(shipment_lines.exfty) desc,
            factory.countries_id,
            -(factory.name),
            -(customers.name),
            -(
            customer_orders.customer_po
            ),
            -(styles.id),
            -(colourways.name)
        ");

        $results = $query->simplePaginate($this->paginate);

        foreach($results as $l => $line){
            $results[$l]->shipped = (bool) $line->shipped;
            $results[$l]->finance_complete = (bool) $line->finance_complete;
            $results[$l]->sizes = collect(json_decode($line->sizes));

            $results[$l]->total_qty = $line->sizes->sum('qty');
            $results[$l]->total_shipped = $line->sizes->sum('shipped_qty');

            $results[$l]->commission = $line->sizes->map(function ($size) {
                return $size->commission == 0 ? 0 : ($size->commission / 100) * $size->price * ($size->shipped_qty == 0 ? $size->qty : $size->shipped_qty);
            })->sum();

            $results[$l]->value = $line->sizes->map(function ($size) {
                return $size->price * ($size->shipped_qty == 0 ? $size->qty : $size->shipped_qty);
            })->sum();

            $results[$l]->valueDiscount = $line->sizes->map(function ($size) {
                return $size->discount == 0 ? 0 : ($size->discount / 100) * $size->price * ($size->shipped_qty == 0 ? $size->qty : $size->shipped_qty);
            })->sum();

            $results[$l]->complete = $line->complete ? true : false;
            $results[$l]->finance_complete = $line->finance_complete ? true : false;
        }

        return $results;
    }

    private function calculateSelectedTotalShipped($drops)
    {
        $total = 0;
        foreach ($drops as $drop) {
            if (!empty($this->selected[$drop->shipment_lines_id])) {
                $total += (int) ($drop->total_shipped ?? 0);
            }
        }
        return $total;
    }
    /**
     * Select all drops.
     */
    public function selectAll()
    {
        $this->selected = [];
        foreach ($this->efficientGetShipment() as $drop) {
            $this->selected[$drop->shipment_lines_id] = true;
        }
    }

    /**
     * Edit selected drops.
     */
    public function edit()
    {
        $this->dispatch('edit-c-ss', selected: json_encode(array_keys($this->selected, true)));
    }

    /**
     * Reset selected items and refresh data after bulk edit.
     */
    #[On('reset-selected')]
    public function resetSelected()
    {
        $this->selected = [];
        $this->efficientGetShipment();
    }
}
