<?php

namespace App\Http\Livewire\Production\Reports;

use Cache;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Shipment;
use App\Helper\Functions;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Helper\Conversions;
use App\Models\Departments;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use App\Models\CommissionDrop;
use Livewire\WithPagination;
use App\Models\StyleVersions;
use Illuminate\Support\Number;
use Livewire\Attributes\Computed;

use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use App\Models\ViewShipmentSchedule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Intervention\Image\Facades\Image;

use App\Models\ShipmentLineExftyDates;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\ServiceProvider;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;
use App\Models\CustomerOrderLineCustDates;


class CommissionSalesValuesExfty extends FilterableComponent
{
	use WithPagination;
    use Filterable;

    #[Validate([
        'view' => ['required', 'in:unshipped,financeNC,all,noRTInvoice'],
        'fromDate' => ['required', 'date_format:Y-m-d', 'before_or_equal:toDate'],
        'toDate' => ['required', 'date_format:Y-m-d', 'after_or_equal:fromDate'],
    ])]
	public $view = 'unshipped', $fromDate = null, $toDate = null;

    public $hideFilters = false;

    public function mount()
    {
        $this->loadFilterSettings();

        // Set dynamic date defaults if not loaded from settings
        if ($this->fromDate === null) {
            $this->fromDate = now()->subYear()->format('Y-m-d');
        }
        if ($this->toDate === null) {
            $this->toDate = now()->addYears(5)->format('Y-m-d');
        }
    }

    protected function filterKey(): string
    {
        return 'orderslistfilters';
    }

    protected function columnsVersion(): float
    {
        return 1;
    }

    protected function filters(): array
    {
        return ['view', 'fromDate', 'toDate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['view', 'fromDate', 'toDate'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }


	public function render()
	{
		Gate::authorize('shipment:read');

		$drops = $this->efficientGetShipment;

        $pivot = [
            "dataSource" => [
                "dataSourceType" => "json",
                'data' => $drops->toArray(),
            ],
            "slice" => [
                "rows" => [
                    [
                        "uniqueName" => "revised_exfty.Month"
                    ],
                    [
                        "uniqueName" => "coo_name"
                    ],
                    [
                        "uniqueName" => "department"
                    ],
                    [
                        "uniqueName" => "factory"
                    ],
                    [
                        "uniqueName" => "customer"
                    ],
                    [
                        "uniqueName" => "category"
                    ],
                    [
                        "uniqueName" => "customer_ref"
                    ]
                ],
                "columns" => [
                    [
                        "uniqueName" => "revised_exfty.Year"
                    ],
                    [
                        "uniqueName" => "Measures"
                    ]
                ],
                "measures" => [
                    [
                        "uniqueName" => "Commission",
                        "aggregation" => "sum",
                        "format" => "5xyiaxxn"
                    ],
                    [
                        "uniqueName" => "qty",
                        "aggregation" => "sum",
                        "format" => "5y64ej8q",
                        "caption" => "Qty"
                    ],
                    [
                        "uniqueName" => "customer_ref",
                        "aggregation" => "distinctcount",
                        "caption" => "Styles Count"
                    ]
                ]
            ],
            "options" => [
                "showAggregationLabels" => false
            ],
            "formats" => [
                [
                    "name" => "5xyiaxxn",
                    "thousandsSeparator" => ",",
                    "decimalSeparator" => ".",
                    "decimalPlaces" => 0,
                    "currencySymbol" => "£",
                    "currencySymbolAlign" => "left",
                    "nullValue" => "",
                    "textAlign" => "right",
                    "isPercent" => false
                ],
                [
                    "name" => "5xyib2vm",
                    "thousandsSeparator" => ",",
                    "decimalSeparator" => ".",
                    "decimalPlaces" => 0,
                    "currencySymbol" => "£",
                    "currencySymbolAlign" => "left",
                    "nullValue" => "",
                    "textAlign" => "right",
                    "isPercent" => false
                ],
                [
                    "name" => "5y64ej8q",
                    "thousandsSeparator" => ",",
                    "decimalSeparator" => ".",
                    "currencySymbol" => "",
                    "currencySymbolAlign" => "left",
                    "nullValue" => "",
                    "textAlign" => "right",
                    "isPercent" => false
                ],
                [
                    "name" => "5yxxehbo",
                    "thousandsSeparator" => " ",
                    "decimalSeparator" => ".",
                    "decimalPlaces" => 1,
                    "currencySymbol" => "%",
                    "currencySymbolAlign" => "left",
                    "nullValue" => "",
                    "textAlign" => "right",
                    "isPercent" => true
                ]
            ]
        ];

        return view('livewire.production.reports.commission-sales-values-exfty', [
			'drops' => $drops,
			'pivot' => $pivot
        ]);
	}

    public function go(){
        return redirect(request()->header('Referer'));
    }

    #[Computed]
    private function efficientGetShipment(){
        // Using new commission tables
        $query = CommissionDrop::select(
            "seasons.description AS season",
            "factory.name AS factory",
            "departments.description AS department",
            "styles.category AS category",
            "customers.name AS customer",
            "commission_orders.customer_po AS customer_po",
            "commission_drops.exfty AS revised_exfty",
            "countries.country AS coo_name",
            "styles.customer_ref",
        )
        ->selectRaw("SUM(COALESCE(cds.shipped_qty, cds.qty)) AS qty")
        ->selectRaw("
            (
                SELECT
                    JSON_OBJECT(
                        'commission_total', CONVERT_CURRENCY(
                            SUM(((colq.price - (colq.price * (col.commission_percent / 100))) * (col.commission_percent / 100)) * COALESCE(NULLIF(cds.shipped_qty, 0), cds.qty)),
                            customers.currency,
                            '£',
                            commission_orders.seasons_id
                        )
                    ) AS totals
                FROM commission_drop_sizes cds
                JOIN sizes ON sizes.id = cds.sizes_id AND cds.commission_drops_id = commission_drops.id
                JOIN commission_order_line_quantities colq ON colq.commission_order_lines_id = commission_order_lines.id AND colq.sizes_id = cds.sizes_id
                WHERE cds.deleted_at IS NULL
                GROUP BY cds.commission_drops_id
            ) AS totals
        ")
        ->join('commission_order_lines', 'commission_order_lines.id', '=', 'commission_drops.commission_order_lines_id')
        ->join('commission_orders', function($join) {
            $join->on('commission_orders.id', '=', 'commission_order_lines.commission_orders_id')
                 ->where('commission_orders.cancelled', '=', 0)
                 ->where('commission_orders.status', '=', 'confirmed')
                 ->whereNull('commission_orders.deleted_at');
        })
        ->join('customers', 'customers.id', '=', 'commission_orders.customers_id')
        ->join('colourways', 'colourways.id', '=', 'commission_order_lines.colourways_id')
        ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->join('departments', 'departments.id', '=', 'styles.departments_id')
        ->join('seasons', 'seasons.id', '=', 'commission_orders.seasons_id')
        ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
        ->join('countries', 'countries.id', '=', 'factory.countries_id')
        ->join('designs', 'designs.id', '=', 'styles.designs_id')
        ->leftJoin('commission_order_line_quantities as colq', 'colq.commission_order_lines_id', '=', 'commission_order_lines.id')
        ->leftJoin('commission_drop_sizes as cds', function($join) {
            $join->on('cds.commission_drops_id', '=', 'commission_drops.id')
                 ->on('cds.sizes_id', '=', 'colq.sizes_id');
        })
        ->groupBy(
            'commission_drops.id',
            'seasons.description',
            'factory.name',
            'departments.description',
            'styles.category',
            'customers.name',
            'commission_orders.customer_po',
            'commission_drops.exfty',
            'countries.country',
            'styles.customer_ref',
            'customers.currency',
            'commission_orders.seasons_id',
            'factory.currency',
            'commission_order_lines.id'
        )
        ->whereNull('commission_drops.deleted_at')
        ->whereNull('commission_order_lines.deleted_at')
        ->where('commission_order_lines.cancelled', false);

        if($this->view == "unshipped"){
			$query->where('commission_drops.shipped', false);
		}
		elseif($this->view == "financeNC"){
			$query->where('commission_drops.invoiced', false);
		}
        elseif($this->view == "noRTInvoice"){
            $query->where(function($q) {
                $q->whereNull('commission_drops.rt_invoice')
                  ->orWhere('commission_drops.rt_invoice', '');
            });
        }
		else{
			$query->where('commission_drops.exfty', '>=', $this->fromDate);
		}

        if(!empty($this->toDate)){
			$query->where('commission_drops.exfty', '<=', $this->toDate);
		};

        $results = $query->get();

		foreach($results as $l=>$line){
			$totals = json_decode($line->totals);
            if($totals){
                $results[$l]->Commission = $totals->commission_total;
            }
            unset($results[$l]->totals);
		}

		return $results;
	}
}
