<?php

namespace App\Http\Livewire\Production;

use Livewire\Component;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\CustomerOrders;
use App\Models\CustomerOrderLines;
use Livewire\Attributes\Validate;

class OrderVsDrop extends Component
{
    public $seasonId;
    public $customerId;
    public $orderId;

    #[Validate([
        'seasonId' => ['nullable', 'exists:seasons,id'],
        'customerId' => ['nullable', 'exists:customers,id'],
        'orderId' => ['nullable', 'exists:customer_orders,id'],
    ])]
    public $filters = [];

    public function updatedSeasonId()
    {
        $this->orderId = null;
    }

    public function updatedCustomerId()
    {
        $this->orderId = null;
    }

    public function render()
    {
        return view('livewire.production.order-vs-drop', [
            'seasons' => $this->seasons(),
            'customers' => $this->customers(),
            'orders' => $this->orders(),
            'orderLines' => $this->orderLines(),
        ]);
    }

    public function seasons()
    {
        return Seasons::allCached()->sortByDesc('created_at');
    }

    public function customers()
    {
        return Customer::get()->sortBy('name');
    }

    public function orders()
    {
        $query = CustomerOrders::select('id', 'customer_po', 'seasons_id', 'customers_id')
            ->where('cancelled', false);

        if (! empty($this->seasonId)) {
            $query->where('seasons_id', $this->seasonId);
        }
        if (! empty($this->customerId)) {
            $query->where('customers_id', $this->customerId);
        }

        return $query->orderBy('order_date', 'desc')->get();
    }

    public function orderLines()
    {
        if (empty($this->orderId)) {
            return collect();
        }

        return CustomerOrderLines::with([
            'customer_orders:id,customer_po,customers_id,seasons_id,order_date',
            'colourways:id,style_versions_id,name' => [
                'style_versions:id,styles_id,factory_id,name' => [
                    'styles:id,designs_id,customer_ref' => [
                        'designs:id,description'
                    ]
                ]
            ],
            'customer_order_line_quantities' => [
                'sizes:id,name,order'
            ],
            'shipment_lines' => [
                'shipment_line_sizes' => [
                    'sizes:id,name,order'
                ],
                'shipments:id,transporter_id'
            ],
        ])
        ->where('customer_orders_id', $this->orderId)
        ->where('cancelled', false)
        ->get();
    }
}


