<?php

namespace App\Http\Livewire\Navigation;

use Livewire\Component;
use Livewire\Attributes\On;
use Illuminate\Support\Facades\Gate;

class NavigationDropdown extends Component
{
    public $menuOpen = false, $dropdownName, $img, $sub_headings, $sub_links, $tertiary; // Assuming you're passing $menuOpen as a public property

    // Map routes to required permissions
    protected $routePermissions = [
        // Wholesale routes
        'departments' => 'department:read',
        'seasons' => 'season:read',
        'yarns' => 'style:read',
        'materials' => 'style:read',
        'counts' => 'style:read',
        'sampletypes' => 'style:read',
        'sizes' => 'order:read',
        'gauges' => 'style:read',
        'invoicelayouts' => 'finance:update',
        'customers' => 'customer:read',
        'suppliers' => 'customer:read',
        
        // Commission routes - Products section
        'commission.styles' => 'style:read',
        'commission.samples' => 'style:read',
        'commission.styles-overview' => 'style:read',
        
        // Commission routes - Orders section
        'commission.orders' => 'order:read',
        'commission.batch-import' => 'order:create',
        
        // Commission routes - Shipments section
        'commission.shipment-schedule' => 'shipment:read',
        'imports.commission_pl' => 'shipment:create',
        
        // Commission routes - Invoicing section
        'commission.invoices' => 'finance:read',
        'commission.import-invoice' => 'finance:create',
        'commission.sl-invoicing' => 'finance:read',
    ];

    public function mount($dropdownName, $img, $subs, $tertiary = NULL)
    {
        $this->dropdownName = $dropdownName;
        $this->img = $img;
        
        // Filter sub_headings based on permissions
        $this->sub_headings = $this->filterByPermissions($subs);
        
        // Filter tertiary items based on permissions
        if ($tertiary) {
            $filtered_tertiary = [];
            foreach ($tertiary as $group => $items) {
                $filtered_items = $this->filterByPermissions($items);
                if (!empty($filtered_items)) {
                    $filtered_tertiary[$group] = $filtered_items;
                }
            }
            $this->tertiary = $filtered_tertiary;
        } else {
            $this->tertiary = null;
        }


        foreach($this->sub_headings as $value){
            $this->sub_links[] = $value;
        }
        foreach($this->tertiary ?? [] as $sub){
            foreach($sub as $value){
                $this->sub_links[] = $value;
            }
        }
    }
    
    /**
     * Get a clean ID-safe slug from the dropdown name
     */
    public function getDropdownId()
    {
        // Remove special characters and convert to lowercase
        $slug = strtolower($this->dropdownName);
        $slug = preg_replace('/[^a-z0-9\s-]/', '', $slug);
        $slug = preg_replace('/[\s-]+/', '-', $slug);
        $slug = trim($slug, '-');
        return $slug;
    }
    
    /**
     * Filter menu items based on user permissions
     */
    protected function filterByPermissions($items)
    {
        $filtered = [];
        foreach ($items as $label => $route) {
            // Check if this route requires a permission
            if (isset($this->routePermissions[$route])) {
                // Only include if user has the required permission
                if (Gate::check($this->routePermissions[$route])) {
                    $filtered[$label] = $route;
                }
            } else {
                // No permission required, include it
                $filtered[$label] = $route;
            }
        }
        return $filtered;
    }
    
    public function render()
    {
        $this->dispatch('dropdownData', ['subLinks' => $this->sub_links]);
        return view('livewire.navigation.navigation-dropdown');
    }
}
