<?php

namespace App\Http\Livewire\Logs;

use Livewire\Component;
use OwenIt\Auditing\Models\Audit;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Support\Str;
use ReflectionClass;

class ViewLogs extends Component
{
    public $selectedModel = '';
    public $selectedId;
    public $logs = [];
    public $models = []; // We'll populate this dynamically.

    public function mount()
    {
        $this->models = $this->getAuditableModels();
    }

    /**
     * Scans the app/Models directory and returns an array of
     * slug => Model::class for each model implementing Auditable.
     */
    protected function getAuditableModels(): array
    {
        // You can set this to wherever your models actually live.
        $modelsPath = app_path('Models');
        $modelFiles = glob($modelsPath.'/*.php');

        $results = [];

        foreach ($modelFiles as $file) {
            $filename = basename($file, '.php');
            $fullNamespace = "App\\Models\\{$filename}";

            if (class_exists($fullNamespace)) {
                $reflection = new ReflectionClass($fullNamespace);

                // Check if it implements Auditable
                if ($reflection->implementsInterface(Auditable::class)) {
                    // We'll use the class name as the key, or you could do something else
                    // like Str::plural(Str::snake($filename)) to match your existing usage
                    $slug = Str::snake($filename);
                    $results[$slug] = $fullNamespace;
                }
            }
        }

        return $results;
    }

    public function updatedSelectedModel()
    {
        $this->selectedId = null;
        $this->logs = [];
    }

    public function updatedSelectedId()
    {
        if ($this->selectedModel && $this->selectedId) {
            $this->fetchLogs();
        }
    }

    protected function fetchLogs()
    {
        $modelClass = $this->models[$this->selectedModel] ?? null;

        if (! $modelClass) {
            $this->logs = [];
            return;
        }

        $this->logs = Audit::with('user') // Eager-load the user relationship
            ->where('auditable_type', $modelClass)
            ->where('auditable_id', $this->selectedId)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public function render()
    {
        return view('livewire.logs.view-logs');
    }
}
