<?php

namespace App\Http\Livewire\Logs;

use App\Models\Styles;
use Livewire\Component;
use OwenIt\Auditing\Audit;
use Illuminate\Support\Collection;

class StyleLog extends Component
{
    public ?Styles $style = null;
    public Collection $logs;
    public $styleId = '';

    public function updatedStyleId($value)
    {
        $this->style = Styles::with(['style_versions.colourways', 'style_versions.prices'])->find($value);

        if ($this->style) {
            $this->logs = $this->getStyleAuditLog($this->style);
        } else {
            $this->logs = collect();
        }
    }

    public function getGroupedLogsProperty()
    {
        return $this->logs
            ->groupBy(function ($log) {
                return implode('|', [
                    $log->auditable_type,
                    $log->auditable_id,
                    $log->event,
                    $log->created_at->format('Y-m-d H:i'),
                    $log->user_id,
                ]);
            })
            ->map(function ($group) {
                return $group->flatMap(function ($log) {
                    $changes = $this->filterMeaningfulChanges($log->getModified());

                    return collect($changes)->map(function ($change, $field) use ($log) {
                        return [
                            'field'     => $field,
                            'old'       => $change['old'] ?? '',
                            'new'       => $change['new'] ?? '',
                            'timestamp' => $log->created_at->format('Y-m-d H:i:s'),
                            'user'      => $log->user->name ?? 'System',
                            'event'     => $log->event,
                            'type'      => class_basename($log->auditable_type),
                            'id'        => $log->auditable_id,
                        ];
                    })->values(); // prevent keyed results
                })->values(); // flatten to collection of change rows
            });
    }


    protected function filterMeaningfulChanges(array $modified): array
    {
        return collect($modified)->filter(function ($change) {
            $old = $this->normaliseAuditValue($change['old'] ?? []);
            $new = $this->normaliseAuditValue($change['new'] ?? []);
            return $old !== $new;
        })->all();
    }

    protected function normaliseAuditValue($value)
    {
        // Treat null, empty string, and false as the same if needed
        if (is_null($value) || $value === '' || $value === false || $value === 0 || $value === '0') {
            return '__empty__';
        }

        // Optionally normalise integers, booleans, etc. more here

        return $value;
    }

    protected function getStyleAuditLog(Styles $style)
    {
        $logs = collect();

        $logs = $logs->merge($style->audits);

        foreach ($style->style_versions as $version) {
            $logs = $logs->merge($version->audits);

            foreach ($version->colourways as $cw) {
                $logs = $logs->merge($cw->audits);
            }

            foreach ($version->prices as $price) {
                $logs = $logs->merge($price->audits);
            }
        }

        return $logs->sortByDesc('created_at');
    }

    public function render()
    {
        return view('livewire.logs.style-log');
    }
}
