<?php

namespace App\Http\Livewire\Logs;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\ShipmentLine;
use App\Models\CustomerOrders;
use App\Models\Styles;
use App\Models\Price;
use OwenIt\Auditing\Models\Audit;
use Illuminate\Support\Collection;

class PriceHistory extends Component
{
    use WithPagination;

    public string $searchType = 'style_code';
    public string $search = '';
    public bool $searched = false;
    public array $results = [];
    public ?int $selectedItemId = null;
    public string $selectedItemType = '';
    public Collection $priceAudits;

    protected $queryString = [
        'search' => ['except' => ''],
        'searchType' => ['except' => 'style_code'],
    ];

    public function mount()
    {
        $this->priceAudits = collect();
    }

    public function updatedSearch()
    {
        $this->resetPage();
        $this->searched = false;
        $this->results = [];
        $this->selectedItemId = null;
        $this->selectedItemType = '';
        $this->priceAudits = collect();
    }

    public function updatedSearchType()
    {
        $this->search = '';
        $this->searched = false;
        $this->results = [];
        $this->selectedItemId = null;
        $this->selectedItemType = '';
        $this->priceAudits = collect();
    }

    public function performSearch()
    {
        $this->validate([
            'search' => 'required|min:1',
        ]);

        $this->searched = true;
        $this->results = [];
        $this->selectedItemId = null;
        $this->selectedItemType = '';
        $this->priceAudits = collect();

        $searchTerm = trim($this->search);

        switch ($this->searchType) {
            case 'style_code':
                $this->searchByStyleCode($searchTerm);
                break;
            case 'order_number':
                $this->searchByOrderNumber($searchTerm);
                break;
            case 'shipment_line_id':
                $this->searchByShipmentLineId($searchTerm);
                break;
        }
    }

    protected function searchByStyleCode(string $searchTerm)
    {
        $styles = Styles::with([
            'customers:id,name',
            'designs:id,description',
            'style_versions' => function ($q) {
                $q->with(['prices:id,style_versions_id,quote,name,colourways_id', 'colourways:id,style_versions_id,name']);
            }
        ])
            ->where('customer_ref', 'like', "%{$searchTerm}%")
            ->orWhere('designs_id', 'like', "%{$searchTerm}%")
            ->limit(50)
            ->get();

        $this->results = $styles->map(function ($style) {
            return [
                'id' => $style->id,
                'type' => 'style',
                'label' => $style->customer_ref ?? $style->designs_id,
                'customer' => $style->customers->name ?? 'N/A',
                'design' => $style->designs->description ?? 'N/A',
                'prices_count' => $style->style_versions->flatMap->prices->count(),
            ];
        })->toArray();
    }

    protected function searchByOrderNumber(string $searchTerm)
    {
        $orders = CustomerOrders::with([
            'customers:id,name',
            'customer_order_lines' => function ($q) {
                $q->with([
                    'colourways:id,style_versions_id,name',
                    'colourways.style_versions:id,styles_id',
                    'colourways.style_versions.styles:id,customer_ref,designs_id',
                    'shipment_lines:id,customer_order_lines_id'
                ]);
            }
        ])
            ->withoutGlobalScopes()
            ->where('customer_po', 'like', "%{$searchTerm}%")
            ->orWhere('id', $searchTerm)
            ->limit(50)
            ->get();

        $this->results = $orders->map(function ($order) {
            $styleRefs = $order->customer_order_lines
                ->map(fn($col) => $col->colourways?->style_versions?->styles?->customer_ref)
                ->filter()
                ->unique()
                ->values()
                ->take(3)
                ->implode(', ');

            return [
                'id' => $order->id,
                'type' => 'order',
                'label' => $order->customer_po,
                'customer' => $order->customers->name ?? 'N/A',
                'order_date' => $order->order_date?->format('d-m-Y'),
                'styles' => $styleRefs ?: 'N/A',
                'lines_count' => $order->customer_order_lines->count(),
            ];
        })->toArray();
    }

    protected function searchByShipmentLineId(string $searchTerm)
    {
        $shipmentLines = ShipmentLine::with([
            'customer_order_lines' => function ($q) {
                $q->with([
                    'customer_orders:id,customer_po',
                    'customer_orders.customers:id,name',
                    'colourways:id,style_versions_id,name',
                    'colourways.style_versions:id,styles_id',
                    'colourways.style_versions.styles:id,customer_ref,designs_id',
                ]);
            },
            'shipments:id,transporter_invoice'
        ])
            ->where('id', $searchTerm)
            ->orWhere('rt_invoice', 'like', "%{$searchTerm}%")
            ->orWhere('factory_invoice', 'like', "%{$searchTerm}%")
            ->limit(50)
            ->get();

        $this->results = $shipmentLines->map(function ($line) {
            return [
                'id' => $line->id,
                'type' => 'shipment_line',
                'label' => "SL #{$line->id}",
                'order_po' => $line->customer_order_lines?->customer_orders?->customer_po ?? 'N/A',
                'customer' => $line->customer_order_lines?->customer_orders?->customers?->name ?? 'N/A',
                'style_ref' => $line->customer_order_lines?->colourways?->style_versions?->styles?->customer_ref ?? 'N/A',
                'colourway' => $line->customer_order_lines?->colourways?->name ?? 'N/A',
                'rt_invoice' => $line->rt_invoice,
            ];
        })->toArray();
    }

    public function selectItem(int $id, string $type)
    {
        $this->selectedItemId = $id;
        $this->selectedItemType = $type;
        $this->loadPriceAudits();
    }

    protected function loadPriceAudits()
    {
        $audits = collect();

        switch ($this->selectedItemType) {
            case 'style':
                $audits = $this->loadStylePriceAudits($this->selectedItemId);
                break;
            case 'order':
                $audits = $this->loadOrderPriceAudits($this->selectedItemId);
                break;
            case 'shipment_line':
                $audits = $this->loadShipmentLinePriceAudits($this->selectedItemId);
                break;
        }

        $this->priceAudits = $audits;
    }

    protected function loadStylePriceAudits(int $styleId): Collection
    {
        $style = Styles::with(['style_versions.prices'])->find($styleId);

        if (!$style) {
            return collect();
        }

        $priceIds = $style->style_versions->flatMap->prices->pluck('id')->toArray();

        return $this->fetchPriceAudits($priceIds);
    }

    protected function loadOrderPriceAudits(int $orderId): Collection
    {
        $order = CustomerOrders::with([
            'customer_order_lines.colourways.style_versions.prices'
        ])->find($orderId);

        if (!$order) {
            return collect();
        }

        $priceIds = $order->customer_order_lines
            ->map(fn($col) => $col->colourways?->style_versions?->prices)
            ->filter()
            ->flatten()
            ->pluck('id')
            ->unique()
            ->toArray();

        return $this->fetchPriceAudits($priceIds);
    }

    protected function loadShipmentLinePriceAudits(int $shipmentLineId): Collection
    {
        $shipmentLine = ShipmentLine::with([
            'customer_order_lines.colourways.style_versions.prices'
        ])->find($shipmentLineId);

        if (!$shipmentLine) {
            return collect();
        }

        $priceIds = $shipmentLine->customer_order_lines
            ?->colourways
            ?->style_versions
            ?->prices
            ?->pluck('id')
            ?->toArray() ?? [];

        return $this->fetchPriceAudits($priceIds);
    }

    protected function fetchPriceAudits(array $priceIds): Collection
    {
        if (empty($priceIds)) {
            return collect();
        }

        // Fetch audits for Price model filtered by price IDs
        $audits = Audit::with('user:id,name')
            ->where('auditable_type', 'App\Models\Price')
            ->whereIn('auditable_id', $priceIds)
            ->orderByDesc('created_at')
            ->limit(500)
            ->get();

        // Filter to show only price-related field changes
        $priceFields = [
            'quote', 'cmt', 'target', 'quote_status', 'cmt_status',
            'yarn', 'yarn_value', 'fabric', 'accessory', 'label',
            'gmt_trans', 'yarn_trans', 'embroidery', 'testing',
            'discount', 'duty', 'weight', 'costed_weight', 'aoc',
            'valid_until', 'min_qty'
        ];

        return $audits->map(function ($audit) use ($priceFields) {
            $changes = collect($audit->getModified())
                ->filter(fn($change, $field) => in_array($field, $priceFields));

            if ($changes->isEmpty()) {
                return null;
            }

            // Get price context
            $price = Price::withTrashed()
                ->with(['style_versions.styles:id,customer_ref', 'sizes:id,name'])
                ->find($audit->auditable_id);

            return [
                'id' => $audit->id,
                'price_id' => $audit->auditable_id,
                'style_ref' => $price?->style_versions?->styles?->customer_ref ?? 'N/A',
                'price_name' => $price?->name ?? 'N/A',
                'size' => $price?->sizes?->name ?? 'All',
                'event' => $audit->event,
                'user' => $audit->user?->name ?? 'System',
                'timestamp' => $audit->created_at->format('d-m-Y H:i:s'),
                'changes' => $changes->map(function ($change, $field) {
                    return [
                        'field' => $field,
                        'old' => $this->formatValue($change['old'] ?? null),
                        'new' => $this->formatValue($change['new'] ?? null),
                    ];
                })->values()->toArray(),
            ];
        })->filter()->values();
    }

    protected function formatValue($value): string
    {
        if (is_null($value) || $value === '') {
            return '-';
        }

        if (is_array($value)) {
            return json_encode($value);
        }

        if (is_numeric($value)) {
            return number_format((float)$value, 2);
        }

        return (string)$value;
    }

    public function clearSearch()
    {
        $this->search = '';
        $this->searched = false;
        $this->results = [];
        $this->selectedItemId = null;
        $this->selectedItemType = '';
        $this->priceAudits = collect();
    }

    public function render()
    {
        return view('livewire.logs.price-history');
    }
}

