<?php

namespace App\Http\Livewire\Logs;

use App\Models\CustomerOrders;
use Livewire\Component;
use OwenIt\Auditing\Audit;
use Illuminate\Support\Collection;

class OrderLogs extends Component
{
    public ?CustomerOrders $order = null;
    public Collection $logs;
    public $orderId = '';

    public function updatedOrderId($value)
    {
        $this->order = CustomerOrders::with([
            'customer_order_lines.customer_order_line_quantities',
            'customer_order_lines.shipment_lines.shipment_line_sizes',
        ])->find($value);

        if ($this->order) {
            $this->logs = $this->getOrderAuditLog($this->order);
        } else {
            $this->logs = collect();
        }
    }

    public function getGroupedLogsProperty()
    {
        return $this->logs
            ->groupBy(function ($log) {
                return implode('|', [
                    $log->auditable_type,
                    $log->auditable_id,
                    $log->event,
                    $log->created_at->format('Y-m-d H:i'),
                    $log->user_id,
                ]);
            })
            ->map(function ($group) {
                return $group->flatMap(function ($log) {
                    $changes = $this->filterMeaningfulChanges($log->getModified());

                    return collect($changes)->map(function ($change, $field) use ($log) {
                        return [
                            'field'     => $field,
                            'old'       => $change['old'] ?? '',
                            'new'       => $change['new'] ?? '',
                            'timestamp' => $log->created_at->format('Y-m-d H:i:s'),
                            'user'      => $log->user->name ?? 'System',
                            'event'     => $log->event,
                            'type'      => class_basename($log->auditable_type),
                            'id'        => $log->auditable_id,
                        ];
                    })->values();
                })->values();
            });
    }

    protected function filterMeaningfulChanges(array $modified): array
    {
        return collect($modified)->filter(function ($change, $field) {
            if (str_contains($field, '_cache')) {
                return false;
            }

            $old = $this->normaliseAuditValue($change['old'] ?? []);
            $new = $this->normaliseAuditValue($change['new'] ?? []);
            return $old !== $new;
        })->all();
    }

    protected function normaliseAuditValue($value)
    {
        if (is_null($value) || $value === '' || $value === false || $value === 0 || $value === '0') {
            return '__empty__';
        }

        return $value;
    }

    protected function getOrderAuditLog(CustomerOrders $order)
    {
        $logs = collect();

        $logs = $logs->merge($order->audits);

        foreach ($order->customer_order_lines as $line) {
            $logs = $logs->merge($line->audits);

            foreach ($line->customer_order_line_quantities as $qty) {
                $logs = $logs->merge($qty->audits);
            }

            foreach ($line->shipment_lines as $shipmentLine) {
                $logs = $logs->merge($shipmentLine->audits);

                foreach ($shipmentLine->shipment_line_sizes as $size) {
                    $logs = $logs->merge($size->audits);
                }
            }
        }

        return $logs->sortByDesc('created_at');
    }

    public function render()
    {
        return view('livewire.logs.order-logs');
    }
}
