<?php

namespace App\Http\Livewire\Imports;

use App\Models\Price;
use App\Models\Styles;
use App\Models\Sizes;
use App\Models\Seasons;
use App\Models\StyleVersions;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class PriceImport extends Component
{
    use WithFileUploads;

    public $csvFile;
    public $importData = [];
    public $processedData = [];
    public $selectedSeason;
    public $seasons = [];
    public $styles = [];
    public $sizes = [];
    public $showPreview = false;
    public $importComplete = false;

    protected $rules = [
        'csvFile' => 'required|file|mimes:csv,txt|max:10240', // 10MB max
        'selectedSeason' => 'required|exists:seasons,id'
    ];

    protected $messages = [
        'csvFile.required' => 'Please select a CSV file to upload.',
        'csvFile.file' => 'The uploaded file must be a valid file.',
        'csvFile.mimes' => 'The file must be a CSV or TXT file.',
        'csvFile.max' => 'The file size must not exceed 10MB.',
        'selectedSeason.required' => 'Please select a season.',
        'selectedSeason.exists' => 'The selected season is invalid.'
    ];

    public function mount()
    {
        $this->seasons = Seasons::orderBy('created_at', 'desc')->get();
        $this->sizes = Sizes::orderBy('order')->get();
    }

    public function updatedCsvFile()
    {
        $this->validate(['csvFile' => 'required|file|mimes:csv,txt|max:10240']);
        // Don't process automatically - wait for user to click "Process File" button
    }

    public function updatedSelectedSeason()
    {
        if ($this->selectedSeason) {
            $this->loadStylesForSeason();
        }
    }

    public function processCsvFile()
    {
        if (!$this->csvFile) {
            session()->flash('error', 'No file selected');
            return;
        }

        if (!$this->selectedSeason) {
            session()->flash('error', 'Please select a season first');
            return;
        }

        try {
            $path = $this->csvFile->store('temp');
            $content = Storage::get($path);
            
            // Parse comma-separated CSV
            $lines = explode("\n", trim($content));
            $this->importData = [];
            
            // Skip header row
            for ($i = 1; $i < count($lines); $i++) {
                $line = trim($lines[$i]);
                if (empty($line)) continue;
                
                // Split by comma (CSV format)
                $parts = str_getcsv($line);
                
                // Handle different column counts - some rows might have empty size
                if (count($parts) >= 4) {
                    $season = $parts[0];
                    $style = $parts[1];
                    
                    $size = '';
                    $cmt = '';
                    $quote = '';
                    
                    if (count($parts) == 4) {
                        // Format: Season,Style,CMT,Quote (no size)
                        $cmt = $parts[2];
                        $quote = $parts[3];
                    } elseif (count($parts) == 5) {
                        // Format: Season,Style,Size,CMT,Quote
                        $size = $parts[2];
                        $cmt = $parts[3];
                        $quote = $parts[4];
                    }
                    
                    $this->importData[] = [
                        'season' => $season,
                        'style' => $style,
                        'size' => $size,
                        'cmt' => $cmt,
                        'quote' => $quote,
                        'original_line' => $i + 1
                    ];
                }
            }
            
            if (empty($this->importData)) {
                session()->flash('error', 'No valid data found in CSV file');
                Storage::delete($path);
                return;
            }
            
            $this->processImportData();
            $this->showPreview = true;
            
            // Clean up temp file
            Storage::delete($path);
            
            session()->flash('success', 'File processed successfully. Found ' . count($this->importData) . ' rows.');
            
        } catch (\Exception $e) {
            session()->flash('error', 'Error processing CSV file: ' . $e->getMessage());
        }
    }

    public function processImportData()
    {
        $this->processedData = [];
        
        foreach ($this->importData as $row) {
            $processedRow = $row;
            
            // Find matching style in the selected season (MABLI only)
            $matchingStyle = null;
            if ($this->selectedSeason) {
                $matchingStyle = Styles::where('seasons_id', $this->selectedSeason)
                    ->where('customers_id', 9) // Only MABLI styles
                    ->where('customer_ref', $row['style'])
                    ->first();
            }
            
            $processedRow['matched_style'] = $matchingStyle;
            $processedRow['style_id'] = $matchingStyle ? $matchingStyle->id : null;
            $processedRow['has_match'] = $matchingStyle ? true : false;
            
            // Find matching size
            $matchingSize = null;
            if (!empty($row['size'])) {
                $matchingSize = Sizes::where('name', $row['size'])->first();
            }
            
            $processedRow['matched_size'] = $matchingSize;
            $processedRow['size_id'] = $matchingSize ? $matchingSize->id : null;
            
            $this->processedData[] = $processedRow;
        }
    }

    public function loadStylesForSeason()
    {
        if ($this->selectedSeason) {
            $this->styles = Styles::where('seasons_id', $this->selectedSeason)
                ->where('customers_id', 9) // Only MABLI styles
                ->with('style_versions')
                ->orderBy('customer_ref')
                ->get();
        }
    }

    public function updateStyleSelection($rowIndex, $styleId)
    {
        if (isset($this->processedData[$rowIndex])) {
            $this->processedData[$rowIndex]['style_id'] = $styleId;
            $this->processedData[$rowIndex]['has_match'] = true;
            $this->processedData[$rowIndex]['matched_style'] = Styles::find($styleId);
        }
    }

    public function savePrices()
    {
        $this->validate([
            'selectedSeason' => 'required|exists:seasons,id'
        ]);

        $savedCount = 0;
        $errors = [];

        DB::beginTransaction();
        
        try {
            foreach ($this->processedData as $row) {
                if (!$row['has_match'] || !$row['style_id']) {
                    $errors[] = "Row {$row['original_line']}: No style selected";
                    continue;
                }

                // Get the style and its first version
                $style = Styles::find($row['style_id']);
                if (!$style || !$style->style_versions->count()) {
                    $errors[] = "Row {$row['original_line']}: Style has no versions";
                    continue;
                }

                $styleVersion = $style->style_versions->first();

                // Create price record
                Price::create([
                    'style_versions_id' => $styleVersion->id,
                    'cmt' => (float) $row['cmt'],
                    'quote' => (float) $row['quote'],
                    'sizes_id' => $row['size_id'],
                    'cmt_status' => 'estimate',
                    'quote_status' => 'estimate',
                    'model' => 'manual'
                ]);

                $savedCount++;
            }

            DB::commit();
            
            if ($savedCount > 0) {
                session()->flash('success', "Successfully imported {$savedCount} price records.");
                $this->importComplete = true;
                $this->showPreview = false;
                $this->reset(['csvFile', 'processedData', 'importData']);
            }
            
            if (!empty($errors)) {
                session()->flash('warning', 'Some records could not be imported: ' . implode(', ', $errors));
            }

        } catch (\Exception $e) {
            DB::rollback();
            session()->flash('error', 'Error saving prices: ' . $e->getMessage());
        }
    }

    public function resetImport()
    {
        $this->reset(['csvFile', 'importData', 'processedData', 'showPreview', 'importComplete']);
        $this->selectedSeason = null;
    }

    public function debugInfo()
    {
        return [
            'csvFile' => $this->csvFile ? 'File uploaded' : 'No file',
            'selectedSeason' => $this->selectedSeason,
            'importDataCount' => count($this->importData),
            'processedDataCount' => count($this->processedData),
            'showPreview' => $this->showPreview
        ];
    }

    public function testCsvContent()
    {
        if (!$this->csvFile) {
            session()->flash('error', 'No file uploaded');
            return;
        }

        try {
            $path = $this->csvFile->store('temp');
            $content = Storage::get($path);
            
            $lines = explode("\n", trim($content));
            $debugInfo = "File content:\n";
            $debugInfo .= "Total lines: " . count($lines) . "\n\n";
            
            foreach ($lines as $i => $line) {
                $debugInfo .= "Line " . ($i + 1) . ": '" . $line . "'\n";
                $parts = str_getcsv(trim($line));
                $debugInfo .= "Parts (" . count($parts) . "): " . json_encode($parts) . "\n\n";
            }
            
            session()->flash('info', $debugInfo);
            Storage::delete($path);
            
        } catch (\Exception $e) {
            session()->flash('error', 'Error reading file: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.imports.price-import');
    }
}
