<?php

namespace App\Http\Livewire\Imports;

use Gemini;
use OpenAI;
use OpenAI\Client;
use Gemini\Data\Blob;
use Livewire\Component;
use Spatie\PdfToText\Pdf;
use Gemini\Enums\MimeType;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithFileUploads;
use App\Models\CustomerOrders;
use Illuminate\Support\Carbon;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Http;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ImportExcelWithValues;

/**
 * PackingListUploadAsosWaltz Component
 * 
 * This component handles the upload and processing of Waltz Fashion packing list PDFs
 * specifically for ASOS orders. It extracts the following information:
 * - Factory invoice number
 * - Export date (exfty)
 * - ASN numbers
 * - Fabric composition
 * - Size breakdowns with quantities
 * - Net and gross weights
 * - Carton counts
 * - PO numbers
 * 
 * The component updates the shipment line and related size information in the database.
 * Customer is hard-coded to ASOS (ID: 6) as this component is designed specifically
 * for ASOS order processing.
 */
class PackingListUploadAsosWaltz extends BaseComponent
{
    use WithFileUploads;
    
    // Constants
    const ASOS_CUSTOMER_ID = 6;
    
    public $packingList = '';
    public $answer = '';
    public $question = '';

    public $packingListFiles = [];
    
    // Hard-coded customer for ASOS - this component is specifically designed for ASOS orders
    public $customer = self::ASOS_CUSTOMER_ID;
    
    // Preview and confirmation properties
    public $previewData = [];
    public $showPreview = false;
    public $processingFiles = false;

    public function mount()
    {
        // Set default values for date range
        $this->startDate = Carbon::now()->subDays(3)->toDateString();
        $this->endDate = Carbon::now()->addDay()->toDateString();
        
        // Customer is hard-coded to ASOS (ID: 6) - no customer selection needed
    }

    /**
     * Upload and process packing list PDFs
     * 
     * This method processes each uploaded PDF file, extracts the required information,
     * and shows a preview table for user confirmation before saving.
     */
    public function uploadPackingList()
    {
        if (empty($this->packingListFiles)) {
            session()->flash('message', 'No files uploaded.');
            return;
        }
        elseif($this->buttonDisabled()){
            session()->flash('message', 'Too many files uploaded. Max 10.');
            return;
        }

        $this->processingFiles = true;
        $this->previewData = [];
        $errorCount = 0;
        
        // Loop through each uploaded file
        foreach ($this->packingListFiles as $file) {
            try {
                $extension = $file->getClientOriginalExtension();
                $fileContent = null;

                if ($extension === 'pdf') {
                    $fileContent = $this->convertPdfToText($file->getRealPath());
                } else {
                    session()->flash('message', 'Unsupported file type: ' . $file->getClientOriginalName());
                    continue;
                }
                
                $extracted = $this->extractSzies($fileContent, $file->getClientOriginalName());
                
                if (empty($extracted['po_number'])) {
                    session()->flash('message', 'Could not extract PO number from file: ' . $file->getClientOriginalName());
                    $errorCount++;
                    continue;
                }
                
                // Check if we have either ASN-specific data or fallback global data
                $hasValidData = false;
                
                if (!empty($extracted['asn_data'])) {
                    // Check if any ASN has valid data
                    foreach ($extracted['asn_data'] as $asnData) {
                        if (!empty($asnData['sizes']) || 
                            !empty($asnData['net_weight']) || 
                            !empty($asnData['gross_weight']) || 
                            !empty($asnData['carton_count'])) {
                            $hasValidData = true;
                            break;
                        }
                    }
                }
                
                // Fallback to old validation if no ASN data
                if (!$hasValidData && !empty($extracted['sizes'])) {
                    $hasValidData = true;
                }
                
                if (!$hasValidData) {
                    \Log::error('Could not extract valid packing list data from file: ' . $file->getClientOriginalName());
                    session()->flash('message', 'Could not extract valid packing list data from file: ' . $file->getClientOriginalName());
                    $errorCount++;
                    continue;
                }
                
                // Get database matches for preview
                $dbMatches = $this->getDatabaseMatches($extracted);
                $extracted['db_matches'] = $dbMatches;
                $extracted['filename'] = $file->getClientOriginalName();
                
                $this->previewData[] = $extracted;
                
            } catch (\Exception $e) {
                \Log::error('Error processing file: ' . $file->getClientOriginalName(), [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                session()->flash('message', 'Error processing file: ' . $file->getClientOriginalName() . ' - ' . $e->getMessage());
                session()->flash('alert-class', 'alert-danger');
                $errorCount++;
            }
        }
        
        $this->processingFiles = false;
        
        if (empty($this->previewData)) {
            session()->flash('message', 'No valid data could be extracted from the uploaded files.');
            session()->flash('alert-class', 'alert-danger');
            return;
        }
        
        // Show preview table
        $this->showPreview = true;
        session()->flash('message', 'Data extracted successfully. Please review and confirm the changes below.');
        session()->flash('alert-class', 'alert-info');
    }

    /**
     * Update database with extracted data
     * 
     * @param array $data Array containing extracted data from the PDF
     */
    private function updateDrop($data)
    {
        $customerPo = $data['po_number'];
        
        // Get the shipment line
        $shipmentLine = ShipmentLine::with('shipment_line_sizes')
                            ->whereRelation('customer_order_lines.customer_orders', 'customer_po', $customerPo)->get();
        if($shipmentLine->count() > 1){
            throw new \Exception("Too many shipment lines matched for PO: " . $customerPo);
        }
        if($shipmentLine->count() == 0){
            session()->flash('message', 'No shipment line found for PO: ' . $customerPo);
            return;
        }
        $shipmentLine = $shipmentLine->first();
        
        // Process each ASN's data
        $processedAsns = [];
        $totalNetWeight = 0;
        $totalGrossWeight = 0;
        $totalCartons = 0;
        $allSizes = [];
        
        if (!empty($data['asn_data'])) {
            foreach ($data['asn_data'] as $asnData) {
                $processedAsns[] = $asnData['asn'];
                
                // Accumulate totals for the shipment line
                if ($asnData['net_weight']) {
                    $totalNetWeight += $asnData['net_weight'];
                }
                if ($asnData['gross_weight']) {
                    $totalGrossWeight += $asnData['gross_weight'];
                }
                if ($asnData['carton_count']) {
                    $totalCartons += $asnData['carton_count'];
                }
                
                // Collect all sizes from all ASNs
                if (!empty($asnData['sizes'])) {
                    foreach ($asnData['sizes'] as $sizeData) {
                        $sizeKey = $sizeData['size'];
                        if (!isset($allSizes[$sizeKey])) {
                            $allSizes[$sizeKey] = 0;
                        }
                        $allSizes[$sizeKey] += $sizeData['quantity'];
                    }
                }
            }
        } else {
            // Fallback to old format if asn_data is not available
            $processedAsns = $data['asn_numbers'] ?? [];
            $totalNetWeight = $data['net_weight'] ?? 0;
            $totalGrossWeight = $data['gross_weight'] ?? 0;
            $totalCartons = $data['no_cartons'] ?? 0;
            
            if (!empty($data['sizes'])) {
                foreach ($data['sizes'] as $size) {
                    $sizeKey = $size['size'];
                    if (!isset($allSizes[$sizeKey])) {
                        $allSizes[$sizeKey] = 0;
                    }
                    $allSizes[$sizeKey] += $size['qty'];
                }
            }
        }
        
        // Update shipment line with aggregated data
        $updateData = [
            'net_weight' => $totalNetWeight > 0 ? $totalNetWeight : null,
            'gross_weight' => $totalGrossWeight > 0 ? $totalGrossWeight : null,
            'no_cartons' => $totalCartons > 0 ? $totalCartons : null,
            'complete' => 1,
        ];
        
        // Add other fields
        if (!empty($data['factory_invoice'])) {
            $updateData['factory_invoice'] = $data['factory_invoice'];
        }
        if (!empty($data['exfty'])) {
            $updateData['exfty'] = $data['exfty'];
        }
        
        // Store all ASN numbers as JSON
        if (!empty($processedAsns)) {
            $updateData['asn'] = json_encode($processedAsns);
        }
        
        $shipmentLine->update($updateData);
        
        // Update sizes in the database
        foreach ($allSizes as $sizeName => $totalQty) {
            // Find matching size in database using normalized size codes
            $sizeModel = $this->findSizeInDatabase($sizeName);
            
            if ($sizeModel) {
                // Find customer order line quantity with this size
                $customerOrderLineQuantity = $shipmentLine->customer_order_lines->customer_order_line_quantities
                    ->where('sizes_id', $sizeModel->id)->first();
                
                if ($customerOrderLineQuantity) {
                    $shipmentLineSize = $shipmentLine->shipment_line_sizes
                        ->where('sizes_id', $sizeModel->id)->first();
                    
                    if ($shipmentLineSize) {
                        $shipmentLineSize->update([
                            'shipped_qty' => $totalQty,
                        ]);
                    }
                }
            }
        }

        session()->flash('message', 'Complete! Updated ' . count($allSizes) . ' sizes for PO: ' . $customerPo . ' across ' . count($processedAsns) . ' ASN(s)');
		session()->flash('alert-class', 'alert-success');
    }

    /**
     * Extract information from PDF text using Gemini AI
     * 
     * @param string $text The text content extracted from the PDF
     * @param string $filename The original filename for fallback extraction
     * @return array Array containing extracted data with each ASN having its own packing list data
     */
    private function extractSzies($text, $filename){
        try {
            // Use Gemini for primary extraction
            $geminiResult = $this->extractWithGemini($text);
            
            if (!empty($geminiResult) && is_array($geminiResult)) {
                // Validate and clean the Gemini result
                $validatedData = $this->validateGeminiResult($geminiResult, $text);
                
                // Check if Gemini got the right size data by comparing with summary extraction
                $summarySizes = $this->extractSummarySizes($text);
                $geminiAccurate = true;
                
                if (!empty($summarySizes) && !empty($validatedData)) {
                    // Compare total quantities to detect if Gemini got wrong data
                    foreach ($validatedData as $asnData) {
                        if (!empty($asnData['sizes'])) {
                            $geminiTotal = array_sum(array_column($asnData['sizes'], 'quantity'));
                            $summaryTotal = array_sum(array_column($summarySizes, 'quantity'));
                            $difference = abs($geminiTotal - $summaryTotal);
                            $threshold = max($geminiTotal, $summaryTotal) * 0.2; // 20% threshold
                            
                            if ($difference > $threshold) {
                                $geminiAccurate = false;
                                break;
                            }
                        }
                    }
                }
                
                // If Gemini is accurate, use it; otherwise fall back to regex
                if ($geminiAccurate) {
                    // Extract ASN numbers from the validated data
                    $asn_numbers = array_column($validatedData, 'asn');
                    
                    // For backward compatibility, also return the old format
                    $globalData = $this->extractGlobalPackingListData($text);
                    
                    return [
                        'asn_data' => $validatedData,
                        'sizes' => $globalData['sizes'],
                        'net_weight' => $globalData['net_weight'],
                        'gross_weight' => $globalData['gross_weight'],
                        'no_cartons' => $globalData['carton_count'],
                        'po_number' => $validatedData[0]['po_number'] ?? null,
                        'po_date' => $validatedData[0]['date'] ?? null,
                        'factory_invoice' => $validatedData[0]['factory_invoice'] ?? null,
                        'exfty' => $validatedData[0]['date'] ?? null,
                        'asn_numbers' => $asn_numbers,
                        'compositions' => $globalData['compositions'],
                        'style_numbers' => $globalData['style_numbers'],
                        'colors' => $globalData['colors'],
                    ];
                }
            }
        } catch (\Exception $e) {
            \Log::error('Gemini extraction failed: ' . $e->getMessage());
        }
        
        // Fallback to regex method for accurate size extraction
        $regexResult = $this->extractWithRegex($text, $filename);
        
        // If regex also fails, create minimal fallback structure
        if (empty($regexResult['asn_data']) && empty($regexResult['sizes'])) {
            
            // Try to extract at least basic information
            $poMatch = preg_match('/\b(\d{10,12})\b/', $text, $poMatches);
            $poNumber = $poMatch ? $poMatches[1] : 'UNKNOWN';
            
            $asnMatch = preg_match('/\b(\d{14})\b/', $text, $asnMatches);
            $asnNumber = $asnMatch ? $asnMatches[1] : 'UNKNOWN';
            
            return [
                'asn_data' => [
                    [
                        'po_number' => $poNumber,
                        'asn' => $asnNumber,
                        'date' => date('d-m-Y'),
                        'factory_invoice' => 'EXTRACTION_FAILED',
                        'net_weight' => null,
                        'gross_weight' => null,
                        'carton_count' => null,
                        'sizes' => []
                    ]
                ],
                'sizes' => [],
                'net_weight' => null,
                'gross_weight' => null,
                'no_cartons' => null,
                'po_number' => $poNumber,
                'po_date' => date('d-m-Y'),
                'factory_invoice' => 'EXTRACTION_FAILED',
                'exfty' => date('d-m-Y'),
                'asn_numbers' => [$asnNumber],
                'compositions' => [],
                'style_numbers' => [],
                'colors' => [],
            ];
        }
        
        return $regexResult;
    }
    
    /**
     * Extract packing list data using Gemini AI
     * 
     * @param string $text PDF text content
     * @return array|null Extracted data or null if failed
     */
    private function extractWithGemini($text)
    {
        try {
            $yourApiKey = getenv('GEMINI_API_KEY');
            $client = Gemini::client($yourApiKey);

            $count = 0;
            $data = null;
            
            while(!$data && $count < 3){
                $result = $client
                    ->generativeModel(config('gemini.models.flash'))
                    ->generateContent([
                        'Extract packing list data from this PDF text. The PDF likely contains an order overview, packing list overview, and then individual packing list sections for each ASN.

IMPORTANT: Only extract data from the ACTUAL packing list sections for each ASN. Do NOT duplicate data from overview sections or order summaries.

CRITICAL FOR SIZES: Extract size quantities ONLY from the "SHIPMENT QTY" section or similar packing list quantity tables. Do NOT use quantities from charts, order summaries, or other sections as these are often incorrect or represent different data.

Look for sections that contain:
- Specific ASN numbers
- Individual carton counts, weights, and sizes for that ASN
- Factory invoice numbers specific to each ASN
- "SHIPMENT QTY" tables with accurate size breakdowns

Avoid extracting from:
- Order overview sections
- Packing list summaries  
- General order information
- Charts or diagrams (these often have wrong quantities)
- Any section that doesn\'t clearly show packing list quantities

Return ONLY a valid JSON array with this exact structure for each UNIQUE ASN (no duplicates):

[
  {
    "po_number": "PO number here",
    "asn": "ASN number here", 
    "date": "date in DD-MM-YYYY format",
    "factory_invoice": "factory invoice number",
    "net_weight": weight as number,
    "gross_weight": weight as number,
    "carton_count": count as number,
    "sizes": [
      {
        "size": "size code (S, M, L, XL, etc.)",
        "quantity": quantity as number
      }
    ]
  }
]

Extract ONLY the unique ASN packing list data. Be precise with numbers and dates. Return ONLY the JSON, no other text. Ensure all quantities are accurate and match the SHIPMENT QTY section exactly. Do not duplicate ASN data from different sections.',
                        $text
                    ]);

                $responseText = $result->text();
                
                // Clean the response to extract just the JSON (same as working component)
                $data = json_decode(str_replace(['```json', '```'], '', $responseText), 1);
                
                if (json_last_error() === JSON_ERROR_NONE && is_array($data)) {
                    // Remove duplicates based on ASN number
                    $data = $this->removeDuplicateAsns($data);
                    break;
                } else {
                    // Fallback: try to extract JSON array from response
                    $jsonMatch = preg_match('/\[.*\]/s', $responseText, $matches);
                    if ($jsonMatch) {
                        $jsonData = $matches[0];
                        $decoded = json_decode($jsonData, true);
                        
                        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                            // Remove duplicates based on ASN number
                            $data = $this->removeDuplicateAsns($decoded);
                            break;
                        }
                    }
                }
                
                $count++;
            }
            
            return $data;
            
        } catch (\Exception $e) {
            \Log::error('Gemini API error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Remove duplicate ASN entries from extracted data
     * 
     * @param array $data Extracted data array
     * @return array Data with duplicates removed
     */
    private function removeDuplicateAsns($data)
    {
        if (!is_array($data)) {
            return $data;
        }
        
        $uniqueAsns = [];
        $seenAsns = [];
        
        foreach ($data as $item) {
            if (!isset($item['asn'])) {
                continue;
            }
            
            $asn = $item['asn'];
            
            // If we haven't seen this ASN before, keep it
            if (!in_array($asn, $seenAsns)) {
                $uniqueAsns[] = $item;
                $seenAsns[] = $asn;
            }
        }
        
        return $uniqueAsns;
    }
    
    /**
     * Normalize size codes to handle variations like 2XL vs XXL
     * 
     * @param string $sizeCode Raw size code from document
     * @return string Normalized size code
     */
    private function normalizeSizeCode($sizeCode)
    {
        if (empty($sizeCode)) {
            return '';
        }
        
        // Convert to uppercase and trim
        $sizeCode = strtoupper(trim($sizeCode));
        
        // Size mapping for common variations
        $sizeMap = [
            // Extra Small variations
            'XS' => 'XS',
            'X-SMALL' => 'XS',
            'XSMALL' => 'XS',
            
            // Small
            'S' => 'S',
            'SMALL' => 'S',
            
            // Medium
            'M' => 'M',
            'MEDIUM' => 'M',
            'MED' => 'M',
            
            // Large
            'L' => 'L',
            'LARGE' => 'L',
            
            // Extra Large variations
            'XL' => 'XL',
            'X-LARGE' => 'XL',
            'XLARGE' => 'XL',
            
            // Double Extra Large variations
            'XXL' => 'XXL',
            '2XL' => 'XXL',
            '2X' => 'XXL',
            'X2L' => 'XXL',
            '2X-LARGE' => 'XXL',
            '2XLARGE' => 'XXL',
            
            // Triple Extra Large variations
            'XXXL' => 'XXXL',
            '3XL' => 'XXXL',
            '3X' => 'XXXL',
            'X3L' => 'XXXL',
            '3X-LARGE' => 'XXXL',
            '3XLARGE' => 'XXXL',
            
            // Quadruple Extra Large variations
            'XXXXL' => 'XXXXL',
            '4XL' => 'XXXXL',
            '4X' => 'XXXXL',
            'X4L' => 'XXXXL',
            '4X-LARGE' => 'XXXXL',
            '4XLARGE' => 'XXXXL',
            
            // Quintuple Extra Large variations
            'XXXXXL' => 'XXXXXL',
            '5XL' => 'XXXXXL',
            '5X' => 'XXXXXL',
            'X5L' => 'XXXXXL',
            '5X-LARGE' => 'XXXXXL',
            '5XLARGE' => 'XXXXXL',
        ];
        
        // Return mapped size or original if no mapping found
        return $sizeMap[$sizeCode] ?? $sizeCode;
    }
    
    /**
     * Find size in database using normalized size codes
     * 
     * @param string $sizeName Size name to find
     * @return \App\Models\Sizes|null Size model or null if not found
     */
    private function findSizeInDatabase($sizeName)
    {
        $normalizedSize = $this->normalizeSizeCode($sizeName);
        
        // First try exact match with normalized size
        $size = \App\Models\Sizes::where('name', $normalizedSize)->first();
        if ($size) {
            return $size;
        }
        
        // If not found, try fuzzy matching with common variations
        $fuzzyMatches = [
            'XXL' => ['2XL', '2X', 'X2L'],
            '2XL' => ['XXL', '2X', 'X2L'],
            'XXXL' => ['3XL', '3X', 'X3L'],
            '3XL' => ['XXXL', '3X', 'X3L'],
            'XXXXL' => ['4XL', '4X', 'X4L'],
            '4XL' => ['XXXXL', '4X', 'X4L'],
            'XXXXXL' => ['5XL', '5X', 'X5L'],
            '5XL' => ['XXXXXL', '5X', 'X5L'],
        ];
        
        foreach ($fuzzyMatches as $standardSize => $variations) {
            if (in_array($normalizedSize, $variations)) {
                $size = \App\Models\Sizes::where('name', $standardSize)->first();
                if ($size) {
                    return $size;
                }
            }
        }
        
        return null;
    }

    /**
     * Validate and clean Gemini extraction result
     * 
     * @param array $geminiResult Raw Gemini result
     * @param string $originalText Original PDF text for validation
     * @return array Validated and cleaned data
     */
    private function validateGeminiResult($geminiResult, $originalText)
    {
        $validated = [];
        
        foreach ($geminiResult as $item) {
            // Validate required fields
            if (empty($item['po_number']) || empty($item['asn'])) {
                continue;
            }
            
            // Clean and validate data
            $validatedItem = [
                'po_number' => (string)$item['po_number'],
                'asn' => (string)$item['asn'],
                'date' => $this->cleanDate($item['date'] ?? ''),
                'factory_invoice' => (string)($item['factory_invoice'] ?? ''),
                'net_weight' => $this->cleanNumber($item['net_weight']),
                'gross_weight' => $this->cleanNumber($item['gross_weight']),
                'carton_count' => $this->cleanNumber($item['carton_count']),
                'sizes' => $this->cleanSizes($item['sizes'] ?? [])
            ];
            
            // Extract summary sizes from PDF for validation
            $summarySizes = $this->extractSummarySizes($originalText);
            
            // Validate and potentially correct sizes using summary data
            if (!empty($validatedItem['sizes'])) {
                $correctedSizes = $this->validateAndCorrectSizes($validatedItem['sizes'], $summarySizes);
                $validatedItem['sizes'] = $correctedSizes;
            }
            
            // Additional validation
            if ($this->validateAsnData($validatedItem)) {
                // Additional SHIPMENT QTY validation if original text is available
                if (!empty($originalText)) {
                    $this->validateShipmentQtyExtraction($validatedItem, $originalText);
                }
                $validated[] = $validatedItem;
            }
        }
        
        return $validated;
    }
    
    /**
     * Validate ASN data for consistency and reasonableness
     * 
     * @param array $asnData ASN data to validate
     * @return bool True if data is valid
     */
    private function validateAsnData($asnData)
    {
        // Check if weights are reasonable (positive numbers)
        if ($asnData['net_weight'] !== null && $asnData['net_weight'] <= 0) {
            return false;
        }
        
        if ($asnData['gross_weight'] !== null && $asnData['gross_weight'] <= 0) {
            return false;
        }
        
        // Check if gross weight is greater than or equal to net weight
        if ($asnData['net_weight'] !== null && $asnData['gross_weight'] !== null) {
            if ($asnData['gross_weight'] < $asnData['net_weight']) {
                return false;
            }
        }
        
        // Check carton count
        if ($asnData['carton_count'] !== null && $asnData['carton_count'] <= 0) {
            return false;
        }
        
        // Validate sizes
        if (empty($asnData['sizes'])) {
            return false;
        }
        
        // Check if quantities are reasonable
        foreach ($asnData['sizes'] as $size) {
            if ($size['quantity'] <= 0) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Validate that sizes were extracted from SHIPMENT QTY sections
     * 
     * @param array $asnData ASN data to validate
     * @param string $originalText Original PDF text for validation
     * @return bool True if sizes appear to be from SHIPMENT QTY
     */
    private function validateShipmentQtyExtraction($asnData, $originalText)
    {
        if (empty($asnData['sizes'])) {
            return false;
        }
        
        // Look for SHIPMENT QTY patterns in the text
        $hasShipmentQtyPattern = false;
        $patterns = [
            '/SHIPMENT QTY/i',
            '/SHIPMENT QUANTITY/i',
            '/PACKING LIST QTY/i',
            '/PACKING QUANTITY/i',
            '/ASN.*QTY/i',
            '/QUANTITY.*SHIPMENT/i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $originalText)) {
                $hasShipmentQtyPattern = true;
                break;
            }
        }
        
        // Check if quantities look reasonable for individual ASN
        $totalQuantity = array_sum(array_column($asnData['sizes'], 'quantity'));
        $isReasonable = $totalQuantity > 0 && $totalQuantity <= 5000; // Reasonable range for single ASN
        
        return $hasShipmentQtyPattern && $isReasonable;
    }

    /**
     * Clean and validate a date string
     * 
     * @param mixed $date Date input
     * @return string|null Cleaned date or null
     */
    private function cleanDate($date)
    {
        if (empty($date)) return null;
        
        $date = trim((string)$date);
        
        // Try to parse various date formats
        $formats = ['d-m-Y', 'd-m-y', 'Y-m-d', 'd/m/Y', 'd/m/y'];
        
        foreach ($formats as $format) {
            try {
                $parsed = \Carbon\Carbon::createFromFormat($format, $date);
                return $parsed->format('d-m-Y');
            } catch (\Exception $e) {
                continue;
            }
        }
        
        return $date; // Return as-is if we can't parse it
    }
    
    /**
     * Clean and validate a number
     * 
     * @param mixed $number Number input
     * @return float|int|null Cleaned number or null
     */
    private function cleanNumber($number)
    {
        if (empty($number) && $number !== 0) return null;
        
        // Remove any non-numeric characters except decimal point
        $cleaned = preg_replace('/[^0-9.]/', '', (string)$number);
        
        if (is_numeric($cleaned)) {
            return strpos($cleaned, '.') !== false ? (float)$cleaned : (int)$cleaned;
        }
        
        return null;
    }
    
    /**
     * Clean and validate sizes array
     * 
     * @param array $sizes Raw sizes array
     * @return array Cleaned sizes array
     */
    private function cleanSizes($sizes)
    {
        if (!is_array($sizes)) return [];
        
        $validSizes = ['XS', 'S', 'M', 'L', 'XL', 'XXL', 'XXXL', 'XXXXL', 'XXXXXL'];
        $cleanedSizes = [];
        
        foreach ($sizes as $size) {
            if (!isset($size['size']) || !isset($size['quantity'])) {
                continue;
            }
            
            $sizeCode = $this->normalizeSizeCode($size['size']);
            $quantity = $this->cleanNumber($size['quantity']);
            
            // Check if normalized size is valid
            if (in_array($sizeCode, $validSizes) && $quantity !== null && $quantity > 0) {
                $cleanedSizes[] = [
                    'size' => $sizeCode,
                    'quantity' => $quantity
                ];
            }
        }
        
        return $cleanedSizes;
    }
    
    /**
     * Fallback extraction using regex (old method)
     * 
     * @param string $text PDF text content
     * @param string $filename Original filename
     * @return array Extracted data using regex
     */
    private function extractWithRegex($text, $filename)
    {
        // Extract global invoice information
        preg_match('/6\. INVOICE NO:\s*([^\n]+)/', $text, $invoiceMatch);
        $factory_invoice = trim($invoiceMatch[1] ?? '');
        
        // Alternative invoice extraction patterns
        if (empty($factory_invoice)) {
            if (preg_match('/WFL\/\d+\/\d{4}/', $text, $wflMatch)) {
                $factory_invoice = $wflMatch[0];
            }
        }
        
        // Extract global date
        preg_match('/6\. INVOICE NO:[^\n]*\n[^\n]*DATE:\s*(\d{2}-[A-Za-z]+-\d{2})/', $text, $dateMatch);
        $exfty = $dateMatch[1] ?? null;
        
        if (empty($exfty)) {
            if (preg_match('/DATE:\s*(\d{2}-[A-Za-z]+-\d{2})/', $text, $altDateMatch)) {
                $exfty = $altDateMatch[1];
            }
        }
        
        if (empty($exfty)) {
            if (preg_match('/\b(\d{2}-[A-Za-z]+-\d{2})\b/', $text, $fallbackDateMatch)) {
                $exfty = $fallbackDateMatch[1];
            }
        }
        
        // Extract PO number
        preg_match_all('/ASOS PO NO:\s*\n(\d+)/', $text, $poMatches);
        $po_numbers = $poMatches[1] ?? [];
        
        if (empty($po_numbers)) {
            if (preg_match_all('/PO Number:\s*(\d+)/', $text, $altPoMatches)) {
                $po_numbers = $altPoMatches[1] ?? [];
            }
        }
        
        if (empty($po_numbers)) {
            if (preg_match_all('/\b(\d{12})\b/', $text, $digitMatches)) {
                $po_numbers = $digitMatches[1] ?? [];
            }
        }
        
        // Get PO number from filename or text
        $po_number = null;
        if (preg_match('/PO-(\d{10,12})/', $filename, $match)) {
            $po_number = $match[1];
        } elseif (!empty($po_numbers)) {
            $po_number = $po_numbers[0];
        }
        
        // Extract ASN numbers
        $asn_numbers = [];
        if (preg_match_all('/ASN NO:\s*\n(\d+)/', $text, $asnMatches)) {
            $asn_numbers = $asnMatches[1] ?? [];
        }
        
        if (empty($asn_numbers)) {
            if (preg_match_all('/ASN Number:\s*(\d+)/', $text, $altAsnMatches)) {
                $asn_numbers = $altAsnMatches[1] ?? [];
            }
        }
        
        if (empty($asn_numbers)) {
            if (preg_match_all('/\b(\d{14})\b/', $text, $digitMatches)) {
                $asn_numbers = $digitMatches[1] ?? [];
            }
        }
        
        // Create minimal ASN data structure for fallback
        $asn_data = [];
        
        // Try to get ASN data with actual size information
        $asnDataWithSizes = $this->extractAsnDataWithRegex($text);
        if (!empty($asnDataWithSizes)) {
            $asn_data = $asnDataWithSizes;
        } else {
            // Fallback to basic ASN structure if detailed extraction fails
            foreach ($asn_numbers as $asnNumber) {
                $asn_data[] = [
                    'po_number' => $po_number,
                    'asn' => $asnNumber,
                    'date' => $exfty,
                    'factory_invoice' => $factory_invoice,
                    'net_weight' => null,
                    'gross_weight' => null,
                    'carton_count' => null,
                    'sizes' => []
                ];
            }
        }
        
        // Extract summary sizes as fallback for global data
        $summarySizes = $this->extractSummarySizes($text);
        
        // For backward compatibility, also return the old format
        $globalData = $this->extractGlobalPackingListData($text);
        
        // Use summary sizes if available, otherwise use global data
        $finalSizes = !empty($summarySizes) ? $summarySizes : $globalData['sizes'];
        
        return [
            'asn_data' => $asn_data,
            'sizes' => $finalSizes,
            'net_weight' => $globalData['net_weight'],
            'gross_weight' => $globalData['gross_weight'],
            'no_cartons' => $globalData['carton_count'],
            'po_number' => $po_number,
            'po_date' => $exfty,
            'factory_invoice' => $factory_invoice,
            'exfty' => $exfty,
            'asn_numbers' => $asn_numbers,
            'compositions' => $globalData['compositions'],
            'style_numbers' => $globalData['style_numbers'],
            'colors' => $globalData['colors'],
        ];
    }
    
    /**
     * Extract ASN data using regex as alternative to Gemini
     * Focuses on summary sections for accurate quantities
     * 
     * @param string $text PDF text content
     * @return array Extracted ASN data
     */
    private function extractAsnDataWithRegex($text)
    {
        $asnData = [];
        
        // Look for ASN sections with summary data
        $asnPatterns = [
            // Pattern 1: ASN with summary quantities
            '/ASN[:\s]*(\d+).*?SHIPMENT QTY.*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/is',
            
            // Pattern 2: ASN with total quantities
            '/ASN[:\s]*(\d+).*?TOTAL.*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/is',
            
            // Pattern 3: ASN with summary section
            '/ASN[:\s]*(\d+).*?SUMMARY.*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/is',
            
            // Pattern 4: Extended sizes with ASN
            '/ASN[:\s]*(\d+).*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+).*?XXL\s+(\d+)/is'
        ];
        
        foreach ($asnPatterns as $pattern) {
            if (preg_match_all($pattern, $text, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    $asnNumber = $match[1];
                    
                    // Check if we already have this ASN
                    $existingIndex = array_search($asnNumber, array_column($asnData, 'asn'));
                    if ($existingIndex !== false) {
                        continue; // Skip duplicates
                    }
                    
                    // Extract sizes based on pattern
                    if (count($match) >= 6) { // Pattern 4 with XXL
                        $sizes = [
                            ['size' => 'S', 'quantity' => (int)$match[2]],
                            ['size' => 'M', 'quantity' => (int)$match[3]],
                            ['size' => 'L', 'quantity' => (int)$match[4]],
                            ['size' => 'XL', 'quantity' => (int)$match[5]],
                            ['size' => 'XXL', 'quantity' => (int)$match[6]]
                        ];
                    } else { // Standard 4 sizes
                        $sizes = [
                            ['size' => 'S', 'quantity' => (int)$match[2]],
                            ['size' => 'M', 'quantity' => (int)$match[3]],
                            ['size' => 'L', 'quantity' => (int)$match[4]],
                            ['size' => 'XL', 'quantity' => (int)$match[5]]
                        ];
                    }
                    
                    // Try to extract other data for this ASN
                    $asnSection = $this->extractAsnSectionData($text, $asnNumber);
                    
                    $asnData[] = [
                        'po_number' => $asnSection['po_number'] ?? 'Unknown',
                        'asn' => $asnNumber,
                        'date' => $asnSection['date'] ?? date('d-m-Y'),
                        'factory_invoice' => $asnSection['factory_invoice'] ?? 'Unknown',
                        'net_weight' => $asnSection['net_weight'] ?? null,
                        'gross_weight' => $asnSection['gross_weight'] ?? null,
                        'carton_count' => $asnSection['carton_count'] ?? null,
                        'sizes' => $sizes
                    ];
                }
                
                if (!empty($asnData)) {
                    break; // Use first successful pattern
                }
            }
        }
        
        return $asnData;
    }
    
    /**
     * Extract additional data for a specific ASN section
     * 
     * @param string $text PDF text content
     * @param string $asnNumber ASN number to search for
     * @return array Additional ASN data
     */
    private function extractAsnSectionData($text, $asnNumber)
    {
        $data = [];
        
        // Look for ASN section
        $asnSectionPattern = '/ASN[:\s]*' . preg_quote($asnNumber, '/') . '(.*?)(?=ASN|$)/is';
        if (preg_match($asnSectionPattern, $text, $asnMatch)) {
            $asnSection = $asnMatch[1];
            
            // Extract PO number
            if (preg_match('/PO[:\s]*(\d+)/i', $asnSection, $poMatch)) {
                $data['po_number'] = $poMatch[1];
            }
            
            // Extract date
            if (preg_match('/(\d{1,2}[-\/]\d{1,2}[-\/]\d{2,4})/', $asnSection, $dateMatch)) {
                $data['date'] = $dateMatch[1];
            }
            
            // Extract factory invoice
            if (preg_match('/INVOICE[:\s]*([A-Z0-9\/]+)/i', $asnSection, $invoiceMatch)) {
                $data['factory_invoice'] = $invoiceMatch[1];
            }
            
            // Extract weights
            if (preg_match('/NET[:\s]*([\d\.]+)/i', $asnSection, $netMatch)) {
                $data['net_weight'] = (float)$netMatch[1];
            }
            if (preg_match('/GROSS[:\s]*([\d\.]+)/i', $asnSection, $grossMatch)) {
                $data['gross_weight'] = (float)$grossMatch[1];
            }
            
            // Extract carton count
            if (preg_match('/CTN[:\s]*(\d+)/i', $asnSection, $ctnMatch)) {
                $data['carton_count'] = (int)$ctnMatch[1];
            }
        }
        
        return $data;
    }

    /**
     * Extract global packing list data as fallback
     * 
     * @param string $text PDF text content
     * @return array Global packing list data
     */
    private function extractGlobalPackingListData($text)
    {
        // Extract global weights and carton information
        preg_match('/Total PCS\s*(\d+)/', $text, $totalPcsMatch);
        preg_match('/Total CTN\s*(\d+)/', $text, $totalCtnMatch);
        preg_match('/Net Weight\s*([\d\.]+)/', $text, $netWeightMatch);
        preg_match('/Gross Weight\s*([\d\.]+)/', $text, $grossWeightMatch);
        
        // Alternative patterns
        if (empty($netWeightMatch[1])) {
            if (preg_match('/Net Weight\s*\n([\d\.]+)/', $text, $altNetMatch)) {
                $netWeightMatch[1] = $altNetMatch[1];
            }
        }
        
        if (empty($grossWeightMatch[1])) {
            if (preg_match('/Gross Weight\s*\n([\d\.]+)/', $text, $altGrossMatch)) {
                $grossWeightMatch[1] = $altGrossMatch[1];
            }
        }
        
        if (empty($totalCtnMatch[1])) {
            if (preg_match('/Total CTN\s*\n(\d+)/', $text, $altCtnMatch)) {
                $totalCtnMatch[1] = $altCtnMatch[1];
            }
        }
        
        // Extract global sizes
        $sizes = [];
        
        // Look for SHIPMENT QTY pattern
        if (preg_match('/SHIPMENT QTY\..*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/s', $text, $shipmentMatch)) {
            $sizes = [
                ['sku' => '', 'size' => 'S', 'qty' => (int)$shipmentMatch[1], 'style' => '', 'composition' => '', 'color' => ''],
                ['sku' => '', 'size' => 'M', 'qty' => (int)$shipmentMatch[2], 'style' => '', 'composition' => '', 'color' => ''],
                ['sku' => '', 'size' => 'L', 'qty' => (int)$shipmentMatch[3], 'style' => '', 'composition' => '', 'color' => ''],
                ['sku' => '', 'size' => 'XL', 'qty' => (int)$shipmentMatch[4], 'style' => '', 'composition' => '', 'color' => ''],
            ];
        }
        
        // Extract other global data
        preg_match_all('/Fabric Compositon\s*\n([^\n]+)/', $text, $compositionMatches);
        $compositions = $compositionMatches[1] ?? [];
        
        preg_match_all('/STYLE No\.\s*\n([^\n]+)/', $text, $styleMatches);
        $style_numbers = $styleMatches[1] ?? [];
        
        preg_match_all('/COLOR\s*\n([^\n]+)/', $text, $colorMatches);
        $colors = $colorMatches[1] ?? [];
        
        return [
            'net_weight' => $netWeightMatch[1] ?? null,
            'gross_weight' => $grossWeightMatch[1] ?? null,
            'carton_count' => $totalCtnMatch[1] ?? null,
            'sizes' => $sizes,
            'compositions' => $compositions,
            'style_numbers' => $style_numbers,
            'colors' => $colors
        ];
    }
    
    /**
     * Extract size quantities from summary section at bottom of PDF
     * This gets the actual totals, not per-carton values
     * 
     * @param string $text PDF text content
     * @return array Size quantities from summary section
     */
    private function extractSummarySizes($text)
    {
        $summarySizes = [];
        
        // Look for summary section patterns (usually at bottom of PDF)
        $patterns = [
            // Pattern 1: SHIPMENT QTY summary
            '/SHIPMENT QTY.*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/is',
            
            // Pattern 2: Total quantities
            '/TOTAL.*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/is',
            
            // Pattern 3: Summary section
            '/SUMMARY.*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/is',
            
            // Pattern 4: Bottom section quantities
            '/BOTTOM.*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/is',
            
            // Pattern 5: Final totals
            '/FINAL.*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/is',
            
            // Pattern 6: Extended sizes
            '/S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+).*?XXL\s+(\d+)/is',
            
            // Pattern 7: Very specific SHIPMENT QTY format
            '/SHIPMENT QTY\s*\n.*?S\s+(\d+).*?M\s+(\d+).*?L\s+(\d+).*?XL\s+(\d+)/is'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                // Extract sizes based on pattern
                if (count($matches) >= 6) { // Pattern 6 with XXL (6 matches: full match + 5 captures)
                    $summarySizes = [
                        ['size' => 'S', 'quantity' => (int)($matches[1] ?? 0)],
                        ['size' => 'M', 'quantity' => (int)($matches[2] ?? 0)],
                        ['size' => 'L', 'quantity' => (int)($matches[3] ?? 0)],
                        ['size' => 'XL', 'quantity' => (int)($matches[4] ?? 0)],
                        ['size' => 'XXL', 'quantity' => (int)($matches[5] ?? 0)]
                    ];
                } elseif (count($matches) >= 5) { // Standard 4 sizes (5 matches: full match + 4 captures)
                    $summarySizes = [
                        ['size' => 'S', 'quantity' => (int)($matches[1] ?? 0)],
                        ['size' => 'M', 'quantity' => (int)($matches[2] ?? 0)],
                        ['size' => 'L', 'quantity' => (int)($matches[3] ?? 0)],
                        ['size' => 'XL', 'quantity' => (int)($matches[4] ?? 0)]
                    ];
                }
                
                // Validate that we have valid quantities before proceeding
                if (!empty($summarySizes)) {
                    $hasValidQuantities = false;
                    foreach ($summarySizes as $size) {
                        if ($size['quantity'] > 0) {
                            $hasValidQuantities = true;
                            break;
                        }
                    }
                    
                    if ($hasValidQuantities) {
                        break;
                    } else {
                        $summarySizes = []; // Reset for next pattern
                    }
                }
            }
        }
        
        return $summarySizes;
    }
    
    /**
     * Validate and correct Gemini size extraction using summary data
     * 
     * @param array $geminiSizes Sizes extracted by Gemini
     * @param array $summarySizes Sizes from summary section
     * @return array Corrected sizes
     */
    private function validateAndCorrectSizes($geminiSizes, $summarySizes)
    {
        if (empty($summarySizes)) {
            return $geminiSizes;
        }
        
        if (empty($geminiSizes)) {
            return $summarySizes;
        }
        
        // Compare total quantities
        $geminiTotal = array_sum(array_column($geminiSizes, 'quantity'));
        $summaryTotal = array_sum(array_column($summarySizes, 'quantity'));
        
        // If there's a significant difference, use summary data
        $difference = abs($geminiTotal - $summaryTotal);
        $threshold = max($geminiTotal, $summaryTotal) * 0.1; // 10% threshold
        
        if ($difference > $threshold) {
            return $summarySizes;
        }
        
        return $geminiSizes;
    }

    /**
     * Convert PDF file to text
     * 
     * @param string $filePath Path to the PDF file
     * @return string Extracted text content from the PDF
     */
    private function convertPdfToText($filePath)
    {
        // Convert PDF to plain text
        return Pdf::getText($filePath);
    }

    /**
     * Check if upload button should be disabled
     * 
     * @return bool True if button should be disabled (too many files)
     */
    private function buttonDisabled()
    {
        return count($this->packingListFiles) > 30;
    }

    /**
     * Get database matches for preview
     * 
     * @param array $extracted Extracted data from PDF
     * @return array Database matches and current values
     */
    private function getDatabaseMatches($extracted)
    {
        // Safety check for required fields
        if (empty($extracted) || !is_array($extracted)) {
            return [
                'found' => false,
                'message' => 'Invalid extracted data structure'
            ];
        }
        
        if (empty($extracted['po_number'])) {
            return [
                'found' => false,
                'message' => 'No PO number found in extracted data'
            ];
        }
        
        $customerPo = $extracted['po_number'];
        
        $shipmentLine = ShipmentLine::with(['shipment_line_sizes', 'customer_order_lines.customer_order_line_quantities.sizes'])
                            ->whereRelation('customer_order_lines.customer_orders', 'customer_po', $customerPo)
                            ->first();
        
        if (!$shipmentLine) {
            return [
                'found' => false,
                'message' => 'No shipment line found for PO: ' . $customerPo
            ];
        }
        
        // Parse ASN numbers from database (handle both string and JSON formats)
        $currentAsns = [];
        if ($shipmentLine->asn) {
            if (is_string($shipmentLine->asn)) {
                // Try to decode JSON first
                $decoded = json_decode($shipmentLine->asn, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $currentAsns = $decoded;
                } else {
                    // If not JSON, treat as single ASN
                    $currentAsns = [$shipmentLine->asn];
                }
            } elseif (is_array($shipmentLine->asn)) {
                $currentAsns = $shipmentLine->asn;
            }
        }
        
        $currentData = [
            'net_weight' => $shipmentLine->net_weight,
            'gross_weight' => $shipmentLine->gross_weight,
            'no_cartons' => $shipmentLine->no_cartons,
            'exfty' => $shipmentLine->exfty,
            'factory_invoice' => $shipmentLine->factory_invoice,
            'asn' => $currentAsns,
            'complete' => $shipmentLine->complete,
        ];
        
        $sizeMatches = [];
        
        // Safety check for sizes array
        if (!empty($extracted['sizes']) && is_array($extracted['sizes'])) {
            foreach ($extracted['sizes'] as $size) {
                // Safety check for size structure
                if (empty($size) || !is_array($size) || !isset($size['size']) || !isset($size['quantity'])) {
                    continue;
                }
                
                $sizeMatch = [
                    'size' => $size['size'],
                    'extracted_qty' => $size['quantity'],
                    'current_qty' => 0,
                    'found' => false,
                    'sku' => '',
                    'composition' => '',
                ];
                
                // Find matching size in database
                if (!empty($size['sku'])) {
                    $customerOrderLineQuantity = $shipmentLine->customer_order_lines->customer_order_line_quantities
                        ->where('SKU', $size['size'])->first();
                } else {
                    // Try to find by size name using normalized size codes
                    $sizeModel = $this->findSizeInDatabase($size['size']);
                    if ($sizeModel) {
                        $customerOrderLineQuantity = $shipmentLine->customer_order_lines->customer_order_line_quantities
                            ->where('sizes_id', $sizeModel->id)->first();
                    } else {
                        // Fallback to old method
                        $customerOrderLineQuantity = $shipmentLine->customer_order_lines->customer_order_line_quantities
                            ->where('sizes.name', $size['size'])->first();
                    }
                }
                
                if ($customerOrderLineQuantity) {
                    $sizeMatch['found'] = true;
                    $shipmentLineSize = $shipmentLine->shipment_line_sizes
                        ->where('sizes_id', $customerOrderLineQuantity->sizes_id)->first();
                    if ($shipmentLineSize) {
                        $sizeMatch['current_qty'] = $shipmentLineSize->shipped_qty;
                    }
                }
                
                $sizeMatches[] = $sizeMatch;
            }
        }
        
        return [
            'found' => true,
            'shipment_line_id' => $shipmentLine->id,
            'current_data' => $currentData,
            'size_matches' => $sizeMatches,
            'customer_po' => $customerPo
        ];
    }

    /**
     * Check if there are any changes to show
     * 
     * @param array $data Preview data
     * @return bool True if there are changes
     */
    public function hasChanges($data)
    {
        if (!$data['db_matches']['found']) {
            return false;
        }
        
        $current = $data['db_matches']['current_data'];
        
        // Check ASN changes - compare arrays
        $currentAsns = $current['asn'] ?? [];
        $newAsns = $data['asn_numbers'] ?? [];
        
        // Sort both arrays for comparison
        sort($currentAsns);
        sort($newAsns);
        
        if ($currentAsns != $newAsns) {
            return true;
        }
        
        // Check changes in ASN-specific data
        if (!empty($data['asn_data'])) {
            foreach ($data['asn_data'] as $asnData) {
                // Check if any ASN has different data than current
                if ($current['net_weight'] != $asnData['net_weight'] ||
                    $current['gross_weight'] != $asnData['gross_weight'] ||
                    $current['no_cartons'] != $asnData['carton_count'] ||
                    $current['exfty'] != $asnData['date'] ||
                    $current['factory_invoice'] != $asnData['factory_invoice']) {
                    return true;
                }
                
                // Check size changes for this ASN
                if (!empty($asnData['sizes'])) {
                    foreach ($asnData['sizes'] as $sizeData) {
                        $sizeMatch = collect($data['db_matches']['size_matches'] ?? [])->firstWhere('size', $sizeData['size']);
                        if ($sizeMatch && $sizeMatch['found'] && $sizeMatch['current_qty'] != $sizeData['quantity']) {
                            return true;
                        }
                    }
                }
            }
        } else {
            // Fallback to old format checking
            if ($current['net_weight'] != $data['net_weight'] ||
                $current['gross_weight'] != $data['gross_weight'] ||
                $current['no_cartons'] != $data['no_cartons'] ||
                $current['exfty'] != $data['exfty'] ||
                $current['factory_invoice'] != $data['factory_invoice']) {
                return true;
            }
            
            // Check size changes
            foreach ($data['db_matches']['size_matches'] as $sizeMatch) {
                if ($sizeMatch['found'] && $sizeMatch['current_qty'] != $sizeMatch['extracted_qty']) {
                    return true;
                }
            }
        }
        
        return false;
    }

    /**
     * Format current value for display
     * 
     * @param mixed $value Current value from database
     * @return string Formatted value for display
     */
    public function formatCurrentValue($value)
    {
        if ($value === null || $value === '') {
            return '<span class="text-muted">Not set</span>';
        }
        
        if (is_bool($value)) {
            return $value ? 'Yes' : 'No';
        }
        
        if ($value instanceof \Carbon\Carbon) {
            return $value->format('Y-m-d');
        }
        
        return (string) $value;
    }
    
    /**
     * Format ASN numbers for display
     * 
     * @param array $asns Array of ASN numbers
     * @return string Formatted ASN display
     */
    public function formatAsnNumbers($asns)
    {
        if (empty($asns)) {
            return '<span class="text-muted">Not set</span>';
        }
        
        if (is_array($asns)) {
            $formatted = [];
            foreach ($asns as $asn) {
                $formatted[] = '<code>' . $asn . '</code>';
            }
            return implode('<br>', $formatted);
        }
        
        return '<code>' . $asns . '</code>';
    }
    
    /**
     * Get change indicator class
     * 
     * @param mixed $current Current value from database
     * @param mixed $new New value from PDF
     * @return string CSS class for change indicator
     */
    public function getChangeClass($current, $new)
    {
        if ($current === null || $current === '') {
            return 'table-success'; // New value
        }
        
        if ($current != $new) {
            return 'table-warning'; // Changed value
        }
        
        return ''; // No change
    }
    
    /**
     * Get ASN change indicator class
     * 
     * @param array $current Current ASN array from database
     * @param array $new New ASN array from PDF
     * @return string CSS class for change indicator
     */
    public function getAsnChangeClass($current, $new)
    {
        if (empty($current)) {
            return 'table-success'; // New value
        }
        
        // Sort both arrays for comparison
        $currentSorted = $current;
        $newSorted = $new;
        sort($currentSorted);
        sort($newSorted);
        
        if ($currentSorted != $newSorted) {
            return 'table-warning'; // Changed value
        }
        
        return ''; // No change
    }

    /**
     * Confirm and save the changes
     */
    public function confirmAndSave()
    {
        $processedCount = 0;
        $errorCount = 0;
        
        foreach ($this->previewData as $data) {
            try {
                $this->updateDrop($data);
                $processedCount++;
            } catch (\Exception $e) {
                \Log::error('Error saving data for PO: ' . $data['po_number'], [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                $errorCount++;
            }
        }
        
        // Reset preview state
        $this->showPreview = false;
        $this->previewData = [];
        $this->packingListFiles = [];
        
        // Show summary message
        if ($processedCount > 0) {
            session()->flash('message', 'Successfully processed ' . $processedCount . ' file(s)' . ($errorCount > 0 ? ' with ' . $errorCount . ' error(s)' : ''));
            session()->flash('alert-class', $errorCount > 0 ? 'alert-warning' : 'alert-success');
        } else {
            session()->flash('message', 'No files were processed successfully.');
            session()->flash('alert-class', 'alert-danger');
        }
    }
    
    /**
     * Cancel the preview and reset the form
     */
    public function cancelPreview()
    {
        $this->showPreview = false;
        $this->previewData = [];
        $this->packingListFiles = [];
        session()->flash('message', 'Upload cancelled.');
        session()->flash('alert-class', 'alert-info');
    }

    public function render()
    {
        Gate::authorize('order:update');
        if($this->buttonDisabled()){
            session()->flash('message', 'Max 30 files.');
            session()->flash('alert-class', 'alert-warning');
        }
        
        return view('livewire.imports.packing-list-upload-asos-waltz', [
            'previewData' => $this->previewData,
            'showPreview' => $this->showPreview,
            'processingFiles' => $this->processingFiles,
        ]);
    }
}
