<?php

namespace App\Http\Livewire\Imports;

use Gemini;
use OpenAI;
use OpenAI\Client;
use Gemini\Data\Blob;
use Livewire\Component;
use App\Models\Customer;
use Spatie\PdfToText\Pdf;
use Gemini\Enums\MimeType;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithFileUploads;
use App\Models\CustomerOrders;
use Illuminate\Support\Carbon;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ImportExcelWithValues;

class PackingListUploadAsos extends BaseComponent
{
    use WithFileUploads;
    public $packingList = '';
    public $answer = '';
    public $question = '';

    public $packingListFiles = [];

    public $customer = 6;

	#[Computed]
	public function customers(){
		return Customer::allCached();
	}

    public function mount()
    {
        // Set default values for date range
        $this->startDate = Carbon::now()->subDays(3)->toDateString();
        $this->endDate = Carbon::now()->addDay()->toDateString();
    }

    public function uploadPackingList()
    {
        if (empty($this->packingListFiles)) {
            session()->flash('message', 'No files uploaded.');
            return;
        }
        elseif($this->buttonDisabled()){
            session()->flash('message', 'Too many files uploaded. Max 10.');
            return;
        }
        elseif(!$this->customer){
            session()->flash('message', 'Select a customer.');
            return;
        }

        // Loop through each uploaded file
        foreach ($this->packingListFiles as $file) {
            $extension = $file->getClientOriginalExtension();
            $fileContent = null;

            if ($extension === 'pdf') {
                // Extract table from page 2 using Tabula-Java
                $tableRows = $this->extractPage2TableWithTabula($file->getRealPath());

                // Also get full text once for PO/date fallback
                $fileContent = $this->convertPdfToText($file->getRealPath());

                // Build structured data and save
                $extracted = $this->buildDataFromTable($tableRows, $fileContent, $file->getClientOriginalName());
                $this->updateDrop($extracted);
                continue;
            } else {
                session()->flash('message', 'Unsupported file type: ' . $file->getClientOriginalName());
                continue;
            }
            // Legacy text flow (kept as fallback for non-PDF types if ever supported)
            $extracted = $this->extractSzies($fileContent, $file->getClientOriginalName());
            $this->updateDrop($extracted);
        }

    }

    private function updateDrop($data){
        // dd('REMEMBER TO ADD PRICING TO ASOS EXPORT TO ZOHO - RERUN THIS AFTER REMOVING LINE');
        $sizes = $data['sizes'];
        $customerPo = $data['po_number'];
        $netWeight = $data['net_weight'];
        $grossWeight = $data['gross_weight'];
        $noCartons = $data['no_cartons'];
        $poDate = Carbon::createFromFormat('d-m-Y', $data['po_date'])->format('Y-m-d');

        $shipmentLine = ShipmentLine::with('shipment_line_sizes')
                            ->whereRelation('customer_order_lines.customer_orders', 'customer_po', $customerPo)->get();
        if($shipmentLine->count() > 1){
            throw new \Exception("Too many shipment lines matched for PO: " . $customerPo);
        }
        $shipmentLine = $shipmentLine->first();
        $shipmentLine->update([
            'net_weight' => $netWeight,
            'gross_weight' => $grossWeight,
            'no_cartons' => $noCartons,
            'exfty' => $poDate,
            'complete' => 1,
        ]);
        
        // Update commodity code if available
        if (!empty($sizes) && isset($sizes[0]['commodity_code'])) {
            $commodityCode = $sizes[0]['commodity_code'];
            $commodityCodeRecord = \App\Models\CommodityCodes::where('id', $commodityCode)->first();
            
            if ($commodityCodeRecord) {
                // Get the style from the shipment line relationship
                $style = $shipmentLine->customer_order_lines->colourways->style_versions->styles;
                if ($style) {
                    $style->update(['commodity_codes_id' => $commodityCodeRecord->id]);
                }
            }
        }
        
        // dd($sizes);
        foreach($sizes as $size){
            $customerOrderLineQuantitiy = $shipmentLine->customer_order_lines->customer_order_line_quantities->where('SKU', $size['sku'])->first();
            if($customerOrderLineQuantitiy == NULL)
                throw new \Exception("Customer order line quantity not found for SKU: " . $size['sku'] . ", PO: " . $customerPo);
            $shipmentLineSize = $shipmentLine->shipment_line_sizes->where('sizes_id', $customerOrderLineQuantitiy->sizes_id)->first();
            $shipmentLineSize->update([
                'shipped_qty' => $size['qty'],
            ]);
        }

        session()->flash('message', 'Complete!');
		session()->flash('alert-class', 'alert-success');
    }

    private function extractSzies($text, $filename){

        // 1. Isolate Summary section
        $summarySection = Str::between($text, 'Summary', 'FULL Product Description');
        // 2. Match size + units in ASOS format
        $flat = preg_replace('/\s+/', ' ', trim($summarySection));

        // Match ASOS format: ASN ID, Option ID, Option, Description, SKU, Website Colour, Brand Size, Commodity Code, Hazardous, Total Units, No. of Cartons, NW, GW, Country of Origin, Preference Agreement
        preg_match_all(
            '/(\d+)\s+(\d+)\s+(\d+)\s+([^\n]+?)\s+(\d{9,})\s+([^\n]+?)\s+([A-Z0-9\/\-]+)\s+(\d+)\s+[^\n]*?N\/A[^\n]*?(\d+(?:\.\d+)?)/',
            $flat,
            $matches,
            PREG_SET_ORDER
        );

        $collection = collect($matches)->map(function ($m) {
            return [
                'asn_id'      => $m[1],
                'option_id'   => $m[2],
                'option'      => $m[3],
                'description' => trim($m[4]),
                'sku'         => $m[5],
                'colour'      => trim($m[6]),
                'size'        => $m[7],
                'commodity_code' => $m[8],
                'qty'         => (int) $m[9],
            ];
                });

                // dd($collection);
 
        preg_match('/Total NW \(Kg\):\s*([\d\.]+)/', $text, $nwMatch);
        preg_match('/Total GW \(Kg\):\s*([\d\.]+)/', $text, $gwMatch);
        preg_match('/Total Cartons\/Hanging Pack:\s*(\d+)/', $text, $cartonsMatch);
        
        // Try to match PO number in various formats
        // 1. Standard PO- format (extract just the number after PO-)
        preg_match('/PO-(\d+)/', $text, $poMatch);
        // 2. ASOS format: Look for 12-digit numbers and filter out ones starting with 0000
        if (empty($poMatch)) {
            preg_match_all('/\b\d{12}\b/', $text, $allMatches);
            if (!empty($allMatches[0])) {
                foreach ($allMatches[0] as $match) {
                    if (!preg_match('/^0{4}/', $match)) {
                        $poMatch[0] = $match;
                        $poMatch[1] = $match;
                        break;
                    }
                }
            }
        }
        // 3. Fallback: If still no match, try the beginning of document
        if (empty($poMatch)) {
            preg_match('/^(\d{12})/', trim($text), $poMatch);
        }

        //GET PO NUMBER
        $po_number = $poMatch[1] ?? null;
        if (empty($po_number)) {
            // 1. Try to find a number after "PO-" in filename
            if (preg_match('/PO-(\d{10,12})/', $filename, $match)) {
                $po_number = $match[1];
            }
            // 2. Else, take the last 10–12 digit number in the filename
            elseif (preg_match_all('/\d{10,12}/', $filename, $matches) && !empty($matches[0])) {
                $po_number = end($matches[0]);
            }
        }

        
        //GET DATE
        if (!empty($po_number) && preg_match(
            '/PO-' . preg_quote($po_number, '/') . '.*?(\d{2}-\d{2}-\d{4})/s',
            $text,
            $dateMatch
        )) {
            $po_date = $dateMatch[1];
        }
        // Try to find date near the PO number or at the beginning of document
        if (empty($po_date) && preg_match('/Date:\s*(\d{2}-\d{2}-\d{4})/', $text, $dateMatch)) {
            $po_date = $dateMatch[1];
        }
        if (empty($po_date) && preg_match('/\b\d{2}-\d{2}-\d{4}\b/', $text, $fallbackMatch)) {
            $po_date = $fallbackMatch[0];
        }
        $net_weight  = $nwMatch[1] ?? null;
        $gross_weight = $gwMatch[1] ?? null;
        $no_cartons  = $cartonsMatch[1] ?? null;
        // dd($collection);
        return [
            'sizes' => $collection,
            'net_weight' => $net_weight,
            'gross_weight' => $gross_weight,
            'no_cartons' => $no_cartons,
            'po_number' => $po_number,
            'po_date' => $po_date,
        ];

    }
    private function convertPdfToText($filePath)
    {
        // Convert PDF to plain text
        return Pdf::getText($filePath);
    }

    public function render()
    {
        Gate::authorize('order:update');
        if($this->buttonDisabled()){
            session()->flash('message', 'Max 30 files.');
            session()->flash('alert-class', 'alert-warning');
        }
        // dd($this->getData());
        return view('livewire.imports.packing-list-upload-asos');
    }

    private function buttonDisabled(){
        return count($this->packingListFiles) > 30;
    }

    /**
     * Ensure Tabula-Java JAR exists locally; download if missing.
     */
    private function ensureTabulaJarPath(): string
    {
        $dir = storage_path('app/tabula');
        if (!is_dir($dir)) {
            @mkdir($dir, 0775, true);
        }
        $jarPath = $dir . DIRECTORY_SEPARATOR . 'tabula-1.0.5-jar-with-dependencies.jar';
        if (!file_exists($jarPath)) {
            $url = 'https://github.com/tabulapdf/tabula-java/releases/download/v1.0.5/tabula-1.0.5-jar-with-dependencies.jar';
            $tmp = $jarPath . '.part';
            $ctx = stream_context_create([
                'http' => ['timeout' => 120],
                'https' => ['timeout' => 120],
            ]);
            $data = @file_get_contents($url, false, $ctx);
            if ($data === false) {
                throw new \RuntimeException('Failed to download Tabula JAR. Ensure internet access or pre-provision the JAR at ' . $jarPath);
            }
            file_put_contents($tmp, $data);
            rename($tmp, $jarPath);
        }
        return $jarPath;
    }

    /**
     * Run Tabula-Java to extract tables from a given PDF page, returning normalized rows.
     * Tries lattice first, then falls back to stream.
     */
    private function extractPage2TableWithTabula(string $pdfPath): array
    {
        $jar = $this->ensureTabulaJarPath();
        if (!$this->hasJavaInstalled()) {
            throw new \RuntimeException('Java is not installed or not on PATH. Install OpenJDK (e.g., apt-get install -y openjdk-17-jre) and retry.');
        }
        $workDir = storage_path('app/tabula');
        $outFile = $workDir . DIRECTORY_SEPARATOR . 'out_page2.json';

        $escapedJar = escapeshellarg($jar);
        $escapedPdf = escapeshellarg($pdfPath);
        $escapedOut = escapeshellarg($outFile);

        // Try lattice mode first
        $cmdLattice = 'java -Dfile.encoding=UTF8 -jar ' . $escapedJar . ' --pages 2 --lattice --guess -f JSON -o ' . $escapedOut . ' ' . $escapedPdf . ' 2>&1';
        $outputLattice = shell_exec($cmdLattice) ?? '';

        $rows = $this->parseTabulaJsonToRows($outFile);
        if (empty($rows)) {
            // Fallback to stream mode if lattice found nothing
            @unlink($outFile);
            $cmdStream = 'java -Dfile.encoding=UTF8 -jar ' . $escapedJar . ' --pages 2 --stream --guess -f JSON -o ' . $escapedOut . ' ' . $escapedPdf . ' 2>&1';
            $outputStream = shell_exec($cmdStream) ?? '';
            $rows = $this->parseTabulaJsonToRows($outFile);
        }

        if (empty($rows)) {
            $debugSnippet = '';
            $debugSnippet .= "Lattice output:\n" . substr((string)($outputLattice ?? ''), 0, 500) . "\n";
            $debugSnippet .= "Stream output:\n" . substr((string)($outputStream ?? ''), 0, 500) . "\n";
            $existsMsg = file_exists($outFile) ? ('Tabula JSON written, size=' . filesize($outFile) . ' bytes') : 'Tabula JSON file not created';
            throw new \RuntimeException('No table rows detected on page 2. ' . $existsMsg . '. Debug:\n' . $debugSnippet);
        }

        return $rows;
    }

    /**
     * Parse Tabula JSON output file into a simple array of rows (array<array<string>>).
     */
    private function parseTabulaJsonToRows(string $jsonPath): array
    {
        if (!file_exists($jsonPath)) {
            return [];
        }
        $json = @file_get_contents($jsonPath);
        if ($json === false || $json === '') {
            return [];
        }
        $decoded = json_decode($json, true);
        if (!is_array($decoded)) {
            return [];
        }
        // Tabula JSON can be either an array of tables, or an object with fields depending on version
        $tables = [];
        if (array_is_list($decoded)) {
            $tables = $decoded;
        } elseif (isset($decoded['tables']) && is_array($decoded['tables'])) {
            $tables = $decoded['tables'];
        } else {
            $tables = [$decoded];
        }

        // Pick the first non-empty table
        foreach ($tables as $table) {
            if (!isset($table['data']) || !is_array($table['data'])) {
                continue;
            }
            $rows = [];
            foreach ($table['data'] as $row) {
                $cells = [];
                foreach ($row as $cell) {
                    $cells[] = isset($cell['text']) ? trim((string) $cell['text']) : '';
                }
                // Skip completely empty rows
                if (implode('', $cells) !== '') {
                    $rows[] = $cells;
                }
            }
            if (!empty($rows)) {
                return $rows;
            }
        }

        return [];
    }

    private function hasJavaInstalled(): bool
    {
        $which = shell_exec('command -v java 2>&1') ?? '';
        if (trim($which) === '') {
            return false;
        }
        return true;
    }

    /**
     * Convert Tabula rows + full text into our structured payload for updateDrop
     */
    private function buildDataFromTable(array $rows, string $fullText, string $filename): array
    {
        if (empty($rows) || count($rows) < 2) {
            throw new \RuntimeException('Table on page 2 has no data rows.');
        }

        $header = $rows[0];

        // Helper to find column index by fuzzy header match
        $findIdx = function(array $hdr, array $needles, int $defaultIdx = -1): int {
            foreach ($hdr as $i => $label) {
                $norm = strtolower(preg_replace('/\s+/', '', (string)$label));
                foreach ($needles as $needle) {
                    $needleNorm = strtolower(preg_replace('/\s+/', '', $needle));
                    if (str_contains($norm, $needleNorm)) {
                        return $i;
                    }
                }
            }
            return $defaultIdx;
        };

        $skuIdx = $findIdx($header, ['SKU']);
        $qtyIdx = $findIdx($header, ['TotalUnits', 'Units']);
        $commodityIdx = $findIdx($header, ['CommodityCode', 'Commodity', 'Code']);
        $nwIdx  = $findIdx($header, ['NW(Kg)', 'NWKg', 'NW']);
        $gwIdx  = $findIdx($header, ['GW(Kg)', 'GWKg', 'GW']);
        $cartIdx = $findIdx($header, ['No.ofCartons', 'Cartons', 'Hanging']);

        // Fallback to known positions from sample if fuzzy match failed
        if ($skuIdx < 0 && isset($header[3]) && strtolower(trim($header[3])) === 'sku') $skuIdx = 3;
        if ($qtyIdx < 0 && isset($header[9])) $qtyIdx = 9;
        if ($commodityIdx < 0 && isset($header[7])) $commodityIdx = 7; // Commodity code is typically in position 7
        if ($cartIdx < 0 && isset($header[10])) $cartIdx = 10;
        if ($nwIdx < 0 && isset($header[11])) $nwIdx = 11;
        if ($gwIdx < 0 && isset($header[12])) $gwIdx = 12;

        $sizes = collect();
        $totalNW = 0.0;
        $totalGW = 0.0;
        $totalCartons = 0.0;

        for ($r = 1; $r < count($rows); $r++) {
            $row = $rows[$r];
            $sku = $skuIdx >= 0 && isset($row[$skuIdx]) ? trim((string)$row[$skuIdx]) : '';
            if ($sku === '') {
                continue;
            }
            $qtyStr = $qtyIdx >= 0 && isset($row[$qtyIdx]) ? (string)$row[$qtyIdx] : '';
            $qty = (int) round((float) preg_replace('/[^0-9\.-]/', '', $qtyStr));
            
            $commodityCode = null;
            if ($commodityIdx >= 0 && isset($row[$commodityIdx])) {
                $commodityCode = trim((string)$row[$commodityIdx]);
            }
            
            $sizes->push([
                'sku' => $sku,
                'qty' => $qty,
                'commodity_code' => $commodityCode,
            ]);

            if ($nwIdx >= 0 && isset($row[$nwIdx])) {
                $totalNW += (float) preg_replace('/[^0-9\.-]/', '', (string)$row[$nwIdx]);
            }
            if ($gwIdx >= 0 && isset($row[$gwIdx])) {
                $totalGW += (float) preg_replace('/[^0-9\.-]/', '', (string)$row[$gwIdx]);
            }
            if ($cartIdx >= 0 && isset($row[$cartIdx])) {
                $totalCartons += (float) preg_replace('/[^0-9\.-]/', '', (string)$row[$cartIdx]);
            }
        }

        // If we failed to compute any totals, fallback to text regexes
        $poAndDate = $this->extractPoAndDateFromText($fullText, $filename);
        // Fallback/override totals from explicit summary lines if present
        $nwMatch = [];$gwMatch=[];$cartonsMatch=[];
        if (preg_match('/Total NW \(Kg\):\s*([\d\.]+)/', $fullText, $nwMatch)) {
            $totalNW = (float)$nwMatch[1];
        }
        if (preg_match('/Total GW \(Kg\):\s*([\d\.]+)/', $fullText, $gwMatch)) {
            $totalGW = (float)$gwMatch[1];
        }
        if (preg_match('/Total Cartons\/Hanging Pack:\s*(\d+)/', $fullText, $cartonsMatch)) {
            $totalCartons = (float)$cartonsMatch[1];
        }

        return [
            'sizes' => $sizes,
            'net_weight' => $totalNW ? number_format($totalNW, 3, '.', '') : null,
            'gross_weight' => $totalGW ? number_format($totalGW, 3, '.', '') : null,
            'no_cartons' => $totalCartons ? (string)$totalCartons : null,
            'po_number' => $poAndDate['po_number'] ?? null,
            'po_date' => $poAndDate['po_date'] ?? null,
        ];
    }

    private function extractPoAndDateFromText(string $text, string $filename): array
    {
        preg_match('/PO-(\d+)/', $text, $poMatch);
        if (empty($poMatch)) {
            preg_match_all('/\b\d{12}\b/', $text, $allMatches);
            if (!empty($allMatches[0])) {
                foreach ($allMatches[0] as $match) {
                    if (!preg_match('/^0{4}/', $match)) {
                        $poMatch[0] = $match;
                        $poMatch[1] = $match;
                        break;
                    }
                }
            }
        }
        if (empty($poMatch)) {
            preg_match('/^(\d{12})/', trim($text), $poMatch);
        }
        $po_number = $poMatch[1] ?? null;
        if (empty($po_number)) {
            if (preg_match('/PO-(\d{10,12})/', $filename, $match)) {
                $po_number = $match[1];
            } elseif (preg_match_all('/\d{10,12}/', $filename, $matches) && !empty($matches[0])) {
                $po_number = end($matches[0]);
            }
        }

        $po_date = null;
        if (!empty($po_number) && preg_match('/PO-' . preg_quote($po_number, '/') . '.*?(\d{2}-\d{2}-\d{4})/s', $text, $dateMatch)) {
            $po_date = $dateMatch[1];
        }
        if (empty($po_date) && preg_match('/Date:\s*(\d{2}-\d{2}-\d{4})/', $text, $dateMatch)) {
            $po_date = $dateMatch[1];
        }
        if (empty($po_date) && preg_match('/\b\d{2}-\d{2}-\d{4}\b/', $text, $fallbackMatch)) {
            $po_date = $fallbackMatch[0];
        }

        return [
            'po_number' => $po_number,
            'po_date' => $po_date,
        ];
    }
}
