<?php

namespace App\Http\Livewire\Imports;

use Gemini;
use OpenAI;
use OpenAI\Client;
use Gemini\Data\Blob;
use Livewire\Component;
use App\Models\Customer;
use Spatie\PdfToText\Pdf;
use Gemini\Enums\MimeType;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithFileUploads;
use Illuminate\Support\Carbon;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Http;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ImportExcelWithValues;

class PackingListUpload extends BaseComponent
{
    use WithFileUploads;
    public $packingList = '';
    public $answer = '';
    public $question = '';

    public $packingListFiles = [];

    public $customer;
    public $startDate;
    public $endDate;

	#[Computed]
	public function customers(){
		return Customer::allCached();
	}

    public function mount()
    {
        // Set default values for date range
        $this->startDate = Carbon::now()->subDays(3)->toDateString();
        $this->endDate = Carbon::now()->addDay()->toDateString();
    }

    public function uploadPackingList()
    {
        if (empty($this->packingListFiles)) {
            session()->flash('message', 'No files uploaded.');
            return;
        }
        elseif($this->buttonDisabled()){
            session()->flash('message', 'Too many files uploaded. Max 10.');
            return;
        }
        elseif(!$this->customer){
            session()->flash('message', 'Select a customer.');
            return;
        }

        $combinedContent = ''; // To store combined content from all files

        // Loop through each uploaded file
        foreach ($this->packingListFiles as $file) {
            $extension = $file->getClientOriginalExtension();
            $fileContent = null;

            if ($extension === 'csv') {
                $fileContent = file_get_contents($file->getRealPath());
            } elseif (in_array($extension, ['xls', 'xlsx'])) {
                $fileContent = $this->convertExcelToCsv($file->getRealPath());
            } elseif ($extension === 'pdf') {
                $fileContent = $this->convertPdfToText($file->getRealPath());
            } else {
                session()->flash('message', 'Unsupported file type: ' . $file->getClientOriginalName());
                continue;
            }

            // Append the content of this file to the combined content
            if ($fileContent) {
                $combinedContent .= "File: {$file->getClientOriginalName()}\n" . $fileContent . "\n\n";
            } else {
                session()->flash('message', 'File conversion failed for ' . $file->getClientOriginalName());
            }
        }

        // Check if combinedContent is empty after processing all files
        if (empty(trim($combinedContent))) {
            session()->flash('message', 'No valid file content to process after file conversions.');
            return;
        }

        // Send the combined content to ChatGPT
        if ($combinedContent) {
            $this->sendToGeminiAI($combinedContent);
        } else {
            session()->flash('message', 'No valid file content to process.');
        }
    }

    private function convertExcelToCsv($filePath)
    {
        // Load the entire Excel file as a collection of sheets
        $excelData = Excel::toArray(new ImportExcelWithValues, $filePath);

        // Initialize a variable to store the combined CSV content
        $csvContent = '';

        // Loop through each sheet and append its content to the combined CSV
        foreach ($excelData as $sheet) {
            $dataStarted = false; // Flag to mark when data has started
            foreach ($sheet as $row) {
                if (!$dataStarted) {
                    // Check if the row has any non-empty cell to mark the start of data
                    if (array_filter($row, function ($cell) { return $cell !== null && $cell !== ''; })) {
                        $dataStarted = true;
                    }
                }

                if ($dataStarted) {
                    // If data has started, filter out only trailing empty columns in the row
                    $filteredRow = $this->trimEmptyColumns($row);

                    // Skip empty rows entirely
                    if (!empty($filteredRow)) {
                        $csvContent .= implode(',', $filteredRow) . "\n";
                    }
                } else {
                    // Before data starts, retain the row as is
                    $csvContent .= implode(',', $row) . "\n";
                }
            }
            $csvContent .= "\n"; // Add a line break between sheets for readability
        }

        return $csvContent;
    }

    private function trimEmptyColumns($row)
    {
        // Remove trailing empty elements from the row
        while (end($row) === null || end($row) === '') {
            array_pop($row);
        }
        return $row;
    }

    private function convertPdfToText($filePath)
    {
        // Convert PDF to plain text
        return Pdf::getText($filePath);
    }

    private function sendToGeminiAI($fileContent)
    {
        $this->answer = [];
        $client = Gemini::client(env('GEMINI_API_KEY'));
        $dbData = arrayToCsv($this->getData());
        $offset = 0;
        $limit = 50;

        try {
            $responseContent = $this->processGeminiRequest($client, $dbData, $fileContent, $offset, $limit);

            if ($responseContent) {
                $this->answer = json_decode($responseContent);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $this->processMatches();

                    while ($this->answer->more_data ?? false) {
                        $offset += $limit;
                        $responseContent = $this->processGeminiRequest($client, $dbData, $fileContent, $offset, $limit);

                        if ($responseContent) {
                            $newAnswer = json_decode($responseContent);
                            if (json_last_error() === JSON_ERROR_NONE) {
                                $this->appendMatches($newAnswer);
                                $this->answer = $newAnswer;
                            } else {
                                session()->flash('message', 'Failed to decode additional JSON response from Gemini.');
                                break;
                            }
                        } else {
                            session()->flash('message', 'Failed to retrieve additional data from Gemini.');
                            break;
                        }
                    }
                } else {
                    session()->flash('message', 'Failed to decode JSON response from Gemini.');
                    \Log::error('Gemini JSON Decode Error: ' . json_last_error_msg());
                    \Log::error('Gemini Response: ' . $responseContent);
                    $this->answer = null;
                }
            } else {
                session()->flash('message', 'Received empty response from Gemini.');
            }
        } catch (\Exception $e) {
            session()->flash('message', 'An error occurred while communicating with Gemini: ' . $e->getMessage());
            \Log::error('Gemini API Error: ' . $e->getMessage());
            $this->answer = null;
        }
    }

    private function processGeminiRequest($client, $dbData, $fileContent, $offset, $limit)
    {
        try {
            $prompt = '
                Here is a packing list. It may have multiple styles and colourways and sizes. Styles, colourways and sizes could be stored in multiple containers - you will need to add them together to get the totals.

                I need you to look at the data from the DB and match it to the details on the packing list. Match with style, colourway and size.

                THE CUSTOMER PO NUMBER MUST MATCH UP.

                Each row from the file must be matched to **only one** row from the shipment data.

                Never assign the same style/colourway/size to more than one PO.
                Choose the best match based on PO number or unallocated quantity.

                Avoid duplication.

                EXAMPLE OUTPUT:
                {
                    "matches":[
                        {
                            "shipment_line_id":8479,
                            "net_weight":10.50,
                            "gross_weight":13.60,
                            "cartons":10,
                            "sizes":[
                                "sls_id":158490,
                                "shipped_qty":8,
                            ],
                            [
                                "sls_id":158491,
                                "shipped_qty":13,
                            ]
                        },
                    ]
                    "more_data": 1
                }
                OFFSET: ' . $offset . ' LIMIT: ' . $limit . '
            ';

            $parts = [
                $prompt,
                "Here are the Packing Lists:\n" . $fileContent,
                "Here is the DB extract:\n" . $dbData,
            ];

            $result = $client->generativeModel(config('gemini.models.flash'))->generateContent($parts);
            $text = trim($result->text());

            // Cleanup if it accidentally adds ```json
            if (str_starts_with($text, '```json')) {
                $text = str_replace(['```json', '```'], '', $text);
            }

            return $text;
        } catch (\Exception $e) {
            \Log::error('Error in Gemini Request: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Process matches and add additional data from the database.
     */
    private function processMatches()
    {
        foreach ($this->answer->matches as $sl => $shipmentLine) {
            $dbLine = ShipmentLine::find($shipmentLine->shipment_line_id);

            $this->answer->matches[$sl]->style_name = $dbLine->customer_order_lines->colourways->style_versions->styles->customer_ref;
            $this->answer->matches[$sl]->customer_po = $dbLine->customer_order_lines->customer_orders->customer_po;
            $this->answer->matches[$sl]->colourway = $dbLine->customer_order_lines->colourways->name;

            foreach ($shipmentLine->sizes as $s => $size) {
                $this->answer->matches[$sl]->sizes[$s]->qty_ordered = $dbLine->shipment_line_sizes->where('id', $size->sls_id)->first()?->qty;
                $this->answer->matches[$sl]->sizes[$s]->size = $dbLine->shipment_line_sizes->where('id', $size->sls_id)->first()?->sizes->name;
            }
        }
    }

    /**
     * Append new matches to the existing answer.
     */
    private function appendMatches($newAnswer)
    {
        if (isset($newAnswer->matches)) {
            $this->answer->matches = array_merge($this->answer->matches, $newAnswer->matches);
        }

        $this->answer->more_data = $newAnswer->more_data ?? false;
    }






    public function updateValues()
    {
        if (empty($this->answer) || empty($this->answer->matches)) {
            session()->flash('message', 'No matches to update.');
            return;
        }

        foreach($this->answer->matches as $match){
            $sl = ShipmentLine::find($match->shipment_line_id);

            foreach($match->sizes as $size){
                $sls = ShipmentLineSizes::find($size->sls_id);
                if($sls->shipment_line_id == $sl->id){
                    $sls->cartons = $size->cartons;
                    $sls->save();
                }
                else{
                    throw new \Exception("ERROR - Shipment Line Size ID mismatch");
                }
            }
            $sl->gross_weight = $match->gross_weight;
            $sl->net_weight = $match->net_weight;
            $sl->no_cartons = $match->total_cartons;
            $sl->complete = TRUE;
            $sl->save();
        }

        // return redirect(request()->header('Referer'));
    }

    public function render()
    {
        Gate::authorize('order:update');
        if($this->buttonDisabled()){
            session()->flash('message', 'Max 10 files.');
            session()->flash('alert-class', 'alert-warning');
        }
        // dd($this->getData());
        return view('livewire.imports.packing-list-upload');
    }

    private function buttonDisabled(){
        return count($this->packingListFiles) > 10;
    }

    public function getData(){
        $results = DB::select("
        SELECT
            `shipment_lines`.`no_invoiced` AS `no_invoiced`,
            `customer_orders`.`order_type` AS `order_type`,
            `shipment_lines`.`id` AS `shipment_line_id`,
            `seasons`.`description` AS `season`,
            `factory`.`name` AS `factory`,
            `factory`.`id` AS `factory_id`,
            `styles`.`category` AS `category`,
            `customers`.`name` AS `customer`,
            `customer_orders`.`customer_po` AS `customer_po`,
            `designs`.`description` AS `style_name`,
            `colourways`.`name` AS `colour`,
            `shipment_lines`.`shipment_id` AS `truck`,
            `transporters`.`name` AS `transporter`,
            `shipment_lines`.`collection_date` AS `collection_date`,
            `shipment_lines`.`complete` AS `shipped`,
            `shipment_lines`.`shipment_id` AS `shipment_id`,
            `shipment_lines`.`exfty` AS `exfty`,
            `customer_orders`.`shipment_mode`,
            `customer_orders`.`id` AS `customer_orders_id`,
            `styles`.`id` AS `styles_id`,
            `colourways`.`id` AS `colourways_id`,
            `style_versions`.`id` AS `style_versions_id`,
            `style_versions`.`name` AS `style_version`,
            `factory`.`countries_id` AS `coo`,
            `seasons`.`id` AS `seasons_id`,
            `styles`.`designs_id`,
            `styles`.`customer_ref`,


            (SELECT
                    CONCAT('[',
                                GROUP_CONCAT(JSON_OBJECT('colq_id',
                                            colq.id,
                                            'sls_id',
                                            sls.id,
                                            'size',
                                            sizes.name,
                                            'qty_ordered',
                                            colq.qty,
                                            'shipped_qty',
                                            sls.shipped_qty)
                                    SEPARATOR ','),
                                ']')
                FROM
                    shipment_line_sizes sls
                        JOIN
                    sizes ON sizes.id = sls.sizes_id
                        AND sls.shipment_line_id = shipment_lines.id
                        JOIN
                    customer_order_line_quantities colq ON colq.customer_order_lines_id = customer_order_lines.id
                        AND colq.sizes_id = sls.sizes_id
                WHERE
                    sls.deleted_at IS NULL
                GROUP BY sls.shipment_line_id) AS sizes
        FROM
            `shipment_lines`
                INNER JOIN
            `customer_order_lines` ON `customer_order_lines`.`id` = `shipment_lines`.`customer_order_lines_id`
                INNER JOIN
            `customer_orders` ON `customer_orders`.`id` = `customer_order_lines`.`customer_orders_id`
                AND `customer_orders`.`order_type` = 'wholesale'
                AND `customer_orders`.`cancelled` = 0
                AND `customer_orders`.`customers_id` = $this->customer
                INNER JOIN
            `customers` ON `customers`.`id` = `customer_orders`.`customers_id`
                INNER JOIN
            `colourways` ON `colourways`.`id` = `customer_order_lines`.`colourways_id`
                INNER JOIN
            `style_versions` ON `style_versions`.`id` = `colourways`.`style_versions_id`
                INNER JOIN
            `styles` ON `styles`.`id` = `style_versions`.`styles_id`
                INNER JOIN
            `seasons` ON `seasons`.`id` = `customer_orders`.`seasons_id`
                INNER JOIN
            `suppliers` AS `factory` ON `factory`.`id` = `style_versions`.`factory_id`
                INNER JOIN
            `designs` ON `designs`.`id` = `styles`.`designs_id`
                LEFT JOIN
            `shipments` ON `shipments`.`id` = `shipment_lines`.`shipment_id`
                LEFT JOIN
            `suppliers` AS `transporters` ON `transporters`.`id` = `shipments`.`transporter_id`
        WHERE
                `shipment_lines`.`exfty` >= '$this->startDate'
                AND `shipment_lines`.`exfty` <= '$this->endDate'
                AND `shipment_lines`.`deleted_at` IS NULL
                AND `styles`.`cancelled` = 0
                AND `colourways`.`cancelled` = 0
                AND `customer_order_lines`.`cancelled` = 0
        ");

        if (empty($results)) {
            session()->flash('message', 'No data retrieved from the database.');
            return [];
        }

        return json_decode(json_encode($results), TRUE);
    }
}
