<?php

namespace App\Http\Livewire\Imports;

use OpenAI\Client;
use Gemini\Enums\Role;
use Livewire\Component;
use Gemini\Data\Content;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use Gemini\Laravel\Facades\Gemini;
use Illuminate\Support\Facades\DB;
use App\Http\Livewire\BaseComponent;

class PackingList extends BaseComponent
{
    public $packingList = '';
    public $answer = '';
    public $question = '';

    public function askQuestion()
    {
        $sendHistory = [];
        $this->question = "
            Here is some data from our database:" .
            json_encode($this->getData()) .
            "Here is a packing list for a shipment:" .
            $this->packingList .
            '
            Match the packing list to the shipment lines. There could be several that it matches - i.e. one for each colourway.
            For each size, add up the numbers in each carton or box. This should give the total number of garments in that size.
            e.g. there could be several boxes which contain size S which will need to be added up for a total number of size S.
            Note that sizes could be numbers, so size 8 may be packed into several boxes and need to be added up to get the total for that size.
            List the shipment_line_id, style_name, colourway_name, sls_id, size, qty_ordered (all from db extract) and shipped_qty (from packing list). Format as JSON.
            DO NOT ADD ``` JSON NOTATION FOR ANY REASON.
            At the end, within the same json object, it should list the total no. cartons/boxes and net weight, and gross weight.

            EXAMPLE:
            {

                "matches":[
                    {
                        "shipment_line_id":8479,
                        "colourway":"Red",
                        "style_name":"Turtle Neck",
                        "sls_id":158490,
                        "size":"XS",
                        "qty_ordered":9
                        "shipped_qty":8
                    },
                    {
                        "shipment_line_id":8479,
                        "colourway":"Red",
                        "style_name":"Turtle Neck",
                        "sls_id":158491,
                        "size":"S",
                        "qty_ordered":45
                        "shipped_qty":47
                    }
                ],
                "total_cartons":5,
                "net_weight":"64.22 kg",
                "gross_weight":"69.72 kg"
            }
        ';

        // Initialize OpenAI client
        $client = \OpenAI::client(env('OPENAI_API_KEY'));

        // Send request to OpenAI's model for completions
        $response = $client->chat()->create([
            'model' => 'gpt-4o-2024-08-06',
            'messages' => [
                ['role' => 'user', 'content' => $this->question],
            ],
        ]);

        // Capture the response text
        $this->answer = json_decode($response->choices[0]->message->content);
        $this->question = '';
    }

    #[On('csvDataPasted')]
    public function handleCsvData($csv)
    {
        $this->packingList = $csv;
        $this->askQuestion();
    }

    public function updateValues(){
        foreach($this->answer->matches as $match){
            $sls = ShipmentLineSizes::find($match->sls_id);
            $sl = ShipmentLine::find($match->shipment_line_id);
            if($sls->shipment_line_id == $sl->id){
                $sls->shipped_qty = $match->shipped_qty;
                $sls->save();
            }
            else{
                throw new \Exception("ERROR - Shipment Line Size ID mismatch");
            }
        }
    }

    public function render()
    {
        // dd($this->getData());
        return view('livewire.imports.packing-list');
    }
    public function getData(){
        $results = DB::select("
        SELECT
            `colourways`.`cancelled` AS `colourway_cancelled`,
            `styles`.`cancelled` AS `style_cancelled`,
            `shipment_lines`.`no_invoiced` AS `no_invoiced`,
            `customer_orders`.`order_type` AS `order_type`,
            `shipment_lines`.`id` AS `shipment_line_id`,
            `seasons`.`description` AS `season`,
            `factory`.`name` AS `factory`,
            `factory`.`id` AS `factory_id`,
            `styles`.`category` AS `category`,
            `customers`.`name` AS `customer`,
            `customer_orders`.`customer_po` AS `customer_po`,
            `designs`.`description` AS `style_name`,
            `colourways`.`name` AS `colour`,
            `shipment_lines`.`shipment_id` AS `truck`,
            `transporters`.`name` AS `transporter`,
            `shipment_lines`.`collection_date` AS `collection_date`,
            `shipment_lines`.`complete` AS `shipped`,
            `shipment_lines`.`shipment_id` AS `shipment_id`,
            `shipment_lines`.`exfty` AS `exfty`,
            `shipment_lines`.`gross_weight`,
            `shipment_lines`.`net_weight`,
            `shipment_lines`.`no_cartons`,
            `customer_orders`.`shipment_mode`,
            `customer_orders`.`id` AS `customer_orders_id`,
            `styles`.`id` AS `styles_id`,
            `colourways`.`id` AS `colourways_id`,
            `style_versions`.`id` AS `style_versions_id`,
            `style_versions`.`name` AS `style_version`,
            `customers`.`currency` AS `cust_currency`,
            `factory`.`currency` AS `fact_currency`,
            `factory`.`countries_id` AS `coo`,
            `seasons`.`id` AS `seasons_id`,
            `shipment_lines`.`sealer_status` AS `sealer_status`,
            `shipment_lines`.`shipment_status` AS `shipment_status`,
            `customer_order_lines`.`cancelled` AS `slCancelled`,
            `styles`.`designs_id`,
            `styles`.`customer_ref`,


            (SELECT
                    CONCAT('[',
                                GROUP_CONCAT(JSON_OBJECT('colq_id',
                                            colq.id,
                                            'sls_id',
                                            sls.id,
                                            'size',
                                            sizes.name,
                                            'qty_ordered',
                                            colq.qty,
                                            'shipped_qty',
                                            sls.shipped_qty)
                                    SEPARATOR ','),
                                ']')
                FROM
                    shipment_line_sizes sls
                        JOIN
                    sizes ON sizes.id = sls.sizes_id
                        AND sls.shipment_line_id = shipment_lines.id
                        JOIN
                    customer_order_line_quantities colq ON colq.customer_order_lines_id = customer_order_lines.id
                        AND colq.sizes_id = sls.sizes_id
                WHERE
                    sls.deleted_at IS NULL
                GROUP BY sls.shipment_line_id) AS sizes
        FROM
            `shipment_lines`
                INNER JOIN
            `customer_order_lines` ON `customer_order_lines`.`id` = `shipment_lines`.`customer_order_lines_id`
                INNER JOIN
            `customer_orders` ON `customer_orders`.`id` = `customer_order_lines`.`customer_orders_id`
                AND `customer_orders`.`order_type` = 'wholesale'
                AND `customer_orders`.`cancelled` = 0
                INNER JOIN
            `customers` ON `customers`.`id` = `customer_orders`.`customers_id`
                INNER JOIN
            `colourways` ON `colourways`.`id` = `customer_order_lines`.`colourways_id`
                INNER JOIN
            `style_versions` ON `style_versions`.`id` = `colourways`.`style_versions_id`
                INNER JOIN
            `styles` ON `styles`.`id` = `style_versions`.`styles_id`
                INNER JOIN
            `seasons` ON `seasons`.`id` = `customer_orders`.`seasons_id`
                INNER JOIN
            `suppliers` AS `factory` ON `factory`.`id` = `style_versions`.`factory_id`
                INNER JOIN
            `designs` ON `designs`.`id` = `styles`.`designs_id`
                LEFT JOIN
            `shipments` ON `shipments`.`id` = `shipment_lines`.`shipment_id`
                LEFT JOIN
            `suppliers` AS `transporters` ON `transporters`.`id` = `shipments`.`transporter_id`
        WHERE
            `shipment_lines`.`exfty` >= CURDATE() - 7
                AND `shipment_lines`.`exfty` <= CURDATE() + 1
                AND `shipment_lines`.`deleted_at` IS NULL
        ");

        return json_decode(json_encode($results), TRUE);
    }
}
