<?php

namespace App\Http\Livewire\Imports;

use Livewire\WithFileUploads;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Gate;
use App\Models\Seasons;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ImportExcelWithValues;
use Illuminate\Support\Str;
use App\Models\Sizes;
use Illuminate\Support\Carbon;

class PLMabliImport extends BaseComponent
{
	use WithFileUploads;

	public $season;
	public $plFiles = [];
	public $parsed = [];
	public $rows = [];
	public $summary = [];
	public $sizeHeaders = [];
	public $matchIndex = [];
	public $colourOverrides = [];
	public $saveToNewDrop = false;
	public $selectedDrops = [];

	#[Computed]
	public function seasons()
	{
		return Seasons::allCached();
	}

	public function updated($property)
	{
		if (Str::startsWith($property, 'colourOverrides')) {
			$this->computeSummary();
			$this->computeRowMatches();
		}
	}

	public function updatedColourOverrides($value)
	{
        $this->computeSummary();
		$this->computeRowMatches();
	}

	public function setColourOverride(string $key, ?string $value): void
	{
		$this->colourOverrides[$key] = $value ?? '';
		$this->computeSummary();
		$this->computeRowMatches();
	}

	public function parse()
	{
		if (empty($this->plFiles)) {
			session()->flash('message', 'No files uploaded.');
			return;
		}

		if (!$this->season) {
			session()->flash('message', 'Please select a season.');
			return;
		}

		$flat = [];
		foreach ($this->plFiles as $file) {
			$ext = strtolower($file->getClientOriginalExtension());
			if (in_array($ext, ['xls', 'xlsx'])) {
				$flat = array_merge($flat, $this->parseExcelFileToFlat($file));
			} elseif ($ext === 'csv') {
				$flat = array_merge($flat, $this->parseCsvFileToFlat($file->getRealPath()));
			} else {
				session()->flash('message', 'Unsupported file type: ' . $file->getClientOriginalName());
			}
		}
		$this->rows = $flat;
		$this->computeSummary();
		$this->computeRowMatches();
		if (empty($this->rows)) {
			session()->flash('message', 'No rows parsed. Ensure your CSV/Excel has a header row with size columns (e.g. 6M, 12M, 2Y...).');
		}
	}

	private function normalizeSize(string $label): string
	{
		$sz = strtoupper(trim($label));
		$sz = preg_replace('/\s+/', ' ', $sz);
		// Add space before (ADULT) if missing
		$sz = preg_replace('/^(XXS|XS|S|M|L|XL)\(ADULT\)$/', '$1 (ADULT)', $sz);
		$sz = preg_replace('/^(XXS|XS|S|M|L|XL)\s*\(\s*ADULT\s*\)$/', '$1 (ADULT)', $sz);
		return $sz;
	}

	private function sizeQueryVariants(string $size): array
	{
		$norm = $this->normalizeSize($size);
		$variants = [$norm];
		$variants[] = str_replace(' (ADULT)', '(ADULT)', $norm); // no space
		$variants[] = str_replace('(ADULT)', ' (ADULT)', $norm); // ensure space
		return array_values(array_unique($variants));
	}

	private function computeSummary(): void
	{
		$this->summary = [];
		$this->sizeHeaders = [];
		$this->matchIndex = [];
		if (empty($this->rows)) return;

		$sizeOrder = [];
		$buckets = [];
		foreach ($this->rows as $row) {
			$style = (string)($row['style'] ?? '');
			$styleName = (string)($row['style_name'] ?? '');
			$colour = (string)($row['colour_normalized'] ?? ($row['colour'] ?? ''));
			$size = $this->normalizeSize((string)($row['size'] ?? ''));
			$qty = (int)($row['qty'] ?? 0);
			if ($style === '' && $styleName === '') continue;
			if ($colour === '' || $size === '' || $qty <= 0) continue;
			$key = $style.'|'.$styleName.'|'.$colour;
			if (!isset($buckets[$key])) {
				$buckets[$key] = [
					'style' => $style,
					'style_name' => $styleName,
					'colour' => $colour,
					'db_colour' => '',
					'sizes' => [],       // PL totals
					'sls_sizes' => [],   // DB totals per size (Drop)
					'matched_row' => false,
					'colour_options' => $this->getColourOptionsForStyle($style),
					'total' => 0,
				];
			}
			$buckets[$key]['sizes'][$size] = ($buckets[$key]['sizes'][$size] ?? 0) + $qty;
			$buckets[$key]['total'] += $qty;
			$sizeOrder[$size] = true; // collect for sorting later
		}

		// Augment with shipment_line size totals and match flags, respecting overrides
		foreach ($buckets as $key => $row) {
			$overrideColour = $this->colourOverrides[$key] ?? null;
			$matchColour = $overrideColour && $overrideColour !== '' ? $overrideColour : $row['colour'];
			$lines = $this->getShipmentLinesWithSizes($row['style'], $matchColour);
			$buckets[$key]['matched_row'] = $lines->isNotEmpty();
			$buckets[$key]['db_colour'] = $lines->first()?->customer_order_lines?->colourways?->name ?? ($buckets[$key]['matched_row'] ? $matchColour : '');
			// Build drop candidates sorted oldest first (by exfty, then created_at, then id)
			$sorted = $lines->sortBy(function ($l) {
				$exftyTs = $l->exfty ? Carbon::parse($l->exfty)->getTimestamp() : PHP_INT_MAX;
				$createdTs = $l->created_at ? $l->created_at->getTimestamp() : PHP_INT_MAX;
				return [$exftyTs, $createdTs, $l->id];
			});
			$dropCandidates = $sorted->map(function ($l) {
				$label = '#'.$l->id;
				if ($l->exfty) { $label .= ' ('.Carbon::parse($l->exfty)->toDateString().')'; }
				return ['id' => $l->id, 'label' => $label];
			})->values()->all();
			$selectedId = $this->selectedDrops[$key] ?? ($dropCandidates[0]['id'] ?? null);
			if (!isset($this->selectedDrops[$key]) && $selectedId) { $this->selectedDrops[$key] = $selectedId; }
			$buckets[$key]['drop_candidates'] = $dropCandidates;
			$buckets[$key]['selected_drop_id'] = $selectedId;
			$slsBySize = [];
			foreach ($lines as $line) {
				foreach ($line->shipment_line_sizes as $sls) {
					$nm = $this->normalizeSize($sls->sizes->name ?? '');
					if ($nm === '') continue;
					$slsBySize[$nm] = ($slsBySize[$nm] ?? 0) + (int)($sls->qty ?? 0);
					$sizeOrder[$nm] = true;
				}
			}
			$buckets[$key]['sls_sizes'] = $slsBySize;
		}

		// Build ordered size header list from DB Sizes order
		$dbSizeOrder = Sizes::pluck('name')->map(fn($n) => strtoupper($n))->toArray();
		$headers = array_keys($sizeOrder);
		usort($headers, function($a, $b) use ($dbSizeOrder) {
			$ia = array_search($a, $dbSizeOrder, true); $ia = $ia === false ? PHP_INT_MAX : $ia;
			$ib = array_search($b, $dbSizeOrder, true); $ib = $ib === false ? PHP_INT_MAX : $ib;
			return $ia <=> $ib;
		});
		$this->sizeHeaders = $headers;
		$this->summary = array_values($buckets);
	}

	private function computeRowMatches(): void
	{
		foreach ($this->rows as $i => $row) {
			$style = (string)($row['style'] ?? '');
			$colour = (string)($row['colour_normalized'] ?? ($row['colour'] ?? ''));
			$size = $this->normalizeSize((string)($row['size'] ?? ''));
			$matched = !empty($this->findShipmentSizeMatches($style, $colour, $size));
			$this->rows[$i]['matched'] = $matched;
			$this->rows[$i]['size'] = $size; // store normalized
		}
	}

	private function findShipmentSizeMatches(string $style, string $colour, string $size): array
	{
		$variants = $this->sizeQueryVariants($size);
		$lines = ShipmentLine::query()
			->select('shipment_lines.id')
			->whereRelation('customer_order_lines.customer_orders.seasons', 'id', $this->season)
			->where(function($q) use ($style) {
				$q->whereHas('customer_order_lines.colourways.style_versions.styles', function($qq) use ($style) {
					$qq->where('styles.customer_ref', '=', $style);
				})
				->orWhereHas('customer_order_lines.colourways.style_versions', function($qq) use ($style) {
					$qq->where('style_versions.name', '=', $style);
				});
			})
			->whereRelation('customer_order_lines.colourways', 'name', 'like', $colour.'%')
			->with(['shipment_line_sizes' => function($q) use ($variants) {
				$q->whereHas('sizes', function($sq) use ($variants) { $sq->whereIn('name', $variants); });
			}])
			->get();
		$ids = [];
		foreach ($lines as $line) {
			foreach ($line->shipment_line_sizes as $sls) {
				$ids[] = $sls->id;
			}
		}
		return $ids;
	}

	private function getShipmentLinesWithSizes(string $style, string $colour)
	{
		return ShipmentLine::query()
			->whereRelation('customer_order_lines.customer_orders.seasons', 'id', $this->season)
			->where(function($q) use ($style) {
				$q->whereHas('customer_order_lines.colourways.style_versions.styles', function($qq) use ($style) {
					$qq->where('styles.customer_ref', '=', $style);
				})
				->orWhereHas('customer_order_lines.colourways.style_versions', function($qq) use ($style) {
					$qq->where('style_versions.name', '=', $style);
				});
			})
			->whereRelation('customer_order_lines.colourways', 'name', 'like', $colour.'%')
			->with(['shipment_line_sizes.sizes', 'customer_order_lines.colourways'])
			->get();
	}

	private function getColourOptionsForStyle(string $style): array
	{
		$rows = DB::table('colourways as cw')
			->join('style_versions as sv', 'sv.id', '=', 'cw.style_versions_id')
			->join('styles as s', 's.id', '=', 'sv.styles_id')
			->whereNull('cw.deleted_at')
			->where('cw.cancelled', 0)
			->whereNull('s.deleted_at')
			->where('s.cancelled', 0)
			->where(function($q) use ($style) {
				$q->where('s.customer_ref', '=', $style)
					->orWhere('sv.name', '=', $style);
			})
			->where('s.seasons_id', $this->season)
			->pluck('cw.name')
			->unique()
			->values()
			->all();
		return $rows;
	}

	public function applyUpdate()
	{
		if (empty($this->rows)) {
			session()->flash('message', 'Nothing to update.');
			return;
		}

		// Build a quick lookup of style|style_name|colour buckets that are matched (i.e., not red)
		$matchedBuckets = [];
		foreach ($this->summary as $bucket) {
			$key = ($bucket['style'] ?? '').'|'.($bucket['style_name'] ?? '').'|'.($bucket['colour'] ?? '');
			if (!empty($bucket['matched_row'])) { $matchedBuckets[$key] = true; }
		}

		$updated = 0;
		$affectedLines = [];
		$newDropShippedByLineAndSize = [];
		foreach ($this->rows as $r) {
			$style = trim((string)($r['style'] ?? ''));
			$styleName = trim((string)($r['style_name'] ?? ''));
			$colour = trim((string)($r['colour_normalized'] ?? ($r['colour'] ?? '')));
			$sizeName = $this->normalizeSize(trim((string)($r['size'] ?? '')));
			$qty = (int) ($r['qty'] ?? 0);
			if (!$style || !$sizeName || !$qty) { continue; }

			// Skip rows belonging to unmatched buckets (red summary lines)
			$bucketKey = $style.'|'.$styleName.'|'.$colour;
			if (empty($matchedBuckets[$bucketKey])) { continue; }

			$key = $bucketKey;
			$overrideColour = $this->colourOverrides[$key] ?? null;
			$effectiveColour = $overrideColour && $overrideColour !== '' ? $overrideColour : $colour;

			$candidateLines = ShipmentLine::query()
				->whereRelation('customer_order_lines.customer_orders.seasons', 'id', $this->season)
				->where(function($q) use ($style) {
					$q->whereHas('customer_order_lines.colourways.style_versions.styles', function($qq) use ($style) {
						$qq->where('styles.customer_ref', '=', $style);
					})
					->orWhereHas('customer_order_lines.colourways.style_versions', function($qq) use ($style) {
						$qq->where('style_versions.name', '=', $style);
					});
				})
				->when($effectiveColour, function ($q) use ($effectiveColour) {
					$q->whereRelation('customer_order_lines.colourways', 'name', 'like', $effectiveColour.'%');
				})
				->pluck('id');

			// Restrict to explicitly selected drop if provided
			$selectedDropId = $this->selectedDrops[$bucketKey] ?? null;
			if ($selectedDropId && $candidateLines->contains($selectedDropId)) {
				$candidateLines = collect([$selectedDropId]);
			}

			if ($candidateLines->isEmpty()) { continue; }

			$sls = ShipmentLineSizes::query()
				->whereIn('shipment_line_id', $candidateLines)
				->whereHas('sizes', function ($q) use ($sizeName) { $q->whereIn('name', $this->sizeQueryVariants($sizeName)); })
				->get();

			if ($sls->isEmpty()) { continue; }

			$target = $sls->sortByDesc(function ($row) {
				$ordered = (int) ($row->qty ?? 0);
				$shipped = (int) ($row->shipped_qty ?? 0);
				return max(0, $ordered - $shipped);
			})->first();

			if ($target) {
				if ($this->saveToNewDrop) {
					$lineId = $target->shipment_line_id;
					$sizeId = $target->sizes_id;
					$newDropShippedByLineAndSize[$lineId][$sizeId] = ($newDropShippedByLineAndSize[$lineId][$sizeId] ?? 0) + $qty;
					$affectedLines[$lineId] = true;
					$updated++;
				} else {
					$target->shipped_qty = $qty; // overwrite with packing list qty
					$target->save();
					$updated++;
					$affectedLines[$target->shipment_line_id] = true;
				}
			}
		}

		if (!empty($affectedLines)) {
			$lineIds = array_keys($affectedLines);
			$lines = ShipmentLine::whereIn('id', $lineIds)->with('shipment_line_sizes')->get();
			foreach ($lines as $line) {
				if ($this->saveToNewDrop) {
					// Create a new drop cloned from the matched line; set qty=0 and shipped to PL totals
					$newDrop = $line->replicate();
					$newDrop->complete = 0;
					$newDrop->shipment_id = 495;
					$newDrop->exfty = $line->exfty ? Carbon::parse($line->exfty)->addDay() : null;
					$newDrop->save();
					$shipMap = $newDropShippedByLineAndSize[$line->id] ?? [];
					foreach ($line->shipment_line_sizes as $sls) {
						$newSls = $sls->replicate();
						$newSls->shipment_line_id = $newDrop->id;
						$newSls->qty = 0;
						$newSls->shipped_qty = (int)($shipMap[$sls->sizes_id] ?? 0);
						$newSls->save();
					}
				} else {
					// Mark as complete and set shipment id as before
					$changes = [];
					$changes['complete'] = 1;
					$changes['shipment_id'] = 495;
					if (!empty($changes)) { $line->update($changes); }
				}
			}
		}

		session()->flash('message', $this->saveToNewDrop
			? "Created new drops and set shipped quantities on them for $updated size rows."
			: "Updated $updated shipment size rows.");
	}

	private function parseExcelFileToFlat($file): array
	{
		$sheets = Excel::toArray(new ImportExcelWithValues, $file);
		$flat = [];
		foreach ($sheets as $rows) {
			$flat = array_merge($flat, $this->rowsToFlat($rows));
		}
		return $flat;
	}

	private function parseCsvFileToFlat(string $path): array
	{
		$rows = [];
		if (($handle = fopen($path, 'r')) !== false) {
			while (($data = fgetcsv($handle)) !== false) {
				$rows[] = $data;
			}
			fclose($handle);
		}
		return $this->rowsToFlat($rows);
	}

	private function rowsToFlat(array $rows): array
	{
		if (empty($rows)) return [];

		// Detect the row that contains size headers (can be separate from the meta header row)
		$sizeHeaderIdx = -1;
		$sizeCols = [];
		$indexes = [ 'carton' => -1, 'style_name' => -1, 'style' => -1, 'colour' => -1 ];

		$isSizeHeader = function ($label) {
			$l = strtoupper(trim((string)$label));
			if (preg_match('/^\d{1,2}M$/', $l)) return true; // 6M, 12M, 18M
			if (preg_match('/^\d{1,2}Y$/', $l)) return true; // 2Y, 4Y, ...
			if (preg_match('/^(XXXS|XXS|XS|S|M|L|XL|XXL|3XL|4XL)$/', $l)) return true;
			if (preg_match('/^(XXS|XS|S|M|L|XL)\s*\(\s*ADULT\s*\)$/', $l)) return true;
			return false;
		};

		$normRow = fn(array $row) => array_map(fn($v) => trim((string)$v), $row);

		// initial header detection
		foreach ($rows as $r => $row) {
			$labels = $normRow($row);
			$tmp = [];
			foreach ($labels as $i => $lab) if ($isSizeHeader($lab)) $tmp[$i] = $this->normalizeSize($lab);
			if (count($tmp) >= 2) { $sizeHeaderIdx = $r; $sizeCols = $tmp; break; }
		}
		if ($sizeHeaderIdx < 0) return [];

		// Find a meta header row at or above the size header containing column labels for style/colour/carton
		for ($r = max(0, $sizeHeaderIdx - 3); $r <= $sizeHeaderIdx; $r++) {
			$labels = $normRow($rows[$r]);
			foreach ($labels as $i => $lab) {
				$n = strtolower(preg_replace('/\s+/', '', $lab));
				if ($indexes['carton'] < 0 && (str_contains($n, 'cartonno') || $n === 'carton')) $indexes['carton'] = $i;
				if ($indexes['style_name'] < 0 && (str_contains($n, 'stylename') || $n === 'style')) $indexes['style_name'] = $i;
				if ($indexes['style'] < 0 && (str_contains($n, 'stylenumber') || str_contains($n, 'styleno'))) $indexes['style'] = $i;
				if ($indexes['colour'] < 0 && (str_contains($n, 'colour') || str_contains($n, 'color'))) $indexes['colour'] = $i;
			}
		}

		$flat = [];
		$lastCarton = '';
		for ($i = $sizeHeaderIdx + 1; $i < count($rows); $i++) {
			$row = $rows[$i]; if (!is_array($row)) continue;
			$labels = $normRow($row);

			// Detect header resets mid-sheet: allow smaller sets (e.g., S/M/L)
			$headerResetSizes = [];
			foreach ($labels as $ci => $lab) if ($isSizeHeader($lab)) $headerResetSizes[$ci] = $this->normalizeSize($lab);
			$candidateCount = count($headerResetSizes);
			$resetDetected = false;
			if ($candidateCount >= 3) {
				$resetDetected = true;
			} elseif ($candidateCount >= 2) {
				// If candidate differs from current size header set, treat as reset
				$currentSet = array_values($sizeCols);
				$candidateSet = array_values($headerResetSizes);
				if ($currentSet !== $candidateSet) { $resetDetected = true; }
			}
			if ($resetDetected) {
				$sizeCols = $headerResetSizes; // update size columns
				// Try to detect meta indexes from the row ABOVE (which likely contains meta labels)
				$prev = $i > 0 ? $normRow($rows[$i-1]) : [];
				$foundAny = false;
				if ($prev) {
					foreach ($prev as $ci => $lab) {
						$n = strtolower(preg_replace('/\s+/', '', $lab));
						if ($indexes['carton'] < 0 && (str_contains($n, 'cartonno') || $n === 'carton')) { $indexes['carton'] = $ci; $foundAny = true; }
						if ($indexes['style_name'] < 0 && (str_contains($n, 'stylename') || $n === 'style')) { $indexes['style_name'] = $ci; $foundAny = true; }
						if ($indexes['style'] < 0 && (str_contains($n, 'stylenumber') || str_contains($n, 'styleno'))) { $indexes['style'] = $ci; $foundAny = true; }
						if ($indexes['colour'] < 0 && (str_contains($n, 'colour') || str_contains($n, 'color'))) { $indexes['colour'] = $ci; $foundAny = true; }
					}
				}
				// If still none found, preserve existing meta indexes (do not zero them out)
				$lastCarton = '';
				continue;
			}

			$style = $indexes['style'] >= 0 && isset($row[$indexes['style']]) ? trim((string)$row[$indexes['style']]) : '';
			$styleName = $indexes['style_name'] >= 0 && isset($row[$indexes['style_name']]) ? trim((string)$row[$indexes['style_name']]) : '';
			$colour = $indexes['colour'] >= 0 && isset($row[$indexes['colour']]) ? trim((string)$row[$indexes['colour']]) : '';
			$carton = $indexes['carton'] >= 0 && isset($row[$indexes['carton']]) ? trim((string)$row[$indexes['carton']]) : '';
			if ($carton === '' && $lastCarton !== '') { $carton = $lastCarton; } elseif ($carton !== '') { $lastCarton = $carton; }

			$joined = strtoupper(implode(' ', (array)$row));
			if (Str::contains($joined, ['TOTAL PRS', 'TOTAL PCS', 'TOTAL NET WT', 'TOTAL GR WT'])) break;
			if ($style === '' && $styleName === '') continue;
			if ($colour === '') continue;

			foreach ($sizeCols as $colIdx => $sizeLabel) {
				$val = isset($row[$colIdx]) ? trim((string)$row[$colIdx]) : '';
				if ($val === '' || !preg_match('/^-?\d+(?:\.\d+)?$/', str_replace(',', '', $val))) continue;
				$qty = (int) round((float) str_replace(',', '', $val));
				if ($qty <= 0) continue;
				$flat[] = [
					'carton' => $carton,
					'style' => $style,
					'style_name' => $styleName,
					'colour' => $colour,
					'colour_normalized' => $colour,
					'size' => $this->normalizeSize($sizeLabel),
					'qty' => $qty,
				];
			}
		}
		return $flat;
	}

	#[On('render')]
	public function render()
	{
		Gate::authorize('order:update');
		return view('livewire.imports.pl-mabli-import');
	}
}
