<?php

namespace App\Http\Livewire\Imports;

use App\Models\Price;
use App\Models\Sizes;
use League\Csv\Writer;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Countries;
use App\Models\Colourways;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use App\Models\StyleVersions;
use Livewire\WithFileUploads;
use App\Models\CustomerOrders;
use App\Models\CustomerAddress;
use App\Services\VertexAiService;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Gate;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ImportExcelWithValues;
use App\Models\CustomerOrderLineQuantities;

class OrdersUpload extends BaseComponent
{
    use WithFileUploads;

    protected VertexAiService $vertexAi;

    public $orderFiles = [];
    public $orderText = '';
    public $inputMethod = 'file'; // 'file' or 'text'
    public $answer;
    public $question = '';
    public $disableUploadButton = false;
    public $customer;
    public $season;
    public $files;

    public function boot(VertexAiService $vertexAi)
    {
        $this->vertexAi = $vertexAi;
    }


    public $incoterms;
    public $order_date;
    public $customer_exfty;
    public $customer_into_wh;
    public $start_knit;
    public $exfty;

    // Bulk apply selectors
    public $bulk_file_index = null;     // which file
    public $bulk_order_index = null;    // which order within file
    public $bulk_drop_index = null;     // which shipment index within a colorway (apply to all colorways in that order)

	#[Computed]
	public function customers(){
		return Customer::allCached();
	}
    #[Computed]
	public function seasons(){
		return Seasons::allCached();
	}

    #[Computed]
	public function addresses(){
		return CustomerAddress::where('customer_id', $this->customer)->get();
	}

    public function styles(){
        if (!$this->customer || !$this->season) {
            return collect();
        }
        
        return StyleVersions::with(['styles.designs'])
            ->whereHas('styles', function($query) {
                $query->where('customers_id', $this->customer)
                      ->where('seasons_id', $this->season)
                      ->where('cancelled', 0)
                      ->where('departments_id', 1);
            })
            ->where('deleted_at', null)
            ->get();
    }

    public function colourways($style_version_id){
		return Colourways::with('style_versions.styles.designs')->where('style_versions_id', $style_version_id)->get();
	}

    public function selectStyle($fileIndex, $orderIndex, $styleIndex, $styleVersionId)
    {
        if ($styleVersionId) {
            // Update the style_version_id
            $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['style_version_id'] = $styleVersionId;
            
            // Load the style model
            $styleVersionDB = StyleVersions::with('styles.designs')->find($styleVersionId);
            if ($styleVersionDB) {
                $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['model'] = $styleVersionDB;
                $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['unmatched'] = false;
            }
            
            // Clear all colorways for this style since style changed
            foreach ($this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['colorways'] as $c => $colorway) {
                $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['colorways'][$c]['colorway_id'] = null;
                $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['colorways'][$c]['unmatched'] = true;
                $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['colorways'][$c]['model'] = null;
            }
        }
    }

    public function selectColorway($fileIndex, $orderIndex, $styleIndex, $colorwayIndex, $colorwayId)
    {
        if ($colorwayId) {
            $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['colorways'][$colorwayIndex]['colorway_id'] = $colorwayId;
            $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['colorways'][$colorwayIndex]['unmatched'] = false;
            
            // Load the colorway model
            $colourwayDB = Colourways::find($colorwayId);
            if ($colourwayDB) {
                $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['colorways'][$colorwayIndex]['model'] = $colourwayDB;
            }
        }
    }

    // Validation rules
    #[Validate([
        'orderFiles.*' => ['required_if:inputMethod,file', 'file', 'mimes:csv,xls,xlsx,pdf,jpg,jpeg,png'], // Validate uploaded files
        'orderText' => ['required_if:inputMethod,text', 'string'], // Validate text input

        // Basic order validation
        'files.*.purchase_orders.*.po_number' => ['required', 'string', 'max:50'], // PO number must be a string
        'files.*.purchase_orders.*.date' => ['required', 'date'], // Order date must be valid
        'files.*.purchase_orders.*.phase' => ['nullable', 'numeric', 'min:1', 'max:2'], // Optional incoterms
        'files.*.purchase_orders.*.incoterms' => ['required', 'string', 'max:20'], // Optional incoterms
        'files.*.purchase_orders.*.delivery_address_id' => ['nullable', 'integer', 'exists:customer_addresses,id'], // Address ID must exist
        'files.*.purchase_orders.*.delivery_address.line1' => ['nullable', 'string', 'max:255'], // Optional delivery address
        'files.*.purchase_orders.*.delivery_address.line2' => ['nullable', 'string', 'max:255'],
        'files.*.purchase_orders.*.delivery_address.line3' => ['nullable', 'string', 'max:255'],
        'files.*.purchase_orders.*.delivery_address.city' => ['nullable', 'string', 'max:100'],
        'files.*.purchase_orders.*.delivery_address.county' => ['nullable', 'string', 'max:100'],
        'files.*.purchase_orders.*.delivery_address.postcode' => ['nullable', 'string', 'max:20'],
        'files.*.purchase_orders.*.delivery_address.country' => ['nullable', 'string', 'max:100'], // Must match a valid country

        // Colorway-level validation
        'files.*.purchase_orders.*.styles.*.colorways.*.colorway_id' => ['nullable', 'integer'], // Colorway ID is optional for manual selection
        'files.*.purchase_orders.*.styles.*.colorways.*.start_knit' => ['nullable', 'date'], // Colorway ID must exist

        // Shipment-level validation
        'files.*.purchase_orders.*.styles.*.colorways.*.shipments.*.customer_exfty' => ['required', 'date'], // Customer ex-factory date for each shipment
        'files.*.purchase_orders.*.styles.*.colorways.*.shipments.*.customer_into_wh' => ['required', 'date'], // Customer into warehouse date for each shipment
        'files.*.purchase_orders.*.styles.*.colorways.*.shipments.*.exfty' => ['required', 'date'], // Ex-factory date for each shipment
        'files.*.purchase_orders.*.styles.*.colorways.*.shipments.*.size_quantities.*.size' => ['required', 'string', 'max:20'], // Size is required
        'files.*.purchase_orders.*.styles.*.colorways.*.shipments.*.size_quantities.*.quantity' => ['required', 'integer', 'min:1'], // Quantity must be a positive integer
        // 'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.sku' => ['nullable', 'string', 'max:50'], // Optional SKU
        // 'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.barcode' => ['nullable', 'string', 'max:50'], // Optional barcode
        // 'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.price' => ['nullable', 'numeric', 'min:0'], // Optional price must be non-negative
        // 'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.quote' => ['nullable', 'numeric', 'min:0'], // Optional quote must be non-negative
    ])]

    public function getPrompt()
    {
        if ($this->customer) {
            $customer = Customer::find($this->customer);
            if ($customer && $customer->custom_prompt) {
                return $customer->custom_prompt;
            }
        }
        
        // Return default prompt if no custom prompt is set
        return $this->getDefaultPrompt();
    }

    private function getDefaultPrompt()
    {
        return '
        I will send JSON of colourways and styles in our database. Then I will send Customer POs in Text format.
        Match the items in the POs to the items in the database. The style name may not match entirely and may be shortened. 
        
        IMPORTANT STYLE MATCHING:
        - Look for style codes on the PO (like RTD8807, ABC123, etc.) and match them to the designs_id field in the database
        - The customer_ref field may also contain style codes that can be matched
        - If you see a code like RTD8807 on the PO, find the matching designs_id in the database data
        - Use the designs_id to determine the correct style_version_id and colorway_id
        
        You may find some match the style, but have no matching colourway - add these as usual but leave the colourway_id blank.
        Double check all data and calculations to make sure youve included line on the POs and ensured the data for each line is in the right place. E.g. Make sure the qty for size S refers to size S and arent being crossed.
        Make sure that the total qty on the PO adds up to the total of the qtys listed in the JSON.
        Data should be returned in JSON format. Return ONLY JSON - NO ADDITIONAL TEXT - so it can be ready directly by the application.
        If there are errors - return a clear error explanation.
        If no incoterm is mentioned, leave it blank.
        There will also be a list of customer addresses listed. *If the address roughly matches (e.g. same postcode and similar first line)*, include the address_id (matching to the ID IN THE DATABASE), if not then leave the delivery_address_id blank.
        Only get the Shipping Address. Do not use the billing address as the shipping address - just leave it blank is there is no address to ship to.
        If using prices to match to the right version, prioritise prices that are confirmed. In JSON use `price` as the customer purchase order price and `quote` as the value in the database.
        Ensure the JSON includes all fields from the example below.
        Remember that some POs may have one line per size - still keep the colourway as one as in below json.
        *If an order has the one colourway but many sizes, put the colourway down once in the json and list each size as in the example below.* Do not create a new colourway just because the name has a size identifyer at the end of the colourway name.
        Each order should have the designs_id, style_version_id, colorway_id if they can be possibly matched.
        - designs_id: Match this to the style code found on the PO (e.g., if PO shows "RTD8807", use the designs_id that corresponds to "RTD8807")
        - style_version_id: Use the style_version_id from the database that matches the designs_id
        - colorway_id: Use the colorway_id from the database that matches the colorway name/description
        
        customer_into_wh should be the Delivery Date.
        Remember to group the lines where styles and colourways are the same.
        CUSTOMER EXFTY and EXFTY DATES ARE LISTED IN THE PO YOU MUST ADD THESE TO THE JSON FOR THE COLOURWAY.
        The style description does not need to match exactly the description in the database. You can use the style code to match to the style in the database.
        
        DOUBLE CHECK ALL VALUES ARE CORRECT.

        There may be multiple tables. Sometimes there are drops to several places, then a summary of the complete order. 
        
        IMPORTANT: Handle multiple drops/shipments for the same colorway:
        - If a colorway has multiple drops with different ex-factory dates, create separate shipment entries
        - Each drop should have its own exfty date and quantities
        - Group by colorway but allow multiple shipments with different dates
        - Use the "shipments" array within each colorway to handle multiple drops


        {
            "purchase_orders": [
                {
                    "po_number": "123456789", //ensure this read directly from the customer PO and not the database export
                    "phase_id": "2",
                    "date": "2023-09-01",
                    "incoterms": "EXW",
                    "delivery_address_id": 1,
                    "styles": [
                        {
                            "designs_id": "1235", // This should match the style code from the PO (e.g., if PO shows "RTD8807", this should be "RTD8807")
                            "style_version_id": 6023, // This should be the style_version_id from the database that corresponds to the designs_id
                            "colorways": [
                                {
                                    "colorway_id": 4, // This should be the colorway_id from the database that matches the colorway name
                                    "shipments": [
                                        {
                                            "customer_exfty": "2023-09-15", //MUST BE ON
                                            "customer_into_wh": "2023-09-25",
                                            "exfty": "2023-09-15", // Ex-factory date for this drop
                                            "size_quantities": [
                                                { "size": "S", "quantity": 5, "sku": "123456", "barcode": "123456", "price": 10.99, "quote": 13.55 },
                                                { "size": "M", "quantity": 8, "sku": "123456", "barcode": "123456", "price": 10.99, "quote": 13.55 },
                                                { "size": "L", "quantity": 6, "sku": "123456", "barcode": "123456", "price": 10.99, "quote": 13.55 },
                                            ]
                                        },
                                        {
                                            "customer_exfty": "2023-09-15", //MUST BE ON
                                            "customer_into_wh": "2023-09-25",
                                            "exfty": "2023-09-22", // Second drop with different ex-factory date
                                            "size_quantities": [
                                                { "size": "S", "quantity": 5, "sku": "123456", "barcode": "123456", "price": 10.99, "quote": 13.55 },
                                                { "size": "M", "quantity": 7, "sku": "123456", "barcode": "123456", "price": 10.99, "quote": 13.55 },
                                                { "size": "L", "quantity": 6, "sku": "123456", "barcode": "123456", "price": 10.99, "quote": 13.55 },
                                            ]
                                        }
                                    ]
                                }
                            ]
                        }
                    ]
                },
            ],
        }

        ensure the Json is laid out as above correctly so it is displayed correctly in the table after.
        ';
    }

    public function bulkDates(){
        // Determine scope
        $targetFiles = isset($this->bulk_file_index) ? [$this->bulk_file_index] : array_keys($this->files ?? []);

        foreach($targetFiles as $f){
            if(!isset($this->files[$f])) continue;
            $orders = isset($this->bulk_order_index) ? [$this->bulk_order_index] : array_keys($this->files[$f]['purchase_orders'] ?? []);

            foreach($orders as $o){
                if(!isset($this->files[$f]['purchase_orders'][$o])) continue;

                // Apply order-level fields
                if($this->incoterms !== null)
                    $this->files[$f]['purchase_orders'][$o]['incoterms'] = $this->incoterms;
                if($this->order_date)
                    $this->files[$f]['purchase_orders'][$o]['date'] = $this->order_date;

                foreach($this->files[$f]['purchase_orders'][$o]['styles'] ?? [] as $s => $style){
                    foreach($style['colorways'] ?? [] as $c => $colorway){
                        if($this->start_knit)
                            $this->files[$f]['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['start_knit'] = $this->start_knit;

                        // Determine which drops to update
                        $shipments = $this->files[$f]['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['shipments'] ?? [];
                        $shipmentIndexes = isset($this->bulk_drop_index) ? [$this->bulk_drop_index] : array_keys($shipments);

                        foreach($shipmentIndexes as $sh){
                            if(!isset($shipments[$sh])) continue;
                            if($this->customer_exfty)
                                $this->files[$f]['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['shipments'][$sh]['customer_exfty'] = $this->customer_exfty;
                            if($this->customer_into_wh)
                                $this->files[$f]['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['shipments'][$sh]['customer_into_wh'] = $this->customer_into_wh;
                            if($this->exfty)
                                $this->files[$f]['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['shipments'][$sh]['exfty'] = $this->exfty;
                        }
                    }
                }
            }
        }
    }


    public function uploadOrders()
    {
        $this->files = [];
        $errorMessages = [];
        $successCount = 0;

        if ($this->inputMethod === 'file') {
            if (empty($this->orderFiles)) {
                session()->flash('message', 'No files uploaded.');
                return;
            }

            // Process each file individually so one failure doesn't break the batch
            foreach ($this->orderFiles as $f => $file) {
                $fileName = $file->getClientOriginalName();
                $ext = strtolower($file->getClientOriginalExtension());

                try {
                    $sendFile = [];
                    if ($ext === 'pdf') {
                        $encoded = base64_encode(file_get_contents($file->getRealPath()));
                        $sendFile[0] = [$encoded, true];
                    } elseif (in_array($ext, ['xls', 'xlsx', 'csv'])) {
                        $text = base64_encode($this->excelToText($file));
                        $sendFile[0] = [$text, false];
                    } else {
                        $errorMessages[] = "Unsupported file type: {$fileName}";
                        continue;
                    }

                    $result = $this->sendToAI($sendFile);
                    
                    if ($result && isset($result['purchase_orders']) && count($result['purchase_orders']) > 0) {
                        $this->files[] = $result;
                        $successCount++;
                    } else {
                        $errorMessages[] = "No orders found in file: {$fileName}";
                    }
                } catch (\Exception $e) {
                    Log::error('KWT Orders Upload - File processing failed', [
                        'file' => $fileName,
                        'error' => $e->getMessage(),
                    ]);
                    $errorMessages[] = "Failed to process file '{$fileName}': " . $e->getMessage();
                }
            }
        } else {
            // Text input method
            if (empty($this->orderText)) {
                session()->flash('message', 'No text content provided.');
                return;
            }

            try {
                $encodedText = base64_encode($this->orderText);
                $sendFiles[0] = [$encodedText, false];
                $result = $this->sendToAI($sendFiles);
                
                if ($result && isset($result['purchase_orders']) && count($result['purchase_orders']) > 0) {
                    $this->files[] = $result;
                    $successCount++;
                } else {
                    $errorMessages[] = "No orders found in the provided text.";
                }
            } catch (\Exception $e) {
                Log::error('KWT Orders Upload - Text processing failed', [
                    'error' => $e->getMessage(),
                ]);
                $errorMessages[] = "Failed to process text: " . $e->getMessage();
            }
        }
        
        // Reset bulk selectors when new files are loaded
        $this->bulk_file_index = null;
        $this->bulk_order_index = null;
        $this->bulk_drop_index = null;

        // Show results summary
        if (!empty($errorMessages)) {
            session()->flash('upload_errors', $errorMessages);
        }
        
        if ($successCount > 0) {
            session()->flash('upload_success', "{$successCount} file(s) processed successfully.");
        } elseif (empty($errorMessages)) {
            session()->flash('message', 'No files were processed.');
        }
    }

    private function excelToText($file): string
    {
        $rows = Excel::toArray(null, $file)[0]; // first sheet
        $lines = [];

        foreach ($rows as $row) {
            $lines[] = implode("\t", array_map('trim', $row));
        }

        return implode("\n", $lines);
    }

    private function sendToAI($files, $maxRetries = 2)
    {
        // Build prompt with database data
        $prompt = $this->getPrompt() .
            "\n\nDB Extract:" . json_encode($this->getData()) . 
            "\n\nAddresses:" . json_encode($this->getCustomerAddresses());

        // Convert files to format expected by VertexAiService
        $vertexFiles = collect($files)->map(fn ($file) => [
            'data' => $file[0],  // base64 encoded content
            'mime_type' => $file[1] ? 'application/pdf' : 'text/plain'
        ])->values()->all();

        Log::info('KWT Orders Upload - Sending to Vertex AI', [
            'file_count' => count($vertexFiles),
            'customer' => $this->customer,
            'season' => $this->season,
        ]);

        $data = null;
        $lastError = null;
        $attempts = 0;

        // Retry loop - if no orders returned, try again
        while ($attempts < $maxRetries) {
            $attempts++;
            
            try {
                $result = $this->vertexAi->extractFromMultipleFiles($prompt, $vertexFiles, 3);
                
                // Check if we got valid orders
                if ($result && isset($result['purchase_orders']) && count($result['purchase_orders']) > 0) {
                    $data = $result;
                    break; // Success - exit retry loop
                }
                
                // No orders returned, log and retry
                Log::warning('KWT Orders Upload - No orders returned, retrying', [
                    'attempt' => $attempts,
                    'max_retries' => $maxRetries,
                ]);
                $lastError = "AI returned no purchase orders";
                
            } catch (\Exception $e) {
                Log::error('KWT Orders Upload - Vertex AI error on attempt ' . $attempts, [
                    'error' => $e->getMessage(),
                ]);
                $lastError = $e->getMessage();
                
                // If this was the last attempt, don't continue
                if ($attempts >= $maxRetries) {
                    throw new \Exception("Error calling AI service after {$attempts} attempts: " . $lastError);
                }
            }
        }

        // If we still don't have data after all retries
        if (!$data || !isset($data['purchase_orders']) || count($data['purchase_orders']) === 0) {
            throw new \Exception("No purchase orders found after {$attempts} attempts. Last error: " . ($lastError ?? 'Unknown'));
        }

        // Validate and enrich the response with database models
        foreach ($data['purchase_orders'] as $o => $order) {
            foreach ($order['styles'] as $s => $style) {
                $styleVersionDB = StyleVersions::with('styles.designs')->find($style['style_version_id'] ?? null);
                if ($styleVersionDB) {
                    $data['purchase_orders'][$o]['styles'][$s]['model'] = $styleVersionDB;

                    foreach ($style['colorways'] as $c => $colorway) {
                        if (array_key_exists('colorway_id', $colorway) && $colorway['colorway_id']) {
                            $colourwayDB = Colourways::find($colorway['colorway_id']);
                            if ($colourwayDB) {
                                $data['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['model'] = $colourwayDB;
                            } else {
                                // Mark as unmatched for manual selection
                                $data['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['unmatched'] = true;
                                $data['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['colorway_id'] = null;
                            }
                        } else {
                            // Mark as unmatched for manual selection
                            $data['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['unmatched'] = true;
                            $data['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['colorway_id'] = null;
                        }
                    }
                } else {
                    // Log the error and mark style as unmatched for manual selection
                    Log::error('Style Version not found in database', [
                        'style_version_id' => $style['style_version_id'] ?? 'null',
                        'order_po' => $order['po_number'] ?? 'unknown'
                    ]);
                    
                    // Mark the style as unmatched so it can be handled manually
                    $data['purchase_orders'][$o]['styles'][$s]['unmatched'] = true;
                    $data['purchase_orders'][$o]['styles'][$s]['model'] = null;
                    
                    // Mark all colorways in this style as unmatched
                    foreach ($style['colorways'] as $c => $colorway) {
                        $data['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['unmatched'] = true;
                        $data['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['colorway_id'] = null;
                    }
                }
            }
        }

        Log::info('KWT Orders Upload - Successfully processed files', [
            'purchase_orders_count' => count($data['purchase_orders']),
            'attempts' => $attempts,
        ]);

        return $data;
    }

    public function getTotalQtyForColourway($fileIndex, $orderIndex, $styleIndex, $colourIndex): int
    {
        $colorway = $this->files[$fileIndex]['purchase_orders'][$orderIndex]['styles'][$styleIndex]['colorways'][$colourIndex];
        $total = 0;
        
        // Sum quantities from all shipments
        foreach ($colorway['shipments'] ?? [] as $shipment) {
            $total += collect($shipment['size_quantities'] ?? [])->sum('quantity');
        }
        
        return $total;
    }

    public function countSizesInOrder($order){
        $total = collect($order['styles'] ?? [])
            ->flatMap(fn($style) => $style['colorways'] ?? [])
            ->flatMap(fn($colorway) => $colorway['shipments'] ?? [])
            ->flatMap(fn($shipment) => $shipment['size_quantities'] ?? [])
            ->count();
        return $total;
    }

    public function getTotalQtyForOrder($order): int
    {
        return collect($order['styles'] ?? [])
            ->flatMap(fn($style) => $style['colorways'] ?? [])
            ->flatMap(fn($colorway) => $colorway['shipments'] ?? [])
            ->flatMap(fn($shipment) => $shipment['size_quantities'] ?? [])
            ->sum('quantity');
    }

    public function countSizesInStyle($style){
        $total = collect($style['colorways'] ?? [])
            ->flatMap(fn($colorway) => $colorway['shipments'] ?? [])
            ->flatMap(fn($shipment) => $shipment['size_quantities'] ?? [])
            ->count();
        return $total;
    }

    public function countSizesInColourway($colorway){
        $total = collect($colorway['shipments'] ?? [])
            ->flatMap(fn($shipment) => $shipment['size_quantities'] ?? [])
            ->count();
        return $total;
    }

    public function countSizesInShipment($shipment){
        return count($shipment['size_quantities'] ?? []);
    }

    #[On('render')]
    public function render()
    {
        // dd($this->files);
        Gate::authorize('order:create');
        if($this->buttonDisabled()){
            session()->flash('message', 'Max 10 files.');
            session()->flash('alert-class', 'alert-warning');
        }
        return view('livewire.imports.orders-upload');
    }

    private function buttonDisabled(){
        return count($this->orderFiles) > 10;
    }

    public function getData(){
        // Check if customer and season are set
        if (!$this->customer || !$this->season) {
            return [];
        }

        $results = DB::select("
        SELECT
            customer_ref,
            d.description AS 'style_description',
            c.name AS 'colourway',
            designs_id,
            styles_id,
            style_versions_id AS style_version_id,
            c.id AS colourways_id,
            seasons.description AS 'season',
            (SELECT
                JSON_ARRAYAGG(
                    JSON_OBJECT(
                        'id', p.id,
                        'quote', p.quote,
                        'cmt', p.cmt,
                        'quote_status', p.quote_status,
                        'created_at', p.created_at
                    )
                )
            FROM
                prices p
            WHERE
                p.style_versions_id = sv.id
            ) AS prices
        FROM
            colourways c
                JOIN
            style_versions sv ON sv.id = c.style_versions_id
                AND sv.deleted_at IS NULL
                JOIN
            styles s ON s.id = sv.styles_id
                AND s.deleted_at IS NULL
                AND s.customers_id = $this->customer
                AND s.seasons_id = $this->season
                AND s.cancelled = 0
                AND s.departments_id = 1
                JOIN
            designs d ON d.id = s.designs_id
                AND d.deleted_at IS NULL
                JOIN
            gauges g ON g.id = sv.gauge_id
                JOIN
            seasons ON seasons.id = s.seasons_id
        WHERE
            c.deleted_at IS NULL
		AND
			c.cancelled = 0
        ");

        $data = json_decode(json_encode($results), TRUE);
        
        // Return empty array instead of null if no results
        return $data ?: [];
    }
    public function getCustomerAddresses(){
        $results = DB::select("
        SELECT
            *
        FROM
            customer_addresses
        WHERE
            deleted_at IS NULL
        AND
            customer_id = " . $this->customer);

        return json_decode(json_encode($results), TRUE);
    }

    public function createOrders()
    {
        $this->validate();

        // Initialize message collections
        $errorMessages = [];
        $savedOrders = [];

        // dd($this->files);

        DB::beginTransaction();

        try {
            // First, collect all orders and group them by PO number
            // This handles cases where the AI returns multiple orders with the same PO number
            // by merging their styles into a single order to prevent duplicate key errors
            $groupedOrders = [];
            foreach ($this->files as $file) {
                foreach ($file['purchase_orders'] as $o => $order) {
                    // Validate the purchase order structure
                    if (!$this->isValidOrder($order)) {
                        $errorMessages[] = "Invalid purchase order structure for PO number: {$order['po_number']}";
                        \Log::error("Invalid purchase order structure", ['order' => $order]);
                        continue;
                    }

                    $poNumber = $order['po_number'];
                    
                    // If this PO number already exists in our grouped orders, merge the styles
                    if (isset($groupedOrders[$poNumber])) {
                        // Merge styles from the duplicate order
                        foreach ($order['styles'] as $style) {
                            $groupedOrders[$poNumber]['styles'][] = $style;
                        }
                    } else {
                        // First occurrence of this PO number
                        $groupedOrders[$poNumber] = $order;
                    }
                }
            }

            // Now process the grouped orders
            foreach ($groupedOrders as $poNumber => $order) {
                if (CustomerOrders::where('customer_po', $poNumber)->exists()) {
                    $errorMessages[] = "PO already exists: {$poNumber}";
                    \Log::error("PO already exists: {$poNumber}");
                    continue;
                }
                // dd();
                // Create the customer order
                $newOrder = new CustomerOrders;
                $newOrder->customer_po = $order['po_number'];
                $newOrder->order_date = $order['date'];
                $newOrder->incoterms = $order['incoterms'] ?? null;
                $newOrder->departments_id = 1;
                $newOrder->customers_id = $this->customer;
                $newOrder->seasons_id = $this->season;
                // $newOrder->phase_id = $order['phase_id'] ?? null;

                if (!empty($order['delivery_address_id'])) {
                    $newOrder->customer_addresses_id = $order['delivery_address_id'];
                } else {
                    // Validate delivery address structure
                    if (!$this->isValidAddress($order['delivery_address'])) {
                        $errorMessages[] = "Invalid delivery address structure for PO number: {$order['po_number']}";
                        \Log::error("Invalid delivery address structure", ['address' => $order['delivery_address']]);
                        continue;
                    }

                    $country = Countries::where('country', $order['delivery_address']['country'])->first();
                    if (!$country) {
                        $errorMessages[] = "Invalid country for address in PO number: {$order['po_number']}";
                        \Log::error("Invalid country", ['country' => $order['delivery_address']['country']]);
                        continue;
                    }

                    $address = CustomerAddress::firstOrCreate(
                        [
                            'line1' => $order['delivery_address']['line1'],
                            'postcode' => $order['delivery_address']['postcode'],
                        ],
                        [
                            'line2' => $order['delivery_address']['line2'] ?? '',
                            'line3' => $order['delivery_address']['line3'] ?? '',
                            'city' => $order['delivery_address']['city'] ?? '',
                            'county' => $order['delivery_address']['county'] ?? '',
                            'postcode' => $order['delivery_address']['postcode'] ?? '',
                            'country' => $country->id,
                        ]
                    );
                    $newOrder->customer_addresses_id = $address->id;
                }
                $newOrder->save();
                // dd();
                $savedOrders[] = $order['po_number']; // Collect saved order POs

                foreach ($order['styles'] as $style) {
                    // Skip styles that are unmatched
                    if (isset($style['unmatched']) && $style['unmatched']) {
                        $errorMessages[] = "Style not found in database for PO number: {$order['po_number']} - Style Version ID: {$style['style_version_id']}";
                        continue;
                    }
                    
                    foreach ($style['colorways'] as $colorway) {
                        // Skip colorways that are still unmatched
                        if (isset($colorway['unmatched']) && $colorway['unmatched'] && !$colorway['colorway_id']) {
                            $styleRef = $style['model']->styles->customer_ref ?? 'Unknown';
                            $errorMessages[] = "Colorway not selected for PO number: {$order['po_number']} - Style: {$styleRef}";
                            continue;
                        }
                        
                        // Create the customer order line (one per colorway)
                        $newLine = new CustomerOrderLines;
                        $newLine->customer_orders_id = $newOrder->id;
                        $newLine->colourways_id = $colorway['colorway_id'] ?? null;
                        if(isset($colorway['start_knit']) && $colorway['start_knit'])
                            $newLine->start_knit = $colorway['start_knit'] ?? null;
                        
                        // Use the first shipment's customer dates for the order line
                        $firstShipment = $colorway['shipments'][0] ?? null;
                        if ($firstShipment) {
                            $newLine->factory_cust_date = $firstShipment['customer_exfty'] ?? null;
                            $newLine->wh_cust_date = $firstShipment['customer_into_wh'] ?? null;
                        }
                        $newLine->save();

                        // Process each shipment (drop) for this colorway
                        foreach ($colorway['shipments'] ?? [] as $shipment) {
                            // Create shipment line for each drop
                            $newShipmentLine = new ShipmentLine;
                            $newShipmentLine->customer_order_lines_id = $newLine->id;
                            $newShipmentLine->exfty = $shipment['exfty'] ?? null;
                            $newShipmentLine->save();

                            // Process size quantities for this shipment
                            foreach ($shipment['size_quantities'] ?? [] as $size) {
                                if (!$this->isValidSize($size)) {
                                    $errorMessages[] = "Invalid size structure for PO number: {$order['po_number']}";
                                    \Log::error("Invalid size structure", ['size' => $size]);
                                    continue;
                                }

                                $sz = Sizes::firstOrCreate(['name' => $size['size']]);
                                
                                // Create customer order line quantities (total quantities for the colorway)
                                $existingQuantity = CustomerOrderLineQuantities::where('customer_order_lines_id', $newLine->id)
                                    ->where('sizes_id', $sz->id)
                                    ->first();
                                
                                if ($existingQuantity) {
                                    // Add to existing quantity
                                    $existingQuantity->qty += $size['quantity'] ?? 0;
                                    $existingQuantity->save();
                                } else {
                                    // Create new quantity record
                                    $newSize = new CustomerOrderLineQuantities;
                                    $newSize->customer_order_lines_id = $newLine->id;
                                    $newSize->sizes_id = $sz->id;
                                    $newSize->qty = $size['quantity'] ?? 0;
                                    $newSize->SKU = $size['sku'] ?? '';
                                    $newSize->barcode = $size['barcode'] ?? '';
                                    $newSize->save();
                                }

                                // Create shipment line sizes (quantities for this specific drop)
                                $newShipmentSize = new ShipmentLineSizes;
                                $newShipmentSize->shipment_line_id = $newShipmentLine->id;
                                $newShipmentSize->sizes_id = $sz->id;
                                $newShipmentSize->qty = $size['quantity'] ?? 0;
                                $newShipmentSize->save();

                                // Update price if needed (only once per size, not per shipment)
                                if (!isset($size['quote']) && !$existingQuantity) {
                                    Price::updateOrCreate(
                                        [
                                            'style_versions_id' => $colorway['model']['style_versions_id'],
                                            'phase_id' => $newOrder->phase_id,
                                        ],
                                        [
                                            'cmt' => $size['cmt'] ?? 0,
                                            'quote' => $size['price'],
                                            'quote_status' => 'confirmed',
                                        ]
                                    );
                                }
                            }
                        }
                    }
                }
            }

            DB::commit(); // Commit transaction if no exceptions occur

            // Flash success and error messages
            session()->flash('messages', [
                'errors' => $errorMessages,
                'saved' => $savedOrders,
            ]);

            $this->resetExcept([
                'customer',
                'season',
                'inputMethod',
                'incoterms',
                'order_date',
                'customer_exfty',
                'customer_into_wh',
                'start_knit',
                'exfty',
        ]);
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback transaction on error
            // dd($e->getMessage());
            \Log::error('Error creating orders', ['error' => $e->getMessage()]);
            session()->flash('messages', [
                'errors' => array_merge(
                    $errorMessages,
                    [$e->getMessage(), 'An unexpected error occurred while processing orders.']
                ),
            ]);
        }
    }

    private function isValidOrder($order)
    {
        return isset($order['po_number'], $order['date'], $order['styles']);
    }

    private function isValidAddress($address)
    {
        return isset($address['line1'], $address['postcode'], $address['country']);
    }

    private function isValidSize($size)
    {
        return isset($size['size'], $size['quantity']);
    }

}
