<?php

namespace App\Http\Livewire\Imports;

use App\Models\Gauge;
use App\Models\Price;
use App\Models\Sizes;
use App\Models\Design;
use App\Models\Styles;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Models\Departments;
use App\Models\ShipmentLine;
use App\Models\StyleVersions;
use App\Models\CommodityCodes;
use App\Models\CustomerOrders;
use Illuminate\Support\Carbon;
use App\Models\CustomerAddress;
use App\Models\ShipmentLineSizes;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use App\Models\CustomerOrderLineQuantities;

class Orders extends Component
{
	public $imported;
    public function render()
    {
        // $this->importOrders();
        return view('livewire.imports.orders');
    }

    public function getCSV()
	{
		$csvFile = public_path('csv/' . "import.csv");
		$array = array('delimiter' => ',');
		$file_handle = fopen($csvFile, 'r');
		while (!feof($file_handle)) {
			$line_of_text[] = fgetcsv($file_handle, 0, $array['delimiter']);
		}
		fclose($file_handle);
		return $line_of_text;
	}
    public function importOrders(){
        Gate::check('adminONLY');

		$csv = array_map("str_getcsv", file(public_path('csv/' . "import.csv"),FILE_SKIP_EMPTY_LINES));
		$keys = array_shift($csv);
		foreach ($csv as $i=>$row) {
			$csv[$i] = (object) array_combine($keys, $row);
		}
		$csv = collect($csv);

		$errors = [];
        foreach($csv as $l=>$line){ //VERIFY DATA
			foreach($line as $i=>$item){
				$line->$i = trim(strip_tags($line->$i));
			}

			$sizes = array_slice((array)$line, 41);

			$validator = Validator::make(
				(array) $line,
				[
				'design_id' => 'nullable|numeric|min:0',
				'gauge' => 'nullable|string|max:10',
				'date_of_issue' => 'nullable|date',
				'composition' => 'nullable|string|max:50',
				'factory_name' => 'nullable|string|max:35',
				'customer_name' => 'nullable|string|max:35',
				'description' => 'nullable|string|max:50',
				'season_description' => 'nullable|string|max:4',
				'department_description' => 'nullable|string|max:30',
				'customer_ref' => 'nullable|string|max:35',
				'commodity_code_id' => 'nullable|numeric',
				'category' => 'nullable|in:mens,ladies,childrens,accessories',
				'style_version_name' => 'nullable|string|max:35',
				'colourways_name' => 'nullable|string|max:35',
				'colour_type' => 'nullable|in:solid,melange,mixed',
				'customer_po' => 'nullable|string|max:35',
				'order_date' => 'nullable|date',
				'incoterms' => 'nullable|in:EXW,FCA,CPT,CIP,DAP,DPU,DDP,FAS,FOB,CFR,CIF',
				'cust_ship_no' => 'nullable|string|max:35',
				'customer_address_name' => 'nullable|string|max:35',
				'incomplete_flag' => 'nullable|boolean',
				'asn' => 'nullable|string|max:35',
				'sent_to_factory' => 'nullable|date',
				'factory_cust_date' => 'nullable|date',
				'colour_approved_date' => 'nullable|date',
				'lab_dips_sent_date' => 'nullable|date',
				'colour_approval_comments' => 'nullable|string|max:50',
				'wh_cust_date' => 'nullable|date',
				'start_knit' => 'nullable|date',
				'exfty' => 'nullable|date',
				'price_model' => 'nullable|in:manual,standard,full_factored',
				'quote' => 'nullable|numeric',
				'quote_status' => 'nullable|in:not_confirmed,confirmed,estimate',
				'cmt' => 'nullable|numeric',
				'cmt_status' => 'nullable|in:not_confirmed,confirmed,estimate',
				]
			);
			if ($validator->fails()) {
				foreach($validator->errors()->all() as $error){
					$validationErrors[] = ["critical", "Line " . $l . ": " . $error];
				}
			}

			$validator = Validator::make(
				(array) $sizes,
				['sizes.*' => 'nullable|string|max:15']
			);
			if ($validator->fails()) {
				foreach($validator->errors()->all() as $error){
					$validationErrors[] = ["critical", "Line " . $l . ": " . $error];
				}
			}
			if(empty($validationErrors)){
				$errors = array_merge($errors, $validationErrors ?? []);
				$errors = array_merge($errors, $this->verifyBeforeImportLine($line, $l));
			}
        }


		if(empty($errors)){
			$this->importData($csv);
			$this->imported = TRUE;
		}
		else{
			throw new \Exception("Import failed: " . collect($errors)->map(fn($e) => $e[1])->implode(', '));
		}
    }

	private function verifyBeforeImportLine($line, $l){
		$errors = [];
		if(empty($line->design_id) || ($design = Design::find($line->design_id)) == NULL){
			if(empty($line->gauge)){
				$errors[] = ["critical", "Line " . $l . ": If no design_no entered, gauge must be filled in correctly."];
			}
			if(empty($line->composition)){
				$errors[] = ["critical", "Line " . $l . ": If no design_no entered, composition must be filled in correctly."];
			}
			if(empty($line->description)){
				$errors[] = ["critical", "Line " . $l . ": If no design_no entered, description must be filled in correctly."];
			}
			if(empty($line->factory_name) || ($factory = Suppliers::where('name', $line->factory_name)->first()) == NULL){
				$errors[] = ["critical", "Line " . $l . ": If no design_no entered, factory_name must be filled in correctly."];
			}
			if(empty($line->customer_name) || ($customer = Customer::where('name', $line->customer_name)->first()) == NULL){
				$errors[] = ["critical", "Line " . $l . ": If no design_no entered, customer_name must be filled in correctly."];
			}
		}
		elseif(($design = Design::find($line->design_id)) == NULL){
				$errors[] = ["critical", "Line " . $l . ": design_no not found."];
		}

		if(empty($line->season_description) || ($season = Seasons::where('description', $line->season_description)->first()) == NULL){
			$errors[] = ["critical", "Line " . $l . ": season_description not found."];
		}
		if(empty($line->department_description) || ($department = Departments::where('description', $line->department_description)->first()) == NULL){
			$errors[] = ["critical", "Line " . $l . ": department_description not found."];
		}

		//STYLE
		if(empty($design) || (!empty($design) && ($style = Styles::where('customers_id', $design->customers_id)->where('designs_id', $design->id)->where('seasons_id', $season->id)->where('departments_id', $department->id)->first())) == NULL){ //IF A DESIGN IS NOT THERE OR IS NOT A STYLE WITHIN DEPT AND SEASON
			if(empty($line->commodity_code_id) || ($commodityCode = CommodityCodes::find($line->commodity_code_id)->first()) == NULL){
				$errors[] = ["critical", "Line " . $l . ": If no style found, commodity_code_id must be filled in correctly."];
			}
			if(empty($line->category) || (strcasecmp("ladies", $line->category) < 0 && strcasecmp("mens", $line->category) < 0 && strcasecmp("childrens", $line->category) < 0 && strcasecmp("accessories", $line->category < 0)) ){
				$errors[] = ["critical", "Line " . $l . ": If no style found, category must be filled in correctly."];
			}
		}

		//STYLE VERSION
		if(!empty($style) && ($styleVersion = StyleVersions::where('name', $line->style_version_name)->first()) == NULL){
			$styleVersion = $style->style_versions->first();
		}


		//COLOURWAY
		if(empty($style) || (!empty($style) && ($colourway = Colourways::where('name', $line->colourways_name)->first()) == NULL)){
			if(empty($line->colour_type)){
				$errors[] = ["critical", "Line " . $l . ": If no colourway found, colour_type must be filled in correctly. (" . $line->colourways_name . ")"];
			}
		}
		return $errors;
	}

	private function importData($csv){
		foreach($csv as $l=>$line){ //IMPORT DATA
			foreach($line as $i=>$item){
				$line->$i = trim(strip_tags($line->$i));
			}

			$sizes = array_slice((array)$line, 41, null, true);

			//GET RE-USED VALUES
			$factory = Suppliers::where('name', $line->factory_name)->first();
			$customer = Customer::where('name', $line->customer_name)->first();
            if(!empty($line->gauge))
			    $gauge = Gauge::firstOrCreate(['gauge' => $line->gauge]);
            if(!empty($line->customer_address_name))
			    $customerAddress = CustomerAddress::firstOrCreate(['name' => $line->customer_address_name, 'customer_id' => $customer->id], ['line1' => "na", 'countries_id' => $customer->countries_id]);
			$season = Seasons::where('description', $line->season_description)->first();
			$department = Departments::where('description', $line->department_description)->first();



			//CREATE/FIND DESIGN
            if(($design = Design::find($line->design_id)) == NULL){
                $design = Design::firstOrCreate(
                    [
                        'factory_id' => $factory->id ?? $design->factory_id,
                        'customers_id' => $customer->id ?? $design->customers_id,
                        'description' => $line->description,
                    ],
                    [
                        'gauge_id' => $gauge->id,
                        'issued' => !empty($line->date_of_issue) ? Carbon::createFromFormat('d/m/Y', $line->date_of_issue)->format('Y-m-d') : NULL,
                        'yarn' => $line->composition,
                    ]
                );
            }

			//CREATE/FIND STYLE
			$style = Styles::firstOrCreate(
				[
					'designs_id' => $design->id,
					'seasons_id' => $season->id,
					'departments_id' => $department->id,
					'customers_id' => $customer->id ?? $design->customers_id,
				],
				[
					'customer_ref' => $line->customer_ref,
					'commodity_codes_id' => $line->commodity_code_id,
					'category' => $line->category,
				]
			);
            // dd($style->load('style_versions.colourways'));

            $styleVersion = $style->style_versions->where('name', $line->style_version_name)->first();
            if($styleVersion == NULL){
                $styleVersion = $style->style_versions->first();
            }

			//CREATE/FIND STYLE VERSION
            if($styleVersion == NULL)
			$styleVersion = StyleVersions::firstOrCreate(
				[
					'name' => $line->style_version_name,
					'styles_id' => $style->id,
					'factory_id' => $factory->id ?? $design->factory_id,
				],
				[
					'gauge_id' => $gauge->id ?? $design->gauge_id,
				]
			);

			//CREATE/FIND COLOURWAY VERSION
            if(($colourway = Colourways::where('style_versions_id', $styleVersion->id)->where('name', 'like', "%{$line->colourways_name}%")->first()) == NULL){

                $colourway = Colourways::firstOrCreate(
                    [
                        'name' => $line->colourways_name,
                        'style_versions_id' => $styleVersion->id,
                    ],
                    [
                        'colour_type' => empty($line->colour_type) ? 'solid' : $line->colour_type,
                    ]
                );
            }


			//CREATE/FIND ORDER
			$order = CustomerOrders::firstOrCreate(
				[
					'customer_po' => $line->customer_po,
					'seasons_id' => $season->id,
					'departments_id' => $department->id,
					'customers_id' => $customer->id ?? $design->customers_id,
				],
				[
					'order_date' => Carbon::createFromFormat('d/m/Y', $line->order_date)->format('Y-m-d'),
					'incoterms' => $line->incoterms,
					'cust_ship_no' => $line->cust_ship_no ?? NULL,
					'customer_addresses_id' => $customerAddress->id ?? NULL,
					'incomplete_flag' => !empty($line->incomplete_flag) ? $line->incomplete_flag : 0,
					'asn' => $line->asn ?? NULL,
				]
			);

			//CREATE ORDER LINE
			$orderLine = CustomerOrderLines::firstOrCreate(
				[
					'customer_orders_id' => $order->id,
					'colourways_id' => $colourway->id,
				],
				[
					'order_sent_factory' => !empty($line->sent_to_factory) ? Carbon::createFromFormat('d/m/Y', $line->sent_to_factory)->format('Y-m-d') : NULL,
					'factory_cust_date' => !empty($line->factory_cust_date) ? Carbon::createFromFormat('d/m/Y', $line->factory_cust_date)->format('Y-m-d') : NULL,
					'colour_approved_date' => !empty($line->colour_approved_date) ? Carbon::createFromFormat('d/m/Y', $line->colour_approved_date)->format('Y-m-d') : NULL,
					'lab_dips_sent_date' => !empty($line->lab_dips_sent_date) ? Carbon::createFromFormat('d/m/Y', $line->lab_dips_sent_date)->format('Y-m-d') : NULL,
					'colour_approval_comments' => $line->colour_approval_comments,
					'wh_cust_date' => !empty($line->wh_cust_date) ? Carbon::createFromFormat('d/m/Y', $line->wh_cust_date)->format('Y-m-d') : NULL,
					'start_knit' => !empty($line->start_knit) ? Carbon::createFromFormat('d/m/Y', $line->start_knit)->format('Y-m-d') : NULL,
				]
			);
			$orderLine->save();

			foreach($sizes as $s=>$qty){

                if(!empty($qty)){
                    $sizeID = Sizes::firstOrCreate(['name' => (string) $s])->id;
                    // if(empty($sizeID));
                    //     dd($sizes, $s, $qty, $sizeID, Sizes::firstOrCreate(['name' => (string) $s]), Sizes::where('name', (string) $s)->first());
                    $orderLineSize = CustomerOrderLineQuantities::firstOrNew(
                        [
                            'sizes_id' => $sizeID,
                            'customer_order_lines_id' => $orderLine->id,
                        ],
                        ['qty' => !empty($qty) ? $qty : 0,]
                    );
                    $orderLineSize->save();
                }
			}

			$drop = ShipmentLine::firstOrCreate(
				[
					'customer_order_lines_id' => $orderLine->id,
					'exfty' => Carbon::createFromFormat('d/m/Y', $line->exfty)->format('Y-m-d'),
				],
				[
					'truck_plates' => $line->truck_plates,
					'collection_date' => !empty($line->collection_date) ? Carbon::createFromFormat('d/m/Y', $line->collection_date)->format('Y-m-d') : NULL,
					'delivery_date' => !empty($line->delivery_date) ? Carbon::createFromFormat('d/m/Y', $line->delivery_date)->format('Y-m-d') : NULL,
					'complete' => !empty($line->complete) ? $line->complete : 0,
					'sealer_deadline' => !empty($line->sealer_deadline) ? Carbon::createFromFormat('d/m/Y', $line->sealer_deadline)->format('Y-m-d') : NULL,
					'notes' => $line->notes,
				]
			);
			$drop->save();

			foreach($sizes as $s=>$qty){
                if(!empty($qty)){
                    $sizeID = Sizes::firstOrCreate(['name' => (string) $s])->id;
                    $shipmentLineSize = ShipmentLineSizes::firstOrCreate(
                        [
                            'sizes_id' => $sizeID,
                            'shipment_line_id' => $drop->id,
                        ],
                        ['qty' => !empty($qty) ? $qty : 0,]
                    );
                    $shipmentLineSize->save();
                }
			}

            if(!empty($line->cmt) || !empty($line->quote)){
                $price = new Price;
                $price->fill([
                    'style_versions_id' => $styleVersion->id,
                    'model' => $line->price_model,
                    'cmt' => $line->cmt,
                    'cmt_status' => $line->cmt_status,
                    'quote' => $line->quote,
                    'quote_status' => $line->quote_status,
                ]);
                $price->save();
            }


		}
	}
}
