<?php

namespace App\Http\Livewire\Imports;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\Customer;
use App\Models\Seasons;
use App\Models\Styles;
use App\Models\StyleVersions;
use App\Models\Colourways;
use App\Models\CustomerOrders;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrderLineQuantities;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Models\Sizes;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;

class OrderImportSimple extends Component
{
    use WithFileUploads;

    public $csv;
    public $season_id;
    public $customer_id;

    public $preview = [];
    public $errorsList = [];
    private $delimiter = ",";

    public function mount()
    {
        Gate::authorize('order:create');
    }

    public function getSeasonsProperty()
    {
        return Seasons::allCached();
    }

    public function getCustomersProperty()
    {
        return Customer::allCached();
    }

    public function updatedCsv()
    {
        $this->reset(['preview', 'errorsList']);
    }

    public function process()
    {
        $this->errorsList = [];
        if (!$this->season_id || !$this->customer_id) {
            $this->errorsList[] = 'Select season and customer before processing.';
            return;
        }
        if (!$this->csv) {
            $this->errorsList[] = 'Please choose a CSV file first.';
            return;
        }
        $this->parseCsv();
    }

    private function parseCsv(): void
    {
        $path = $this->csv->getRealPath();
        // detect delimiter from first line
        $this->delimiter = $this->detectDelimiter($path);
        $handle = fopen($path, 'r');
        if (!$handle) {
            $this->errorsList[] = 'Unable to open file.';
            return;
        }

        $header = fgetcsv($handle, 0, $this->delimiter);
        if (!$header) {
            $this->errorsList[] = 'CSV appears to be empty.';
            return;
        }
        // Clean header names (handle UTF-8 BOM and stray control chars)
        $header = array_map(fn($h) => $this->cleanHeaderName($h), $header);
        // expected minimal headers: PO,Colour,Size,Customer Ref,RTD,SUPP REF 1,SUPP REF 2,Qty,Exfty,Customer Exfty,Into W/H
        $indexes = $this->mapHeaderIndexes($header);

        // Validate required headers exist
        $required = ['PO','Colour','Customer Ref','RTD','Qty'];
        $missing = [];
        foreach ($required as $req) {
            if (!isset($indexes[$req]) || $indexes[$req] === null) {
                $missing[] = $req;
            }
        }
        if (!empty($missing)) {
            $this->errorsList[] = 'Missing required header(s): ' . implode(', ', $missing) . ' | Found headers: ' . implode(', ', $header) . ' | Using delimiter: ' . ($this->delimiter === "\t" ? 'TAB' : $this->delimiter);
            return;
        }

        $orders = [];
        while (($row = fgetcsv($handle, 0, $this->delimiter)) !== false) {
            if (count(array_filter($row, fn($v) => $v !== null && $v !== '')) === 0) {
                continue;
            }
            $po = trim($row[$indexes['PO']] ?? '');
            $colourName = trim($row[$indexes['Colour']] ?? '');
            $sizeName = trim((string)($row[$indexes['Size']] ?? ''));
            $customerRef = trim($row[$indexes['Customer Ref']] ?? '');
            $rtd = trim((string)($row[$indexes['RTD']] ?? '')); // numeric RT design id
            $qty = (int) trim((string)($row[$indexes['Qty']] ?? '0'));
            $exfty = isset($indexes['Exfty']) ? trim($row[$indexes['Exfty']] ?? '') : '';
            $custExfty = isset($indexes['Customer Exfty']) ? trim($row[$indexes['Customer Exfty']] ?? '') : '';
            $intoWh = isset($indexes['Into W/H']) ? trim($row[$indexes['Into W/H']] ?? '') : '';

            if (empty($po) || empty($customerRef) || empty($rtd) || empty($colourName) || empty($qty)) {
                // skip incomplete lines
                continue;
            }

            $order =& $orders[$po];
            if (!isset($order)) {
                $order = [
                    'po' => $po,
                    'order_date' => Carbon::now()->format('Y-m-d'),
                    'incoterms' => 'EXW',
                    'styles' => [],
                ];
            }

            $styleKey = $rtd . '|' . $customerRef;
            if (!isset($order['styles'][$styleKey])) {
                $order['styles'][$styleKey] = [
                    'rtd' => $rtd,
                    'customer_ref' => $customerRef,
                    'matches' => $this->matchStyle((int)$rtd, $customerRef),
                    'colourways' => [],
                ];
            }

            $colourKey = strtolower($colourName);
            if (!isset($order['styles'][$styleKey]['colourways'][$colourKey])) {
                $order['styles'][$styleKey]['colourways'][$colourKey] = [
                    'name' => $colourName,
                    'match' => $this->matchColourway($order['styles'][$styleKey]['matches'], $colourName),
                    'exfty' => $this->parseDate($exfty),
                    'customer_exfty' => $this->parseDate($custExfty),
                    'into_wh' => $this->parseDate($intoWh),
                    'sizes' => [],
                ];
            }

            if (!empty($sizeName)) {
                $order['styles'][$styleKey]['colourways'][$colourKey]['sizes'][$sizeName] = ($order['styles'][$styleKey]['colourways'][$colourKey]['sizes'][$sizeName] ?? 0) + $qty;
            } else {
                $order['styles'][$styleKey]['colourways'][$colourKey]['sizes']['One Size'] = ($order['styles'][$styleKey]['colourways'][$colourKey]['sizes']['One Size'] ?? 0) + $qty;
            }
        }
        fclose($handle);

        // normalize for preview
        $this->preview = collect($orders)->map(function ($order) {
            $order['styles'] = collect($order['styles'])->map(function ($style) {
                $style['colourways'] = collect($style['colourways'])->map(function ($cw) {
                    $cw['sizes'] = collect($cw['sizes'])
                        ->map(fn($q, $s) => ['size' => (string)$s, 'qty' => (int)$q])
                        ->values()
                        ->all();
                    return $cw;
                })->values()->all();
                return $style;
            })->values()->all();
            return $order;
        })->values()->all();
    }

    private function cleanHeaderName($name): string
    {
        if ($name === null) return '';
        // remove UTF-8 BOM and control chars, then trim
        $name = (string)$name;
        // Common BOM encodings at start
        $name = preg_replace('/^\xEF\xBB\xBF/u', '', $name); // UTF-8 BOM bytes
        $name = preg_replace('/^\x{FEFF}/u', '', $name);       // Unicode BOM char
        $name = preg_replace('/[\x00-\x1F\x7F]/u', '', $name);
        // normalize unicode whitespace to space
        $name = preg_replace('/\x{00A0}|\x{1680}|\x{180E}|\x{2000}|\x{2001}|\x{2002}|\x{2003}|\x{2004}|\x{2005}|\x{2006}|\x{2007}|\x{2008}|\x{2009}|\x{200A}|\x{202F}|\x{205F}|\x{3000}/u', ' ', $name);
        // remove wrapping quotes
        $name = trim($name, " \"'\t\n\r");
        // collapse internal multiple spaces
        $name = preg_replace('/\s+/', ' ', $name);
        return trim($name);
    }

    private function mapHeaderIndexes(array $header): array
    {
        $map = [
            'PO' => null,
            'Colour' => null,
            'Size' => null,
            'Customer Ref' => null,
            'RTD' => null,
            'Qty' => null,
            'Exfty' => null,
            'Customer Exfty' => null,
            'Into W/H' => null,
        ];
        // synonyms (case-insensitive)
        $synonyms = [
            'PO' => ['PO', 'PO Number', 'PO No', 'PO No.', 'PO#', 'Order No', 'Order Number', 'Purchase Order'],
            'Colour' => ['Colour', 'Color', 'Colourway', 'Colour Way', 'Colorway'],
            'Size' => ['Size'],
            'Customer Ref' => ['Customer Ref', 'Cust Ref', 'Customer Reference', 'Style Ref', 'Customer Style', 'Style Name'],
            'RTD' => ['RTD', 'RT No', 'RT No.', 'RT Number', 'RTD Number', 'RT'],
            'Qty' => ['Qty', 'Quantity', 'QTY'],
            'Exfty' => ['Exfty', 'Ex-Factory', 'Ex Factory', 'EXFTY'],
            'Customer Exfty' => ['Customer Exfty', 'Customer EXFTY', 'Customer Ex-Factory', 'Customer Ex Factory'],
            'Into W/H' => ['Into W/H', 'Into WH', 'Into Warehouse', 'Into W H'],
        ];
        foreach ($header as $i => $name) {
            $key = $this->cleanHeaderName($name);
            foreach ($synonyms as $std => $alts) {
                foreach ($alts as $alt) {
                    if (strcasecmp($key, $alt) === 0) {
                        $map[$std] = $i;
                        break 2;
                    }
                }
            }
            if (array_key_exists($key, $map)) {
                $map[$key] = $i;
            }
        }
        return $map;
    }

    private function detectDelimiter(string $path): string
    {
        $sample = @fopen($path, 'r');
        if (!$sample) return ',';
        $line = fgets($sample, 4096) ?: '';
        fclose($sample);
        $delims = ["," => substr_count($line, ","), ";" => substr_count($line, ";"), "\t" => substr_count($line, "\t"), "|" => substr_count($line, "|")];
        arsort($delims);
        $best = array_key_first($delims);
        return $delims[$best] > 0 ? $best : ',';
    }

    private function parseDate(?string $value): ?string
    {
        $value = trim((string)$value);
        if (empty($value)) return null;
        // try d/m/Y and d-m-Y
        foreach (['d/m/Y', 'd-m-Y', 'Y-m-d'] as $fmt) {
            try {
                $dt = Carbon::createFromFormat($fmt, $value);
                if ($dt) return $dt->format('Y-m-d');
            } catch (\Exception $e) {
            }
        }
        return null;
    }

    private function matchStyle(int $rtd, string $customerRef)
    {
        // RTD is designs_id, match styles by design id + season + customer
        return Styles::with('style_versions')
            ->where('designs_id', $rtd)
            ->where('seasons_id', $this->season_id)
            ->where('customers_id', $this->customer_id)
            ->first();
    }

    private function matchColourway($style, string $colourName)
    {
        if (!$style) return null;
        $sv = $style->style_versions->first();
        if (!$sv) return null;
        $match = Colourways::where('style_versions_id', $sv->id)
            ->where('name', 'like', "%{$colourName}%")
            ->first();
        return $match;
    }

    public function save()
    {
        $this->validate([
            'season_id' => 'required|exists:seasons,id',
            'customer_id' => 'required|exists:customers,id',
            'preview' => 'required|array|min:1',
        ]);

        // Basic per-order validation
        foreach ($this->preview as $idx => $order) {
            if (empty($order['incoterms'])) {
                session()->flash('message', 'Order ' . ($order['po'] ?? '#'.($idx+1)) . ': incoterms is required.');
                return;
            }
            if (empty($order['order_date'])) {
                session()->flash('message', 'Order ' . ($order['po'] ?? '#'.($idx+1)) . ': order date is required.');
                return;
            }
        }

        DB::beginTransaction();
        try {
            foreach ($this->preview as $order) {
                if (CustomerOrders::where('customer_po', $order['po'])->exists()) {
                    continue;
                }
                $newOrder = new CustomerOrders;
                $newOrder->customer_po = $order['po'];
                $newOrder->order_date = $order['order_date'];
                $newOrder->incoterms = $order['incoterms'];
                $newOrder->departments_id = 2; // default department if needed
                $newOrder->customers_id = $this->customer_id;
                $newOrder->seasons_id = $this->season_id;
                $newOrder->save();

                foreach ($order['styles'] as $style) {
                    $styleModel = $this->matchStyle((int)$style['rtd'], $style['customer_ref']);
                    $styleVersion = $styleModel?->style_versions?->first();
                    foreach ($style['colourways'] as $cw) {
                        $cwModel = $this->matchColourway($styleModel, $cw['name']);

                        $line = new CustomerOrderLines;
                        $line->customer_orders_id = $newOrder->id;
                        $line->colourways_id = $cwModel?->id;
                        $line->factory_cust_date = $cw['customer_exfty'] ?? null;
                        $line->wh_cust_date = $cw['into_wh'] ?? null;
                        $line->save();

                        foreach ($cw['sizes'] as $sz) {
                            $size = Sizes::firstOrCreate(['name' => (string)$sz['size']]);
                            $q = new CustomerOrderLineQuantities;
                            $q->customer_order_lines_id = $line->id;
                            $q->sizes_id = $size->id;
                            $q->qty = (int)$sz['qty'];
                            $q->save();
                        }

                        $drop = new ShipmentLine;
                        $drop->customer_order_lines_id = $line->id;
                        $drop->exfty = $cw['exfty'] ?? null;
                        $drop->save();

                        foreach ($cw['sizes'] as $sz) {
                            $size = Sizes::firstOrCreate(['name' => (string)$sz['size']]);
                            $sl = new ShipmentLineSizes;
                            $sl->shipment_line_id = $drop->id;
                            $sl->sizes_id = $size->id;
                            $sl->qty = (int)$sz['qty'];
                            $sl->save();
                        }
                    }
                }
            }

            DB::commit();
            session()->flash('message', 'Orders saved.');
        } catch (\Throwable $e) {
            DB::rollBack();
            session()->flash('message', 'Error saving: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.imports.order-import-simple');
    }
}


