<?php

namespace App\Http\Livewire\Imports;

use Gemini;
use OpenAI;
use App\Models\Price;
use App\Models\Sizes;
use Gemini\Data\Blob;
use League\Csv\Writer;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Countries;
use Spatie\PdfToText\Pdf;
use App\Models\Colourways;
use Gemini\Enums\MimeType;
use App\Facades\ZohoFacade;
use App\Models\Departments;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Smalot\PdfParser\Parser;
use App\Models\StyleVersions;
use App\Services\ZohoService;
use Livewire\WithFileUploads;
use App\Models\CustomerOrders;
use Illuminate\Support\Carbon;
use App\Models\CustomerAddress;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Gate;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ImportExcelWithValues;
use App\Services\Zoho\ReferenceService;
use Spatie\PdfToImage\Pdf as PdfToImage;
use App\Http\Livewire\Traits\HandlesZohoOAuth;
use App\Models\CustomerOrderLineQuantities;

class ErdosImportAi extends BaseComponent
{
    use WithFileUploads;
    use HandlesZohoOAuth;

    public $orderFiles = [];
    public $answer;
    public $question = '';
    public $disableUploadButton = false;
    public $customer;
    public $season;
    public $department;
    public $invoices;
    public $rtinvoice;
    public $manualSelections = [];


    public $org = '20094201933';
    public $cust = '';
    public $template = '479608000000045843';
    public $account = '479608000000045228';
    public $reportingTagDepartment = '479608000000063057';
    public $reportingTagCustomer;
    public $reportingTagCommissionsCustomer;
    public $vat;

    // Validation rules
    #[Validate([

    ])]

    public $prompt =
    '
        TASK
        You are a strict data-extraction agent.
        Read every page of the attached invoice(s) and return ONE JSON object
        that matches exactly the schema shown below—no Markdown, no commentary.

        RULES – FOLLOW IN ORDER OF PRIORITY
        1. Use ONLY the authoritative data blocks: the table that lists each line item
        and the header/footer fields that carry the official invoice metadata.
        ✘ Ignore any “Notes”, “Special Instructions”, “SAY: …”, or other prose.
        ✘ Ignore carton-label blocks and packing-list sections.
        2. If a field appears more than once, choose the instance that sits in the
        Invoice Header (top) or Totals block (bottom)—whichever is labelled for
        that field.
        • Example: if the PO number shows in header and again in a footer note,
            take the header value.
        3. Numeric checks:
        • `total_value` **must equal** Σ(line-quantity × unit_price).
        • If they disagree by ≥ 0.01, raise `"error": "total mismatch"` instead of data.
        4. Omit any invoice lines where quantity = 0 or the description is blank.
        5. Output currency and thousands separators exactly as printed
        (“25,542.00” → keep the comma).
        6. Dates → ISO format `YYYY-MM-DD` (convert if source is `DD/MM/YYYY` etc.).
        7. Return **only** the JSON—no extra keys, no surrounding text.
        8. If no season found, base it on the date - e.g. AW25 - autumn/winter OR SS25 - spring/summer. DO NOT LEAVE BLANK.
        9. Include a simple colour name per line (e.g. "Crimson", "Chocolate"). If multiple appear, pick the one nearest the product/line entry.

        SCHEMA – copy verbatim
        [
            {
                "invoice_no": "string",       // e.g. "OZB254802-1A-AW25"
                "date": "YYYY-MM-DD",         // invoice date
                "po_number": "string",        // main PO #
                "season": "string",           // e.g. "AW25"
                "total_value": "#####.##",   // as printed - do not use commas so it is seen as a number not string
                "lines": [
                    {
                        "product_id": "string",   // e.g. "K0834"
                        "colour": "string",       // e.g. "Crimson"
                        "quantity": integer,
                        "unit_price": "####.##" //do not use commas so it is seen as a number not string
                        "line_total": "####.##" //do not use commas so it is seen as a number not string
                    }
                ]
            },
        ]
    ';

    public bool $zohoConnected = false;
    public string $authUrl = '';
    private $zoho;

    public function organisations(){
        return $this->zoho->getOrganisations();
    }

    public function customers(){
        return $this->zoho->getCustomers($this->org);
    }

    public function getReportingTagId($name){
        $tag = $this->zoho->getReportingTags($this->org);
        $tag_id = collect($tag)->where('tag_name', $name)?->first()['tag_id'] ?? "";
        return $tag_id;
    }

    public function getInvoiceCustomFieldId($name){
        $fields = $this->zoho->getCustomFields($this->org);
        $id = collect($fields['invoice'])->firstWhere('label', $name)['customfield_id'] ?? "";
        return $id;
    }

    public function reportingTagCustomers(){
        $tag_id = $this->getReportingTagId('Customer');
        $tag_values = $this->zoho->getReportingTagOptions($this->org, $tag_id);

        return $tag_values;
    }

    public function reportingTagDepartments(){
        $tag_id = $this->getReportingTagId('Department');
        $tag_values = $this->zoho->getReportingTagOptions($this->org, $tag_id);

        return $tag_values;
    }

    public function accounts(){
        return $this->zoho->getChartOfAccounts($this->org);
    }

    public function templates(){
        return $this->zoho->getInvoiceTemplates($this->org);
    }

    public function uploadOrders()
    {
        // $invoices = json_decode(
        // '[{"invoice_no":"OZB254802-1A-AW25","date":"2025-06-02","po_number":"65972603","season":"AW25","total_value":"25542.00","lines":[{"product_id":"K0834","quantity":660,"unit_price":"38.70","line_total":"25542.00"}]},{"invoice_no":"OZB254804-1A-AW25","date":"2025-06-02","po_number":"65975803","season":"AW25","total_value":"6472.50","lines":[{"product_id":"K0862","quantity":150,"unit_price":"43.15","line_total":"6472.50"}]}]'
        // , 1);

        // $this->invoices = $this->processInvoices($invoices);
        // return 1;

        if (empty($this->orderFiles)) {
            session()->flash('message', 'No files uploaded.');
            return;
        }

        $this->files = [];
        foreach ($this->orderFiles as $idx => $file) {
            $ext = strtolower($file->getClientOriginalExtension());

            if (in_array($ext, ['pdf'])) {
                $plain   = $this->pdfToText($file->getRealPath());    // ← text string
                $encoded = base64_encode($plain);              // Gemini needs base64 bytes
                $sendFiles[$idx] = $encoded;          // false = text/plain
            } elseif (in_array($ext, ['xls', 'xlsx', 'csv'])) {
                $plain   = $this->excelToText($file);          // already returns text
                $encoded = base64_encode($plain);
                $sendFiles[$idx] = $encoded;
            }
        }

        $this->invoices = $this->processInvoices($this->sendToAI($sendFiles));
    }

    private function excelToText($file): string
    {
        $rows = Excel::toArray(null, $file)[0]; // first sheet
        $lines = [];

        foreach ($rows as $row) {
            $lines[] = implode("\t", array_map('trim', $row));
        }

        return implode("\n", $lines);
    }

    function pdfToText(string $pdfPath): string
    {
        $tmp = tempnam(sys_get_temp_dir(), 'pdfTxt');
        exec('pdftotext -layout ' . escapeshellarg($pdfPath) . ' ' . escapeshellarg($tmp));
        $text = file_get_contents($tmp);
        unlink($tmp);

        // normalise whitespace for Gemini
        $text = preg_replace('/\h+/u', ' ', $text);   // collapse runs of spaces
        $text = preg_replace('/ *\R+/u', "\n", $text); // trim rogue leading spaces
        return $text;
    }

    private function sendToAI($files)
    {
        $yourApiKey = getenv('GEMINI_API_KEY');
        $client = Gemini::client($yourApiKey);

        $count = 0;
        $data = [];
        while(!$data && $count < 3){

            $parts = [
                $this->prompt,
                ...collect($files)->map(fn ($file) =>   // build one Blob per file
                    new Blob(
                        mimeType: MimeType::TEXT_PLAIN,
                        data: $file
                    )
                )->all(),
            ];

            $result = $client
                ->generativeModel(config('gemini.models.flash'))
                ->generateContent($parts);
            $data = json_decode(str_replace(['```json', '```'], '', $result->text()), 1); // Clean up formatting

            $count++;
        }

        return $data;
    }

    private function normalizePoNumber($po)
    {
        $po = trim((string) $po);
        // Remove leading "PO" and common variants like "PO No.", "PO:", "PO-", "PO#"
        $po = preg_replace('/^\s*PO\s*(?:No\.?)*\s*[:#-]?\s*/i', '', $po);

        // If there is an embedded 6–9 digit sequence (the typical PO core), prefer that
        if (preg_match('/\b(\d{6,9})\b/', $po, $m)) {
            return $m[1];
        }

        // Fallback: strip non-digits to salvage a numeric core
        $digitsOnly = preg_replace('/\D+/', '', $po);
        return $digitsOnly ?: $po;
    }

    private function processInvoices($invoices){
        foreach($invoices as $i=>$invoice){
            // Normalise PO number to drop any leading "PO"
            if (isset($invoice['po_number'])) {
                $invoices[$i]['po_number'] = $this->normalizePoNumber($invoice['po_number']);
            }

            // 1) Gather candidate shipment lines per invoice line
            $contexts = [];
            foreach($invoice['lines'] as $l=>$line){
                // Build unfiltered candidate list (no colour constraint) for manual assignment UI
                $slAll = ShipmentLine::withSum('shipment_line_sizes', 'shipped_qty')
                    ->selectSub(function ($q) {
                        $q->from('customer_order_line_quantities')
                            ->select('price')
                            ->whereColumn('customer_order_lines_id', 'shipment_lines.customer_order_lines_id')
                            ->orderBy('id')
                            ->limit(1);
                    }, 'rt_price')
                    ->selectSub(function ($q) {
                        $q->from('customer_order_line_quantities')
                            ->select('commission')
                            ->whereColumn('customer_order_lines_id', 'shipment_lines.customer_order_lines_id')
                            ->orderBy('id')
                            ->limit(1);
                    }, 'commission')
                    ->whereHas('customer_order_lines.customer_orders', function($q) use ($invoices, $i) {
                        $prefix = $this->poPrefix($invoices[$i]['po_number']);
                        $norm   = $invoices[$i]['po_number'];
                        $q->where('customer_po', 'like', $prefix . '%')
                          ->orWhere('customer_po', 'like', '%' . $norm . '%');
                    })
                    ->whereHas('customer_order_lines.colourways.style_versions.styles', function($q) use ($line) {
                        $ref = $this->refPrefix($line['product_id'] ?? '');
                        if ($ref !== '') {
                            $q->where('customer_ref', 'like', $ref . '%')
                              ->orWhere('customer_ref', 'like', '%' . $ref . '%');
                        }
                    })
                    ->whereRelation(
                        'customer_order_lines.customer_orders.seasons',
                        'description',
                        'like',
                        $invoice['season'])
                    ->whereComplete(true)
                    ->whereInvoiced(false)
                    ->having('shipment_line_sizes_sum_shipped_qty', '>', 0)
                    ->get();
                // Fallback: try again without season filter if no candidates
                if ($slAll->isEmpty()) {
                    $slAll = ShipmentLine::withSum('shipment_line_sizes', 'shipped_qty')
                        ->selectSub(function ($q) {
                            $q->from('customer_order_line_quantities')
                                ->select('price')
                                ->whereColumn('customer_order_lines_id', 'shipment_lines.customer_order_lines_id')
                                ->orderBy('id')
                                ->limit(1);
                        }, 'rt_price')
                        ->selectSub(function ($q) {
                            $q->from('customer_order_line_quantities')
                                ->select('commission')
                                ->whereColumn('customer_order_lines_id', 'shipment_lines.customer_order_lines_id')
                                ->orderBy('id')
                                ->limit(1);
                        }, 'commission')
                        ->whereHas('customer_order_lines.customer_orders', function($q) use ($invoices, $i) {
                            $prefix = $this->poPrefix($invoices[$i]['po_number']);
                            $norm   = $invoices[$i]['po_number'];
                            $q->where('customer_po', 'like', $prefix . '%')
                              ->orWhere('customer_po', 'like', '%' . $norm . '%');
                        })
                        ->whereHas('customer_order_lines.colourways.style_versions.styles', function($q) use ($line) {
                            $ref = $this->refPrefix($line['product_id'] ?? '');
                            if ($ref !== '') {
                                $q->where('customer_ref', 'like', $ref . '%')
                                  ->orWhere('customer_ref', 'like', '%' . $ref . '%');
                            }
                        })
                        ->whereComplete(true)
                        ->whereInvoiced(false)
                        ->having('shipment_line_sizes_sum_shipped_qty', '>', 0)
                        ->get();
                }

                // Auto-match set prioritising colour
                $sl = ShipmentLine::withSum('shipment_line_sizes', 'shipped_qty')
                    ->selectSub(function ($q) {
                        $q->from('customer_order_line_quantities')
                            ->select('price')
                            ->whereColumn('customer_order_lines_id', 'shipment_lines.customer_order_lines_id')
                            ->orderBy('id')
                            ->limit(1);
                    }, 'rt_price')
                    ->selectSub(function ($q) {
                        $q->from('customer_order_line_quantities')
                            ->select('commission')
                            ->whereColumn('customer_order_lines_id', 'shipment_lines.customer_order_lines_id')
                            ->orderBy('id')
                            ->limit(1);
                    }, 'commission')
                    ->whereHas('customer_order_lines.customer_orders', function($q) use ($invoices, $i) {
                        $prefix = $this->poPrefix($invoices[$i]['po_number']);
                        $norm   = $invoices[$i]['po_number'];
                        $q->where('customer_po', 'like', $prefix . '%')
                          ->orWhere('customer_po', 'like', '%' . $norm . '%');
                    })
                    ->whereHas('customer_order_lines.colourways.style_versions.styles', function($q) use ($line) {
                        $ref = $this->refPrefix($line['product_id'] ?? '');
                        if ($ref !== '') {
                            $q->where('customer_ref', 'like', $ref . '%')
                              ->orWhere('customer_ref', 'like', '%' . $ref . '%');
                        }
                    })
                    // Colour priority match if provided
                    ->when(!empty($line['colour'] ?? null), function($q) use ($line) {
                        $colour = trim((string) $line['colour']);
                        $q->whereHas('customer_order_lines.colourways', function($qq) use ($colour) {
                            $qq->where('colourways.name', 'like', $colour)
                               ->orWhere('colourways.name', 'like', $colour . '%')
                               ->orWhere('colourways.name', 'like', '%' . $colour . '%');
                        });
                    })
                    ->whereRelation(
                        'customer_order_lines.customer_orders.seasons',
                        'description',
                        'like',
                        $invoice['season'])
                    ->whereComplete(true)
                    ->whereInvoiced(false)
                    ->having('shipment_line_sizes_sum_shipped_qty', '>', 0)
                    ->get();
                // Fallback: try again without season filter if no candidates (still honour colour)
                if ($sl->isEmpty()) {
                    $sl = ShipmentLine::withSum('shipment_line_sizes', 'shipped_qty')
                        ->selectSub(function ($q) {
                            $q->from('customer_order_line_quantities')
                                ->select('price')
                                ->whereColumn('customer_order_lines_id', 'shipment_lines.customer_order_lines_id')
                                ->orderBy('id')
                                ->limit(1);
                        }, 'rt_price')
                        ->selectSub(function ($q) {
                            $q->from('customer_order_line_quantities')
                                ->select('commission')
                                ->whereColumn('customer_order_lines_id', 'shipment_lines.customer_order_lines_id')
                                ->orderBy('id')
                                ->limit(1);
                        }, 'commission')
                        ->whereHas('customer_order_lines.customer_orders', function($q) use ($invoices, $i) {
                            $prefix = $this->poPrefix($invoices[$i]['po_number']);
                            $norm   = $invoices[$i]['po_number'];
                            $q->where('customer_po', 'like', $prefix . '%')
                              ->orWhere('customer_po', 'like', '%' . $norm . '%');
                        })
                        ->whereHas('customer_order_lines.colourways.style_versions.styles', function($q) use ($line) {
                            $ref = $this->refPrefix($line['product_id'] ?? '');
                            if ($ref !== '') {
                                $q->where('customer_ref', 'like', $ref . '%')
                                  ->orWhere('customer_ref', 'like', '%' . $ref . '%');
                            }
                        })
                        ->when(!empty($line['colour'] ?? null), function($q) use ($line) {
                            $colour = trim((string) $line['colour']);
                            $q->whereHas('customer_order_lines.colourways', function($qq) use ($colour) {
                                $qq->where('colourways.name', 'like', $colour)
                                   ->orWhere('colourways.name', 'like', $colour . '%')
                                   ->orWhere('colourways.name', 'like', '%' . $colour . '%');
                            });
                        })
                        ->whereComplete(true)
                        ->whereInvoiced(false)
                        ->having('shipment_line_sizes_sum_shipped_qty', '>', 0)
                        ->get();
                }

                // Expose candidates for manual assignment UI (unfiltered by colour) as scalar arrays (no models)
                $invoices[$i]['all_candidates'][$l] = $slAll->map(function($c){
                    return [
                        'id' => (int) $c->id,
                        'colour' => (string) optional($c->customer_order_lines->colourways)->name,
                        'shipped' => (int) round($c->shipment_line_sizes_sum_shipped_qty),
                        'no_invoiced' => (int) ($c->no_invoiced ?? 0),
                    ];
                })->values()->all();

                $contexts[] = [
                    'line_index' => $l,
                    'target'     => (int) ($line['quantity'] ?? 0),
                    'candidates' => $sl,
                    'colour'     => trim((string) ($line['colour'] ?? '')),
                ];

                $invoices[$i]['date'] = Carbon::parse($invoice['date']);
            }

            // Initialize UI remaining based on base availability (shipped - already invoiced)
            $this->recomputeUiRemaining($i);

            // 2) Compute a global, non-overlapping assignment
            $assignment = $this->assignShipmentLinesGlobally($contexts);

            // 3) Write assignment back to invoice lines
            foreach($invoice['lines'] as $l=>$line){
                $match = $assignment[$l] ?? null;
                $invoices[$i]['lines'][$l]['match'] = $match;
                if(isset($match[0])){
                    $first = $match[0];
                    $model = $first['model'] ?? null;
                    $comm = $model?->commission ?? 0;
                    $rtp  = $model?->rt_price ?? 0;
                    $invoices[$i]['lines'][$l]['commission'] = number_format(($invoices[$i]['lines'][$l]['line_total'] ?? 0) * ($comm / 100), 2, '.', '');
                    $invoices[$i]['lines'][$l]['rt_price'] = $rtp;
                    $invoices[$i]['lines'][$l]['commission_percent'] = $comm;
                }
            }
        }
        return($invoices);
    }

    // Recompute remaining qty per drop for an invoice after manual selections
    public function recomputeUiRemaining(int $invoiceIndex): void
    {
        if (!isset($this->invoices[$invoiceIndex])) { return; }

        $invoice = $this->invoices[$invoiceIndex];
        $remaining = [];

        // Build base remaining from all candidates across lines (using scalar arrays)
        foreach (($invoice['all_candidates'] ?? []) as $l => $cands) {
            foreach ($cands as $cand) {
                $id = (int) ($cand['id'] ?? 0);
                if (!$id) { continue; }
                $shipped = (int) ($cand['shipped'] ?? 0);
                $already = (int) ($cand['no_invoiced'] ?? 0);
                $avail = max(0, $shipped - $already);
                if (!isset($remaining[$id])) { $remaining[$id] = $avail; }
            }
        }

        // Apply manual selections in line order to reduce remaining
        $selections = $this->manualSelections[$invoiceIndex] ?? [];
        foreach ($invoice['lines'] as $l => $line) {
            $need = (int) ($line['quantity'] ?? 0);
            $selected = (array) ($selections[$l] ?? []);
            foreach ($selected as $dropId) {
                $dropId = (int) $dropId;
                if ($need <= 0) { break; }
                if (!isset($remaining[$dropId])) { continue; }
                $take = min($need, $remaining[$dropId]);
                $remaining[$dropId] -= $take;
                $need -= $take;
            }
        }

        $this->invoices[$invoiceIndex]['ui_remaining'] = $remaining;
    }

    /**
     * Assign shipment lines to invoice lines globally so that:
     * - Each invoice line receives a subset whose shipped_qty sums exactly to its target
     * - Each shipment line is used at most once across the whole invoice
     * - We maximize the number of matched invoice lines
     */
    private function assignShipmentLinesGlobally(array $contexts): array
    {
        $globalStart = microtime(true);
        $globalTimeBudget = 3.0; // seconds per invoice

        // Precompute candidate exact subsets for each line
        $lineToSubsets = [];
        foreach ($contexts as $ctx) {
            /** @var \Illuminate\Support\Collection $cands */
            $cands = $ctx['candidates'];
            $target = (int) $ctx['target'];
            // Generate exact subsets with pruning/time budget
            $subsets = $this->allExactSubsets(
                lines: $cands,
                target: $target,
                maxSubsets: 120,
                maxSubsetCardinality: 4,
                maxItems: 22,
                timeBudgetSeconds: 1.0
            );
            // Prefer smaller-cardinality subsets first
            usort($subsets, function($a, $b) {
                return count($a) <=> count($b);
            });
            // Keep only the top K options to cap branching
            $subsets = array_slice($subsets, 0, 15);
            $lineToSubsets[$ctx['line_index']] = $subsets; // array of arrays of ShipmentLine models
        }

        // Order lines by fewest candidate subsets first (MRV heuristic)
        $lineOrder = array_keys($lineToSubsets);
        usort($lineOrder, function($a, $b) use ($lineToSubsets) {
            return count($lineToSubsets[$a]) <=> count($lineToSubsets[$b]);
        });

        $bestAssignment = [];
        $bestMatched = 0;

        $usedIds = [];

        // Greedy baseline to set a lower bound
        $greedyAssignment = [];
        $greedyUsed = [];
        $greedyMatched = 0;
        foreach ($lineOrder as $line) {
            $picked = null;
            foreach ($lineToSubsets[$line] as $subset) {
                $conflict = false;
                foreach ($subset as $entry) {
                    if (!empty($greedyUsed[$entry['model']->id])) { $conflict = true; break; }
                }
                if (!$conflict) { $picked = $subset; break; }
            }
            // Greedy partial fallback: if no exact subset, try single closest under/over allocation
            if (!$picked) {
                $picked = $this->closestSubsetBySingle($lineToSubsets[$line], $greedyUsed);
            }
            if ($picked) {
                foreach ($picked as $entry) { $greedyUsed[$entry['model']->id] = 1; }
                $greedyAssignment[$line] = $picked;
                $greedyMatched++;
            } else {
                $greedyAssignment[$line] = null;
            }
        }
        $bestAssignment = $greedyAssignment;
        $bestMatched = $greedyMatched;

        $recurse = function($idx, $currentAssignment, $matchedSoFar) use (&$recurse, &$usedIds, &$bestAssignment, &$bestMatched, $lineOrder, $lineToSubsets, $globalStart, $globalTimeBudget) {
            // Time budget check
            if ((microtime(true) - $globalStart) > $globalTimeBudget) {
                return;
            }
            $numLines = count($lineOrder);
            if ($idx >= $numLines) {
                if ($matchedSoFar > $bestMatched) {
                    $bestMatched = $matchedSoFar;
                    $bestAssignment = $currentAssignment;
                }
                return;
            }

            $line = $lineOrder[$idx];
            $options = $lineToSubsets[$line];

            // Upper bound prune: even if we match all remaining, can we beat best?
            $remaining = $numLines - $idx;
            if ($matchedSoFar + $remaining <= $bestMatched) {
                return;
            }

            // Try all non-conflicting subset options first
            foreach ($options as $subset) {
                $conflict = false;
                foreach ($subset as $entry) {
                    if (!empty($usedIds[$entry['model']->id])) { $conflict = true; break; }
                }
                if ($conflict) { continue; }
                // apply
                foreach ($subset as $entry) { $usedIds[$entry['model']->id] = 1; }
                $currentAssignment[$line] = $subset;
                $recurse($idx + 1, $currentAssignment, $matchedSoFar + 1);
                // rollback
                foreach ($subset as $entry) { unset($usedIds[$entry['model']->id]); }
                unset($currentAssignment[$line]);
            }

            // Also consider leaving this line unmatched to potentially match others
            $currentAssignment[$line] = null;
            $recurse($idx + 1, $currentAssignment, $matchedSoFar);
        };

        $recurse(0, [], 0);

        // Normalize: ensure an entry for every line index
        $final = [];
        foreach (array_keys($lineToSubsets) as $l) {
            $final[$l] = $bestAssignment[$l] ?? null;
        }
        return $final;
    }

    /**
     * Return up to $maxSubsets subsets of $lines whose int shipped qty sums to $target.
     * Each subset is an array of ShipmentLine models.
     */
    private function allExactSubsets(Collection $lines, int $target, int $maxSubsets = 200, int $maxSubsetCardinality = 4, int $maxItems = 20, float $timeBudgetSeconds = 1.0): array
    {
        $start = microtime(true);
        // Build simplified items with qty and model
        $items = $lines
            ->map(function ($l) {
                $shipped = (int) round($l->shipment_line_sizes_sum_shipped_qty);
                $alreadyInvoiced = (int) ($l->no_invoiced ?? 0);
                $available = max(0, $shipped - $alreadyInvoiced);
                return [
                    'id'    => $l->id,
                    'qty'   => $available,
                    'model' => $l,
                    'colour'=> optional($l->customer_order_lines->colourways)->name,
                ];
            })
            ->filter(function ($x) { return $x['qty'] > 0; })
            ->values()
            ->all();

        // Sort by larger qty first to prune faster
        usort($items, function($a, $b) { return $b['qty'] <=> $a['qty']; });

        // Cap number of items considered to control complexity
        if (count($items) > $maxItems) {
            $items = array_slice($items, 0, $maxItems);
        }

        $n = count($items);
        $suffix = array_fill(0, $n + 1, 0);
        for ($i = $n - 1; $i >= 0; $i--) {
            $suffix[$i] = $suffix[$i + 1] + $items[$i]['qty'];
        }

        $results = [];
        $current = [];

        $dfs = function($idx, $remaining) use (&$dfs, &$results, &$current, $items, $suffix, $maxSubsets, $maxSubsetCardinality, $timeBudgetSeconds, $start) {
            if (count($results) >= $maxSubsets) { return; }
            if ((microtime(true) - $start) > $timeBudgetSeconds) { return; }
            if ($remaining === 0) {
                $results[] = $current; // keep rich entries with model & qty
                return;
            }
            if ($idx >= count($items)) { return; }
            // Prune: if even taking all remaining can't reach target
            if ($suffix[$idx] < $remaining) { return; }
            // Prune by cardinality cap
            if (count($current) >= $maxSubsetCardinality) { return; }

            // Option 1: take this item if it helps
            $qty = $items[$idx]['qty'];
            if ($qty <= $remaining) {
                $current[] = $items[$idx];
                $dfs($idx + 1, $remaining - $qty);
                array_pop($current);
            }

            // Option 2: skip this item
            $dfs($idx + 1, $remaining);
        };

        $dfs(0, $target);
        return $results;
    }

    /**
     * Greedy helper: choose a single available shipment line (no conflict) as partial
     * allocation when no exact subset exists. Prefers quantities closest to the target.
     */
    private function closestSubsetBySingle(array $subsets, array $used): ?array
    {
        // Subsets here are already exact; this function is a placeholder to enable
        // future extension for partial allocation. For now, return null to keep exact-only
        // during greedy seeding, as partials are handled by invoice creation updates.
        return null;
    }

    /**
     * Derive a reasonable PO prefix used for matching customer_po.
     * If the numeric core is long (>= 8), drop the last two digits which often vary per sub-drop.
     */
    private function poPrefix(string $po): string
    {
        $norm = $this->normalizePoNumber($po);
        $len  = strlen($norm);
        if ($len >= 8) {
            return substr($norm, 0, -2);
        }
        return $norm;
    }

    /**
     * Normalise product id/customer_ref for matching and return a stable prefix.
     * Strips common size/variant suffixes (e.g., trailing letters) while keeping the core code.
     */
    private function refPrefix(string $ref): string
    {
        $ref = trim($ref);
        if ($ref === '') {
            return '';
        }
        // Prefer the leading alphanumeric block up to the first non-word separator
        if (preg_match('/^[A-Z0-9]+/i', $ref, $m)) {
            return $m[0];
        }
        return $ref;
    }

    /**
     * Match any combination of ShipmentLine models whose summed
     * shipped_qty == $target. Returns null if no exact combo exists.
     *
     * Safe for up to about 25 lines (2^25 ≈ 33 M bit-masks).
     */
    private function matchShipmentLines(Collection $lines, int $target): ?Collection
    {
        // ---------- coerce to int and re-index ----------
        $lines = $lines
            ->map(fn ($l) => tap($l, function ($item) {
                $item->int_qty = (int) round($item->shipment_line_sizes_sum_shipped_qty);
            }))
            ->filter->int_qty           // drop zeros
            ->values();                 // 0-based index

        $n    = $lines->count();
        $max  = 1 << $n;               // 2^n subsets

        for ($mask = 1; $mask < $max; $mask++) {
            $sum    = 0;
            $subset = [];

            for ($i = 0; $i < $n; $i++) {
                if ($mask & (1 << $i)) {
                    $sum    += $lines[$i]->int_qty;
                    $subset[] = $lines[$i];
                }
            }

            if ($sum === $target) {
                return collect($subset);      // 🎯 exact combo
            }
        }

        return null;                           // none found
    }

    public function createInvoices(){
        // Basic pre-checks
        if (!$this->zoho->isConnected()) {
            session()->flash('messages', ['errors' => ['Zoho is not connected. Please connect and try again.']]);
            return;
        }
        if (empty($this->cust)) {
            session()->flash('messages', ['errors' => ['Select a Customer before creating invoices.']]);
            return;
        }

        $errors = [];
        $savedRefs = [];

        foreach($this->invoices as $invoiceIndex => $invoice){
            $lines = [];
            foreach($invoice['lines'] as $l => $line){
                // Determine commission percent robustly
                $commissionPercent = $line['commission_percent'] ?? null;
                if ($commissionPercent === null) {
                    if (!empty($line['match'][0]['model'])) {
                        $commissionPercent = (float) ($line['match'][0]['model']->commission ?? 0);
                    } else {
                        $firstManual = (array) ($this->manualSelections[$invoiceIndex][$l] ?? []);
                        $firstId = isset($firstManual[0]) ? (int) $firstManual[0] : null;
                        if ($firstId) {
                            $sl = ShipmentLine::find($firstId);
                            if ($sl) {
                                $commissionPercent = (float) (DB::table('customer_order_line_quantities')
                                    ->where('customer_order_lines_id', $sl->customer_order_lines_id)
                                    ->orderBy('id')
                                    ->limit(1)
                                    ->value('commission') ?? 0);
                            }
                        }
                    }
                }
                $commissionPercent = (float) ($commissionPercent ?? 0);
                $lineTotal = (float) ($line['line_total'] ?? 0);
                $commissionAmount = number_format($lineTotal * ($commissionPercent / 100), 2, '.', '');
                $lines[] = [
                    "name" => 'Commissions due on your invoice',
                    "description" => $invoice['invoice_no'] . ' - ' . $line['product_id'],
                    "rate" => $commissionAmount,
                    "quantity" => 1,
                    "account_id" => $this->account,
                    "tags" => [
                        [ //customer
                            "tag_id" => $this->getReportingTagId('Customer'),
                            "tag_option_id" => $this->reportingTagCustomer,
                        ],
                        [ //department
                            "tag_id" => $this->getReportingTagId('Department'),
                            "tag_option_id" => $this->reportingTagDepartment,
                        ]
                    ]
                ];
            }
            $invoiceSend = [
                "customer_id" => $this->cust,
                "template_id" => $this->template,
                "reference_number" => $invoice['po_number'],
                "date" => ($invoice['date'] instanceof \Illuminate\Support\Carbon)
                    ? $invoice['date']->format('Y-m-d')
                    : ((string) $invoice['date']),
                "line_items" => $lines,
                "status" => "draft",
                "custom_fields" => [
                    [
                        "customfield_id" => $this->getInvoiceCustomFieldId('Commissions Customer'),
                        "value" => $this->reportingTagCommissionsCustomer,
                    ]
                ],
            ];

            // Create invoice in Zoho with error handling
            try {
                $zohoInvoice = $this->zoho->createInvoice($invoiceSend, $this->org);
            } catch (\Throwable $e) {
                Log::error('Zoho createInvoice failed', [
                    'error' => $e->getMessage(),
                    'payload' => $invoiceSend,
                ]);
                $errors[] = 'Zoho error for PO ' . ($invoice['po_number'] ?? '') . ': ' . $e->getMessage();
                continue;
            }

            if(!$zohoInvoice){
                $errors[] = 'Zoho returned no response for PO ' . ($invoice['po_number'] ?? '');
                continue;
            }
            $invoiceNumber = is_array($zohoInvoice) && isset($zohoInvoice['invoice_number'])
                ? $zohoInvoice['invoice_number']
                : null;
            if (!$invoiceNumber) {
                $errors[] = 'Zoho did not return invoice_number for PO ' . ($invoice['po_number'] ?? '');
                continue;
            }

            foreach($invoice['lines'] as $l=>$line){
                // Allow partial invoicing: increment only by allocated amount (available qty)
                $matchedSubsets = $line['match'] ?? [];
                // Prepend manual selections if any: allocate in line order from remaining
                $invIdx = $invoiceIndex;
                $manual = (array) ($this->manualSelections[$invoiceIndex][$l] ?? []);
                if (!empty($manual)) {
                    // Allocate based on this line's remaining requirement, not UI page remaining
                    $lineTarget = (int) ($line['quantity'] ?? 0);
                    $alreadyAllocated = 0;
                    foreach ((array) $matchedSubsets as $ms) {
                        $alreadyAllocated += (int) ($ms['alloc'] ?? $ms['qty'] ?? 0);
                    }
                    $remainingForLine = max(0, $lineTarget - $alreadyAllocated);

                    foreach ($manual as $dropId) {
                        if ($remainingForLine <= 0) { break; }
                        $dropId = (int) $dropId;
                        $cand = collect($this->invoices[$invoiceIndex]['all_candidates'][$l] ?? [])->firstWhere('id', $dropId);
                        if (!$cand) { continue; }
                        $avail = max(0, (int)($cand['shipped'] ?? 0) - (int)($cand['no_invoiced'] ?? 0));
                        $take = min($avail, $remainingForLine);
                        if ($take > 0) {
                            $model = ShipmentLine::find($dropId);
                            if ($model) {
                                $matchedSubsets[] = [ 'model' => $model, 'qty' => $take, 'alloc' => $take ];
                                $remainingForLine -= $take;
                            }
                        }
                    }
                }
                // Update in-memory matches so the UI shows tables instead of selectors
                $this->invoices[$invoiceIndex]['lines'][$l]['match'] = $matchedSubsets;
                foreach((array) $matchedSubsets as $entry){
                    if (empty($entry)) { continue; }
                    [$shipmentLine, $alloc] = $this->resolveShipmentLineAndAlloc($entry);
                    if(!$shipmentLine) { continue; }
                    if ($alloc <= 0) { continue; }
                    $already = (int) ($shipmentLine->no_invoiced ?? 0);
                    $shipped = (int) ($shipmentLine->total_pieces_shipped ?? $shipmentLine->shipment_line_sizes()->sum('shipped_qty'));
                    $newNoInvoiced = min($shipped, $already + $alloc);
                    $fullyInvoiced = $newNoInvoiced >= $shipped;

                    $shipmentLine->update([
                        'rt_invoice' => $invoiceNumber,
                        'no_invoiced' => $newNoInvoiced,
                        'invoiced' => $fullyInvoiced ? 1 : 0,
                    ]);
                    $this->rtinvoice[$shipmentLine['id']] = $invoiceNumber;
                }
            $savedRefs[] = $invoiceNumber;
            // Refresh UI remaining for this invoice so subsequent lines reflect updated no_invoiced
            $this->recomputeUiRemaining($invoiceIndex);
            }
        }
        // Deduplicate and clean saved invoice numbers for UX
        $savedRefs = array_values(array_filter(array_unique($savedRefs)));
        if (!empty($errors)) {
            session()->flash('messages', ['errors' => $errors, 'saved' => $savedRefs]);
        } else {
            session()->flash('messages', ['saved' => $savedRefs]);
        }
    }

    private function resolveShipmentLineAndAlloc($entry): array
    {
        $model = null;
        $alloc = 0;
        if (is_array($entry)) {
            $maybeModel = $entry['model'] ?? null;
            if ($maybeModel instanceof ShipmentLine) {
                $model = $maybeModel;
            } elseif (is_array($maybeModel) && isset($maybeModel['id'])) {
                $model = ShipmentLine::find((int) $maybeModel['id']);
            } elseif (isset($entry['id'])) {
                $model = ShipmentLine::find((int) $entry['id']);
            }
            $alloc = (int) ($entry['alloc'] ?? $entry['qty'] ?? 0);
        } elseif ($entry instanceof ShipmentLine) {
            $model = $entry;
            $alloc = (int) round($entry->shipment_line_sizes_sum_shipped_qty);
        }
        return [$model, $alloc];
    }

    public function hydrate(){
        $this->zoho = app(ZohoService::class);

        if ($this->zoho->isConnected()) {
            $this->zohoConnected = true;
        } else {
            $this->authUrl = $this->zoho->getAuthUrl();
            $this->zohoConnected = false;
        }
    }

    // Recompute UI remaining whenever manual selections change
    public function updatedManualSelections($value, $key = null): void
    {
        // $key example: "manualSelections.0.2" → invoiceIndex = 0
        if (is_string($key) && str_starts_with($key, 'manualSelections.')) {
            $parts = explode('.', $key);
            $invoiceIndex = isset($parts[1]) ? (int) $parts[1] : null;
            $lineIndex    = isset($parts[2]) ? (int) $parts[2] : null;
            if ($invoiceIndex !== null) {
                $this->recomputeUiRemaining($invoiceIndex);
                if ($lineIndex !== null) {
                    $this->recomputeInvoiceLinePricing($invoiceIndex, $lineIndex);
                }
            }
        } else {
            // Fallback: recompute all invoices
            foreach (array_keys((array) $this->invoices) as $idx) {
                $this->recomputeUiRemaining((int) $idx);
                foreach (array_keys((array) ($this->invoices[$idx]['lines'] ?? [])) as $l) {
                    $this->recomputeInvoiceLinePricing((int) $idx, (int) $l);
                }
            }
        }
    }

    private function recomputeInvoiceLinePricing(int $invoiceIndex, int $lineIndex): void
    {
        if (!isset($this->invoices[$invoiceIndex]['lines'][$lineIndex])) { return; }
        $line =& $this->invoices[$invoiceIndex]['lines'][$lineIndex];

        // 1) Prefer manual selection for pricing
        $manual = (array) ($this->manualSelections[$invoiceIndex][$lineIndex] ?? []);
        if (!empty($manual)) {
            $firstId = (int) $manual[0];
            [$percent, $price] = $this->getCommissionAndPriceForShipmentLine($firstId);
            $line['commission_percent'] = $percent;
            $line['rt_price'] = $price;
            $line['commission'] = number_format(((float) ($line['line_total'] ?? 0)) * ($percent / 100), 2, '.', '');
            return;
        }

        // 2) Else use first auto match if available
        if (!empty($line['match'][0]['model'])) {
            $model = $line['match'][0]['model'];
            $percent = (float) ($model->commission ?? 0);
            $price   = (float) ($model->rt_price ?? 0);
            if ($price == 0.0) {
                [$percent2, $price2] = $this->getCommissionAndPriceForShipmentLine((int) $model->id);
                if ($price == 0.0) { $price = $price2; }
                if ($percent == 0.0) { $percent = $percent2; }
            }
            $line['commission_percent'] = $percent;
            $line['rt_price'] = $price;
            $line['commission'] = number_format(((float) ($line['line_total'] ?? 0)) * ($percent / 100), 2, '.', '');
            return;
        }

        // 3) Fallback to zeros
        $line['commission_percent'] = 0.0;
        $line['rt_price'] = 0.0;
        $line['commission'] = number_format(0, 2, '.', '');
    }

    private function getCommissionAndPriceForShipmentLine(int $shipmentLineId): array
    {
        $sl = ShipmentLine::find($shipmentLineId);
        if (!$sl) { return [0.0, 0.0]; }
        $row = DB::table('customer_order_line_quantities')
            ->where('customer_order_lines_id', $sl->customer_order_lines_id)
            ->orderBy('id')
            ->limit(1)
            ->first();
        $percent = (float) ($row->commission ?? 0);
        $price   = (float) ($row->price ?? 0);
        return [$percent, $price];
    }

    #[On('render')]
    public function render()
    {
        Gate::authorize('finance:create');
        if($this->buttonDisabled()){
            session()->flash('message', 'Max 20 files.');
            session()->flash('alert-class', 'alert-warning');
        }

        $this->zoho = app(ZohoService::class);

        if ($this->zoho->isConnected()) {
            $this->zohoConnected = true;
        } else {
            $this->authUrl = $this->zoho->getAuthUrl();
            $this->zohoConnected = false;
        }

        return view('livewire.imports.erdos-import-ai');
    }

    private function buttonDisabled(){
        return count($this->orderFiles) > 20;
    }


}
