<?php

namespace App\Http\Livewire\Imports;

use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Departments;
use App\Http\Livewire\BaseComponent;
use App\Services\PackingListImports\PackingListExtractorFactory;
use App\Services\PackingListImports\PackingListImportService;
use App\Services\ZohoService;
use App\Http\Livewire\Traits\HandlesZohoOAuth;
use Livewire\WithFileUploads;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class CommissionPLImport extends BaseComponent
{
    use WithFileUploads, HandlesZohoOAuth;

    // Form properties
    public string $selectedCustomer = '';
    public string $selectedSeason = '';
    public array $uploadedFiles = [];
    public array $selectedOrderLines = []; // For user selection of colors
    public bool $includeCompletedLines = false; // Allow matching to completed but not invoiced lines
    
    // Processing state
    public bool $isProcessing = false;
    public bool $showPreview = false;
    public array $extractedData = [];
    public array $validationErrors = [];
    public array $importResults = [];
    
    // Zoho integration
    public bool $zohoConnected = false;
    public string $authUrl = '';
    public bool $createInvoice = false;
    public string $zohoCustomerId = '';
    public string $zohoOrg = '20094201933';
    public string $zohoTemplate = '479608000000045843';
    public string $zohoAccount = '479608000000045228';
    public string $reportingTagDepartment = '479608000000063057';
    public string $reportingTagCustomer = '';
    private $zoho;

    #[Validate('required')]
    public $packingListFiles = [];

    public function render()
    {
        Gate::authorize('order:update');
        return view('livewire.imports.commission-pl-import');
    }
    
    /**
     * Check if current user has finance permissions
     */
    public function isFinanceUser(): bool
    {
        return Gate::allows('finance:create') || Gate::allows('finance:update');
    }

    private function getImportService(): PackingListImportService
    {
        return new PackingListImportService();
    }

    #[Computed]
    public function availableCustomers(): array
    {
        return PackingListExtractorFactory::getAvailableCustomers();
    }
    
    /**
     * Get the customer name from Loop database
     */
    public function getLoopCustomerName(): string
    {
        if (empty($this->selectedCustomer)) {
            return '';
        }
        
        $availableCustomers = $this->availableCustomers();
        return $availableCustomers[$this->selectedCustomer] ?? '';
    }

    #[Computed]
    public function seasons(): \Illuminate\Database\Eloquent\Collection
    {
        return Seasons::where('locked', false)->orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function departments(): \Illuminate\Database\Eloquent\Collection
    {
        return Departments::where('type', 'commission')->where('hidden', false)->get();
    }

    public function processFiles(): void
    {
        set_time_limit(300);
        ini_set('memory_limit', '1G');

        $this->validate([
            'selectedCustomer' => 'required',
            'packingListFiles' => 'required|array|min:1',
            'packingListFiles.*' => 'file|mimes:pdf,xlsx,xls|max:10240',
        ]);

        if ($this->isProcessing) return;

        $this->isProcessing = true;
        $this->extractedData = [];
        $this->validationErrors = [];
        $this->showPreview = false;

        try {
            $extractor = PackingListExtractorFactory::make($this->selectedCustomer);
            $extractor->setOptions([
                'include_completed_lines' => $this->includeCompletedLines,
            ]);

            $allExtractedData = [];
            $fileErrors = [];

            foreach ($this->packingListFiles as $file) {
                if (!$extractor->isFileSupported($file)) {
                    $fileErrors[] = "File {$file->getClientOriginalName()}: Unsupported file type";
                    continue;
                }

                try {
                    $fileData = $extractor->extractData($file);
                    $fileData['metadata']['filename'] = $file->getClientOriginalName();
                    $allExtractedData[] = $fileData;
                } catch (\Exception $e) {
                    $fileErrors[] = "File {$file->getClientOriginalName()}: " . $e->getMessage();
                }
            }

            if (!empty($fileErrors)) {
                $this->validationErrors = array_merge($this->validationErrors, $fileErrors);
            }

            if (empty($allExtractedData)) {
                throw new \RuntimeException('No files could be processed successfully');
            }

            $this->extractedData = $this->mergeExtractedData($allExtractedData);

            // Validate extracted data
            $this->validationErrors = array_merge(
                $this->validationErrors,
                $this->getImportService()->validateImportData($this->extractedData),
                $extractor->validateCustomerData($this->extractedData)
            );

            // Check if we have actual packing list data to show
            $hasData = !empty($this->extractedData['packing_lists']);
            
            // Show preview if we have data, even if there are warnings
            $this->showPreview = $hasData;

            if ($hasData) {
                if (empty($this->validationErrors)) {
                    session()->flash('message', 'Files processed successfully. Review and click Import to apply.');
                    session()->flash('alert-class', 'alert-success');
                } else {
                    session()->flash('message', 'Files processed with warnings. Review the data below.');
                    session()->flash('alert-class', 'alert-warning');
                }
            } else {
                session()->flash('message', 'No packing list data could be extracted.');
                session()->flash('alert-class', 'alert-warning');
            }

        } catch (\Exception $e) {
            $this->validationErrors[] = "Error processing files: " . $e->getMessage();
            session()->flash('message', 'Error processing files: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        } finally {
            $this->isProcessing = false;
        }
    }

    public function saveData(): void
    {
        set_time_limit(600);
        ini_set('memory_limit', '2G');

        if (!$this->showPreview || empty($this->extractedData)) {
            session()->flash('message', 'Cannot save: Process files first.');
            session()->flash('alert-class', 'alert-warning');
            return;
        }

        // Check if all items have order line IDs
        $unmatchedCount = 0;
        foreach ($this->extractedData['packing_lists'] as $pl) {
            if (empty($pl['customer_order_line_id'])) {
                $unmatchedCount++;
            }
        }

        if ($unmatchedCount > 0) {
            session()->flash('message', "Cannot save: {$unmatchedCount} item(s) not matched to order lines. Please select colors from dropdowns or ensure order is imported.");
            session()->flash('alert-class', 'alert-warning');
            return;
        }

        $this->isProcessing = true;
        $this->importResults = [];

        try {
            // Apply any user-selected order lines from dropdowns
            foreach ($this->selectedOrderLines as $index => $orderLineId) {
                if (!empty($orderLineId) && isset($this->extractedData['packing_lists'][$index])) {
                    $this->extractedData['packing_lists'][$index]['customer_order_line_id'] = (int) $orderLineId;
                }
            }

            // Save without marking as shipped
            $results = $this->getImportService()->processImport($this->extractedData, false);
            $this->importResults = $results;

            if ($results['success']) {
                $message = "Packing list data saved successfully! ";
                $message .= "Updated {$results['shipment_lines_updated']} shipment line(s) with {$results['sizes_updated']} size record(s). ";
                $message .= "Lines NOT marked as shipped - you can review and then mark as shipped later.";
                
                if (!empty($results['errors'])) {
                    $message .= " Some items were skipped: " . implode(', ', $results['errors']);
                }
                
                session()->flash('message', $message);
                session()->flash('alert-class', 'alert-success');
                $this->reset(['packingListFiles', 'extractedData', 'showPreview', 'validationErrors', 'selectedOrderLines']);
            } else {
                session()->flash('message', 'Save failed: ' . implode(', ', $results['errors']));
                session()->flash('alert-class', 'alert-danger');
            }
        } catch (\Exception $e) {
            $this->importResults = [
                'success' => false,
                'shipment_lines_updated' => 0,
                'sizes_updated' => 0,
                'errors' => [$e->getMessage()],
            ];
            session()->flash('message', 'Save failed: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        } finally {
            $this->isProcessing = false;
        }
    }

    public function importData(): void
    {
        set_time_limit(600);
        ini_set('memory_limit', '2G');

        if (!$this->showPreview || empty($this->extractedData)) {
            session()->flash('message', 'Cannot import: Process files first.');
            session()->flash('alert-class', 'alert-warning');
            return;
        }

        // Check if all items have order line IDs
        $unmatchedCount = 0;
        foreach ($this->extractedData['packing_lists'] as $pl) {
            if (empty($pl['customer_order_line_id'])) {
                $unmatchedCount++;
            }
        }

        if ($unmatchedCount > 0) {
            session()->flash('message', "Cannot import: {$unmatchedCount} item(s) not matched to order lines. Please select colors from dropdowns or ensure order is imported.");
            session()->flash('alert-class', 'alert-warning');
            return;
        }

        $this->isProcessing = true;
        $this->importResults = [];

        try {
            // Apply any user-selected order lines from dropdowns
            foreach ($this->selectedOrderLines as $index => $orderLineId) {
                if (!empty($orderLineId) && isset($this->extractedData['packing_lists'][$index])) {
                    $this->extractedData['packing_lists'][$index]['customer_order_line_id'] = (int) $orderLineId;
                }
            }

            // Save and mark as shipped
            $results = $this->getImportService()->processImport($this->extractedData, true);
            $this->importResults = $results;

            if ($results['success']) {
                $message = "Packing list import completed! ";
                $message .= "Updated {$results['shipment_lines_updated']} shipment line(s) with {$results['sizes_updated']} size record(s). ";
                $message .= "All lines marked as shipped.";
                
                if (!empty($results['errors'])) {
                    $message .= " Some items were skipped: " . implode(', ', $results['errors']);
                }
                
                session()->flash('message', $message);
                session()->flash('alert-class', 'alert-success');
                $this->reset(['packingListFiles', 'extractedData', 'showPreview', 'validationErrors', 'selectedOrderLines']);
            } else {
                session()->flash('message', 'Import failed: ' . implode(', ', $results['errors']));
                session()->flash('alert-class', 'alert-danger');
            }
        } catch (\Exception $e) {
            $this->importResults = [
                'success' => false,
                'shipment_lines_updated' => 0,
                'sizes_updated' => 0,
                'errors' => [$e->getMessage()],
            ];
            session()->flash('message', 'Import failed: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        } finally {
            $this->isProcessing = false;
        }
    }

    public function resetForm(): void
    {
        $this->reset();
        session()->flash('message', 'Form reset successfully.');
        session()->flash('alert-class', 'alert-info');
    }

    public function updatedSelectedCustomer(): void
    {
        $this->reset(['extractedData', 'showPreview', 'validationErrors', 'packingListFiles']);
    }

    /**
     * Update match status when user selects an order line from dropdown
     */
    public function updatedSelectedOrderLines($value, $index): void
    {
        if (!empty($value) && isset($this->extractedData['packing_lists'][$index])) {
            $this->extractedData['packing_lists'][$index]['customer_order_line_id'] = (int) $value;
            $this->extractedData['packing_lists'][$index]['_match_status'] = 'matched';
            
            // Force re-render by clearing and resetting the preview
            $this->dispatch('$refresh');
        }
    }

    public function getPreviewData(): array
    {
        if (empty($this->extractedData['packing_lists'])) return [];
        
        $preview = [];
        $fileTotal = 0;
        
        foreach ($this->extractedData['packing_lists'] as $index => $pl) {
            $sizeBreakdown = [];
            $totalQty = 0;
            foreach ($pl['sizes'] as $row) {
                $qty = (int) ($row['qty'] ?? 0);
                $sizeBreakdown[] = $row['size'] . ':' . $qty;
                $totalQty += $qty;
            }
            
            $fileTotal += $totalQty;
            
            $matchStatus = $pl['_match_status'] ?? 'unknown';
            $statusBadge = $this->getMatchStatusBadge($matchStatus);
            
            // Get color name and system price if shipment line is matched
            $orderLineDisplay = null;
            $colourName = null;
            $systemUnitPrice = null;
            $priceDifference = null;
            $priceMatch = null;
            
            if (!empty($pl['customer_order_line_id'])) {
                // customer_order_line_id is actually the shipment_line_id
                $shipmentLine = \App\Models\ShipmentLine::with(['customer_order_lines.colourways', 'customer_order_lines.customer_order_line_quantities'])
                    ->find($pl['customer_order_line_id']);
                if ($shipmentLine && $shipmentLine->customer_order_lines) {
                    $orderLine = $shipmentLine->customer_order_lines;
                    if ($orderLine->colourways) {
                        $colourName = $orderLine->colourways->name;
                        $orderLineDisplay = $colourName . ' (Line ' . $shipmentLine->id . ')';
                    } else {
                        $orderLineDisplay = 'Line ' . $shipmentLine->id;
                    }
                    
                    // Get system unit price from customer_order_line_quantities
                    $firstQuantity = $orderLine->customer_order_line_quantities->first();
                    $systemUnitPrice = $firstQuantity->price ?? null;
                    
                    if ($systemUnitPrice && !empty($pl['_invoice_unit_price'])) {
                        $invoicePrice = (float) $pl['_invoice_unit_price'];
                        $priceDifference = $invoicePrice - $systemUnitPrice;
                        
                        // Determine if prices match (within 0.01 tolerance)
                        if (abs($priceDifference) < 0.01) {
                            $priceMatch = 'match';
                        } elseif ($invoicePrice > $systemUnitPrice) {
                            $priceMatch = 'higher';
                        } else {
                            $priceMatch = 'lower';
                        }
                    }
                }
            }

            $preview[] = [
                'index' => $index,
                'customer_order_line_id' => $pl['customer_order_line_id'] ?? null,
                'order_line_display' => $orderLineDisplay,
                'colour_name' => $colourName,
                'style' => $pl['_style'] ?? 'N/A',
                'color' => $pl['_color'] ?? 'N/A',
                'exfty' => $pl['exfty'] ?? null,
                'cartons' => $pl['cartons'] ?? null,
                'total_qty' => $totalQty,
                'net_weight' => $pl['net_weight'] ?? null,
                'gross_weight' => $pl['gross_weight'] ?? null,
                'size_breakdown' => implode(', ', $sizeBreakdown),
                'source_file' => $pl['_source_file'] ?? ($this->extractedData['metadata']['filename'] ?? 'Unknown'),
                'match_status' => $matchStatus,
                'status_badge' => $statusBadge,
                'possible_matches' => $pl['_possible_matches'] ?? [],
                'invoice_unit_price' => $pl['_invoice_unit_price'] ?? null,
                'system_unit_price' => $systemUnitPrice,
                'price_difference' => $priceDifference,
                'price_match' => $priceMatch,
            ];
        }
        
        // Add file total to metadata
        $this->extractedData['_file_total'] = $fileTotal;
        
        return $preview;
    }
    
    private function getMatchStatusBadge(string $status): string
    {
        return match($status) {
            'matched' => '<span class="badge bg-success">Matched</span>',
            'style_matched_select_color' => '<span class="badge bg-warning text-dark">Select Color</span>',
            'multiple_matches' => '<span class="badge bg-warning text-dark">Multiple Matches - Select</span>',
            'not_matched' => '<span class="badge bg-danger">Not Matched</span>',
            'order_not_found' => '<span class="badge bg-secondary">Order Not Found</span>',
            default => '<span class="badge bg-secondary">Unknown</span>',
        };
    }

    private function mergeExtractedData(array $allFileData): array
    {
        $mergedPackingLists = [];
        $filenames = [];

        foreach ($allFileData as $fileData) {
            foreach ($fileData['packing_lists'] ?? [] as $pl) {
                $pl['_source_file'] = $fileData['metadata']['filename'] ?? null;
                $mergedPackingLists[] = $pl;
            }

            if (!empty($fileData['metadata']['filename'])) {
                $filenames[] = $fileData['metadata']['filename'];
            }
        }

        return [
            'packing_lists' => $mergedPackingLists,
            'metadata' => [
                'extraction_method' => count($allFileData) > 1 ? 'multiple_files' : 'single_file',
                'filenames' => $filenames,
                'files_processed' => count($allFileData),
            ],
        ];
    }
    
    public function hydrate()
    {
        $this->zoho = app(ZohoService::class);

        if ($this->zoho->isConnected()) {
            $this->zohoConnected = true;
        } else {
            $this->authUrl = $this->zoho->getAuthUrl();
            $this->zohoConnected = false;
        }
    }
    
    public function organisations()
    {
        if (!$this->zoho || !$this->zoho->isConnected()) {
            return [];
        }
        return $this->zoho->getOrganisations();
    }
    
    public function customers()
    {
        if (!$this->zoho || !$this->zoho->isConnected()) {
            return [];
        }
        return $this->zoho->getCustomers($this->zohoOrg);
    }
    
    public function templates()
    {
        if (!$this->zoho || !$this->zoho->isConnected()) {
            return [];
        }
        return $this->zoho->getInvoiceTemplates($this->zohoOrg);
    }
    
    public function accounts()
    {
        if (!$this->zoho || !$this->zoho->isConnected()) {
            return [];
        }
        return $this->zoho->getChartOfAccounts($this->zohoOrg);
    }
    
    public function reportingTagCustomers()
    {
        if (!$this->zoho || !$this->zoho->isConnected()) {
            return [];
        }
        $tagId = $this->getReportingTagId('Customer');
        return $this->zoho->getReportingTagOptions($this->zohoOrg, $tagId);
    }
    
    public function reportingTagDepartments()
    {
        if (!$this->zoho || !$this->zoho->isConnected()) {
            return [];
        }
        $tagId = $this->getReportingTagId('Department');
        return $this->zoho->getReportingTagOptions($this->zohoOrg, $tagId);
    }
    
    private function getReportingTagId($name)
    {
        $tags = $this->zoho->getReportingTags($this->zohoOrg);
        $tag = collect($tags)->where('tag_name', $name)->first();
        return $tag['tag_id'] ?? '';
    }
    
    private function getInvoiceCustomFieldId($name)
    {
        $fields = $this->zoho->getCustomFields($this->zohoOrg);
        $id = collect($fields['invoice'] ?? [])->firstWhere('label', $name)['customfield_id'] ?? '';
        return $id;
    }
    
    public function createZohoInvoice(): void
    {
        if (!$this->zoho || !$this->zoho->isConnected()) {
            session()->flash('message', 'Zoho is not connected. Please connect and try again.');
            session()->flash('alert-class', 'alert-danger');
            return;
        }
        
        if (empty($this->zohoCustomerId)) {
            session()->flash('message', 'Please select a Zoho customer.');
            session()->flash('alert-class', 'alert-warning');
            return;
        }
        
        if (!$this->showPreview || empty($this->extractedData)) {
            session()->flash('message', 'Cannot create invoice: Process files first.');
            session()->flash('alert-class', 'alert-warning');
            return;
        }
        
        try {
            $invoiceData = $this->extractedData['metadata']['invoice_data'] ?? [];
            $packingLists = $this->extractedData['packing_lists'] ?? [];
            
            $factoryInvoiceNo = $invoiceData['invoice_number'] ?? 'Unknown';
            $invoiceDate = $invoiceData['invoice_date'] ?? now()->format('Y-m-d');
            $poNumber = $invoiceData['po_number'] ?? '';
            
            // Group by order line for invoice lines
            $invoiceLines = [];
            foreach ($packingLists as $pl) {
                if (empty($pl['customer_order_line_id'])) continue;
                
                $orderLine = \App\Models\CustomerOrderLines::with([
                    'colourways.style_versions.styles',
                    'customer_order_line_quantities'
                ])->find($pl['customer_order_line_id']);
                
                if (!$orderLine) continue;
                
                $style = $orderLine->colourways->style_versions->styles->customer_ref ?? 'N/A';
                $color = $orderLine->colourways->name ?? 'N/A';
                $qty = $pl['_total_qty'] ?? 0;
                $price = $orderLine->customer_order_line_quantities->first()->price ?? 0;
                $lineTotal = $qty * $price;
                
                $invoiceLines[] = [
                    'name' => "Freight on {$style} - {$color}",
                    'description' => "Factory Invoice: {$factoryInvoiceNo}, PO: {$poNumber}",
                    'rate' => number_format($price, 2, '.', ''),
                    'quantity' => $qty,
                    'account_id' => $this->zohoAccount,
                    'tags' => [
                        [
                            'tag_id' => $this->getReportingTagId('Customer'),
                            'tag_option_id' => $this->reportingTagCustomer,
                        ],
                        [
                            'tag_id' => $this->getReportingTagId('Department'),
                            'tag_option_id' => $this->reportingTagDepartment,
                        ],
                    ],
                ];
            }
            
            if (empty($invoiceLines)) {
                session()->flash('message', 'No valid lines to invoice.');
                session()->flash('alert-class', 'alert-warning');
                return;
            }
            
            $invoicePayload = [
                'customer_id' => $this->zohoCustomerId,
                'template_id' => $this->zohoTemplate,
                'reference_number' => $factoryInvoiceNo,
                'date' => $invoiceDate,
                'line_items' => $invoiceLines,
                'status' => 'draft',
                'custom_fields' => [
                    [
                        'customfield_id' => $this->getInvoiceCustomFieldId('Commissions Customer'),
                        'value' => $this->getLoopCustomerName(),
                    ],
                ],
            ];
            
            $zohoInvoice = $this->zoho->createInvoice($invoicePayload, $this->zohoOrg);
            
            if (!$zohoInvoice || empty($zohoInvoice['invoice_number'])) {
                session()->flash('message', 'Failed to create invoice in Zoho.');
                session()->flash('alert-class', 'alert-danger');
                return;
            }
            
            $rtInvoiceNo = $zohoInvoice['invoice_number'];
            
            // Send email notification
            $this->sendInvoiceNotification($factoryInvoiceNo, $rtInvoiceNo, $invoicePayload);
            
            session()->flash('message', "Invoice created successfully: {$rtInvoiceNo} (Draft)");
            session()->flash('alert-class', 'alert-success');
            
        } catch (\Exception $e) {
            Log::error('Failed to create Zoho invoice for packing list', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            session()->flash('message', 'Error creating invoice: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        }
    }
    
    private function sendInvoiceNotification(string $factoryInvoiceNo, string $rtInvoiceNo, array $invoiceData): void
    {
        try {
            $subject = "Packing List Invoice Created: {$rtInvoiceNo}";
            $totalAmount = collect($invoiceData['line_items'])->sum(function($line) {
                return ($line['rate'] ?? 0) * ($line['quantity'] ?? 0);
            });
            
            $body = "A new invoice has been created from packing list import:\n\n";
            $body .= "RT Invoice Number: {$rtInvoiceNo}\n";
            $body .= "Factory Invoice Number: {$factoryInvoiceNo}\n";
            $body .= "PO Number: " . ($invoiceData['reference_number'] ?? 'N/A') . "\n";
            $body .= "Date: " . ($invoiceData['date'] ?? 'N/A') . "\n";
            $body .= "Total Amount: £" . number_format($totalAmount, 2) . "\n";
            $body .= "Status: Draft\n\n";
            $body .= "Line Items:\n";
            foreach ($invoiceData['line_items'] as $line) {
                $body .= "- {$line['name']}: {$line['quantity']} @ £{$line['rate']} = £" . 
                         number_format(($line['rate'] ?? 0) * ($line['quantity'] ?? 0), 2) . "\n";
            }
            
            Mail::raw($body, function ($message) use ($subject) {
                $message->to('neil@roberttodds.com')
                        ->subject($subject);
            });
            
            Log::info('Invoice notification sent', [
                'factory_invoice' => $factoryInvoiceNo,
                'rt_invoice' => $rtInvoiceNo,
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to send invoice notification email', [
                'error' => $e->getMessage(),
                'factory_invoice' => $factoryInvoiceNo,
                'rt_invoice' => $rtInvoiceNo,
            ]);
        }
    }
}


