<?php

namespace App\Http\Livewire\Imports;

use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Departments;
use App\Http\Livewire\BaseComponent;
use App\Services\CommissionImports\CommissionExtractorFactory;
use App\Services\CommissionImports\CommissionImportService;
use App\Services\CommissionImports\StyleMatchingService;
use Livewire\WithFileUploads;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;

class CommissionImport extends BaseComponent
{
    use WithFileUploads;

    // Form properties
    public string $selectedCustomer = '';
    public string $selectedSeason = '';
    public array $uploadedFiles = [];
    
    // Processing state
    public bool $isProcessing = false;
    public bool $showPreview = false;
    public array $extractedData = [];
    public array $validationErrors = [];
    public array $importResults = [];
    public array $duplicateOrders = [];
    public array $duplicateResolution = [];
    
    #[Validate('required')]
    public $orderFiles = [];

    public function mount()
    {
        // Initialization handled in getImportService method
    }

    /**
     * Get the import service instance
     */
    private function getImportService(): CommissionImportService
    {
        return new CommissionImportService(new StyleMatchingService());
    }

    public function render()
    {
        Gate::authorize('order:create');
        
        return view('livewire.imports.commission-import');
    }

    #[Computed]
    public function availableCustomers(): array
    {
        return CommissionExtractorFactory::getAvailableCustomers();
    }

    #[Computed]
    public function seasons(): \Illuminate\Database\Eloquent\Collection
    {
        return Seasons::where('locked', false)->orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function departments(): \Illuminate\Database\Eloquent\Collection
    {
        return Departments::where('type', 'commission')->where('hidden', false)->get();
    }

    /**
     * Process uploaded files and extract data
     */
    public function processFiles(): void
    {
        // Increase execution time and memory limits for file processing
        set_time_limit(300); // 5 minutes
        ini_set('memory_limit', '1G');
        
        $this->validate([
            'selectedCustomer' => 'required',
            'orderFiles' => 'required|array|min:1',
            'orderFiles.*' => 'file|mimes:pdf,xlsx,xls|max:10240', // 10MB max
        ]);

        if ($this->isProcessing) {
            return;
        }

        $this->isProcessing = true;
        $this->extractedData = [];
        $this->validationErrors = [];
        $this->showPreview = false;

        try {
            $extractor = CommissionExtractorFactory::make($this->selectedCustomer);
            
            $allExtractedData = [];
            $fileErrors = [];
            
            foreach ($this->orderFiles as $fileIndex => $file) {
                if (!$extractor->isFileSupported($file)) {
                    $fileErrors[] = "File {$file->getClientOriginalName()}: Unsupported file type";
                    continue;
                }
                
                try {
                    $fileData = $extractor->extractData($file);
                    $fileData['metadata']['filename'] = $file->getClientOriginalName();
                    $allExtractedData[] = $fileData;
                } catch (\Exception $e) {
                    $fileErrors[] = "File {$file->getClientOriginalName()}: " . $e->getMessage();
                }
            }
            
            if (!empty($fileErrors)) {
                $this->validationErrors = array_merge($this->validationErrors, $fileErrors);
            }
            
            if (empty($allExtractedData)) {
                throw new \RuntimeException('No files could be processed successfully');
            }
            
            // Merge all file data
            $this->extractedData = $this->mergeExtractedData($allExtractedData);
            
            // Auto-detect season if not selected
            if (empty($this->selectedSeason)) {
                $detectedSeason = $this->getImportService()->detectSeason($this->extractedData);
                if ($detectedSeason) {
                    $this->selectedSeason = $detectedSeason->id;
                }
            }
            
            // Check for season conflicts
            if (!empty($this->extractedData['metadata']['season_conflict'])) {
                $seasons = implode(', ', $this->extractedData['metadata']['all_detected_seasons']);
                $this->validationErrors[] = "Season conflict detected across files: {$seasons}. Please manually select a season.";
            }
            
            // Detect duplicate POs against existing orders for selected customer and season
            $this->duplicateOrders = $this->detectDuplicateOrders($allExtractedData);

            // Validate extracted data
            $this->validationErrors = array_merge(
                $this->validationErrors,
                $this->getImportService()->validateImportData($this->extractedData),
                $extractor->validateCustomerData($this->extractedData)
            );
            
            if (empty($this->validationErrors)) {
                $this->showPreview = true;
                session()->flash('message', 'Files processed successfully. Please review the data below and click Import to proceed.');
                session()->flash('alert-class', 'alert-success');
            } else {
                session()->flash('message', 'Validation errors found. Please review and correct the issues.');
                session()->flash('alert-class', 'alert-warning');
            }
            
        } catch (\Exception $e) {
            $this->validationErrors[] = "Error processing files: " . $e->getMessage();
            session()->flash('message', 'Error processing files: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        } finally {
            $this->isProcessing = false;
        }
    }

    /**
     * Import the processed data
     */
    public function importData(): void
    {
        // Increase execution time and memory limits for data import
        set_time_limit(600); // 10 minutes
        ini_set('memory_limit', '2G');
        
        if (!$this->showPreview || empty($this->extractedData) || !empty($this->validationErrors)) {
            session()->flash('message', 'Cannot import: Please process files first and ensure no validation errors.');
            session()->flash('alert-class', 'alert-warning');
            return;
        }


        $this->validate([
            'selectedCustomer' => 'required',
            'selectedSeason' => 'required|exists:seasons,id',
        ]);

        $this->isProcessing = true;
        $this->importResults = [];

        try {
            $customer = $this->getCustomerFromSelection();
            $season = Seasons::find($this->selectedSeason);
            $department = $this->getCommissionDepartment();

            $results = $this->getImportService()->processImport(
                $this->extractedData,
                $customer->id,
                $season->id,
                $department->id,
                $this->orderFiles,
                $this->duplicateResolution
            );

            $this->importResults = $results;

            if ($results['success']) {
                session()->flash('message', "Import completed successfully! Created {$results['orders_created']} orders with {$results['lines_created']} lines.");
                session()->flash('alert-class', 'alert-success');
                
                // Reset form
                $this->reset(['orderFiles', 'extractedData', 'showPreview', 'validationErrors']);
            } else {
                session()->flash('message', 'Import failed: ' . implode(', ', $results['errors']));
                session()->flash('alert-class', 'alert-danger');
            }

        } catch (\Exception $e) {
            $this->importResults = [
                'success' => false,
                'orders_created' => 0,
                'lines_created' => 0,
                'errors' => [$e->getMessage()],
            ];
            session()->flash('message', 'Import failed: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        } finally {
            $this->isProcessing = false;
        }
        
    }

    /**
     * Reset the form
     */
    public function resetForm(): void
    {
        $this->reset();
        session()->flash('message', 'Form reset successfully.');
        session()->flash('alert-class', 'alert-info');
    }

    /**
     * Update season selection
     */
    public function updatedSelectedSeason(): void
    {
        // Re-validate if data is already extracted
        if (!empty($this->extractedData)) {
            $this->validateExtractedData();
        }
    }

    /**
     * Update customer selection
     */
    public function updatedSelectedCustomer(): void
    {
        // Reset extracted data when customer changes
        $this->reset(['extractedData', 'showPreview', 'validationErrors', 'orderFiles']);
    }

    /**
     * Get preview data for display
     */
    public function getPreviewData(): array
    {
        if (empty($this->extractedData['orders'])) {
            return [];
        }

        $preview = [];
        foreach ($this->extractedData['orders'] as $orderIndex => $order) {
            foreach ($order['lines'] as $lineIndex => $line) {
                $totalQty = array_sum(array_column($line['quantities'], 'qty'));
                $avgPrice = count($line['quantities']) > 0 
                    ? array_sum(array_column($line['quantities'], 'price')) / count($line['quantities']) 
                    : 0;

                // Create size breakdown string
                $sizeBreakdown = [];
                foreach ($line['quantities'] as $qty) {
                    $sizeBreakdown[] = $qty['size'] . ':' . $qty['qty'];
                }

                $preview[] = [
                    'order_index' => $orderIndex,
                    'line_index' => $lineIndex,
                    'purchase_order' => $order['purchase_order_number'],
                    'order_date' => $order['order_date'],
                    'style_number' => $line['colourway']['style_number'],
                    'colour_name' => $line['colourway']['colour_name'],
                    'customer_exfty_date' => $line['customer_exfty_date'],
                    'total_qty' => $totalQty,
                    'avg_price' => number_format($avgPrice, 2),
                    'size_breakdown' => implode(', ', $sizeBreakdown),
                    'quantities' => $line['quantities'], // Full quantity data for detailed view
                    'source_file' => $order['_source_file'] ?? 'Unknown', // Track which file this came from
                ];
            }
        }

        return $preview;
    }

    /**
     * Merge data from multiple files
     */
    private function mergeExtractedData(array $allFileData): array
    {
        $mergedOrders = [];
        $detectedSeasons = [];
        $filenames = [];

        foreach ($allFileData as $fileData) {
            $mergedOrders = array_merge($mergedOrders, $fileData['orders'] ?? []);
            
            if (!empty($fileData['metadata']['detected_season'])) {
                $detectedSeasons[] = $fileData['metadata']['detected_season'];
            }
            
            if (!empty($fileData['metadata']['filename'])) {
                $filenames[] = $fileData['metadata']['filename'];
            }
        }

        // Check for season conflicts
        $uniqueSeasons = array_unique($detectedSeasons);
        $seasonConflict = count($uniqueSeasons) > 1;

        return [
            'orders' => $mergedOrders,
            'metadata' => [
                'detected_season' => !empty($detectedSeasons) ? $detectedSeasons[0] : null,
                'all_detected_seasons' => $uniqueSeasons,
                'season_conflict' => $seasonConflict,
                'extraction_method' => count($allFileData) > 1 ? 'multiple_files' : 'single_file',
                'filenames' => $filenames,
                'files_processed' => count($allFileData),
            ],
        ];
    }

    /**
     * Detect duplicate purchase orders for the selected customer and season
     */
    private function detectDuplicateOrders(array $allFileData): array
    {
        $poNumbers = [];
        foreach ($allFileData as $fileData) {
            foreach ($fileData['orders'] ?? [] as $order) {
                if (!empty($order['purchase_order_number'])) {
                    $poNumbers[] = strtoupper(trim($order['purchase_order_number']));
                }
            }
        }
        $poNumbers = array_values(array_unique($poNumbers));

        if (empty($poNumbers)) return [];

        $customer = $this->getCustomerFromSelection();
        $seasonId = !empty($this->selectedSeason) ? (int)$this->selectedSeason : null;

        $query = \App\Models\CustomerOrders::query()
            ->where('customers_id', $customer->id)
            ->whereIn('customer_po', $poNumbers);
        if ($seasonId) $query->where('seasons_id', $seasonId);

        $existing = $query->get(['id','customer_po','seasons_id']);

        $duplicates = [];
        foreach ($existing as $order) {
            $duplicates[strtoupper($order->customer_po)] = [
                'existing_order_id' => $order->id,
                'season_id' => $order->seasons_id,
            ];
        }

        return $duplicates;
    }

    /**
     * Get customer model from selection
     */
    private function getCustomerFromSelection(): Customer
    {
        $extractor = CommissionExtractorFactory::make($this->selectedCustomer);
        return Customer::findOrFail($extractor->getCustomerId());
    }

    /**
     * Get commission department
     */
    private function getCommissionDepartment(): Departments
    {
        return Departments::where('type', 'commission')->where('hidden', false)->firstOrFail();
    }

    /**
     * Re-validate extracted data
     */
    private function validateExtractedData(): void
    {
        if (empty($this->extractedData)) {
            return;
        }

        $extractor = CommissionExtractorFactory::make($this->selectedCustomer);
        
        $this->validationErrors = array_merge(
            $this->getImportService()->validateImportData($this->extractedData),
            $extractor->validateCustomerData($this->extractedData)
        );

        $this->showPreview = empty($this->validationErrors);
    }

    /**
     * Get currency symbol based on selected customer from database
     */
    public function getCurrencySymbol(): string
    {
        if (empty($this->selectedCustomer)) {
            return '£'; // Default to GBP
        }

        $customer = Customer::where('name', $this->selectedCustomer)->first();
        
        if ($customer && !empty($customer->currency)) {
            return $customer->currency;
        }
        
        // Default fallback
        return '£';
    }
}

