<?php

namespace App\Http\Livewire\Imports;

use App\Http\Livewire\BaseComponent;
use App\Models\CommissionImportBatch;
use App\Models\CommissionImportBatchFile;
use App\Models\Customer;
use App\Models\Seasons;
use App\Models\Departments;
use App\Services\CommissionImports\CommissionImportService;
use App\Services\CommissionImports\StyleMatchingService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\StreamedResponse;

class CommissionBatchReview extends BaseComponent
{
    public CommissionImportBatch $batch;
    public ?CommissionImportBatchFile $currentFile = null;
    public ?string $selectedSeason = null;
    public array $priceOverrides = [];
    public array $exftyDateOverrides = [];
    public array $colourOverrides = [];
    public array $styleNumberOverrides = [];

    public function mount($batchId, $fileId = null)
    {
        $this->batch = CommissionImportBatch::where('user_id', Auth::id())
            ->with('files')
            ->findOrFail($batchId);

        if ($fileId) {
            $this->currentFile = CommissionImportBatchFile::where('batch_id', $this->batch->id)
                ->findOrFail($fileId);
                
            // Auto-detect season from file metadata or batch
            if (!$this->selectedSeason) {
                // First try to get from file metadata
                $detectedSeason = $this->currentFile->metadata['detected_season'] ?? null;
                
                if ($detectedSeason) {
                    // Find the season by description (e.g., "SS26")
                    $season = \App\Models\Seasons::where('description', $detectedSeason)
                        ->where('locked', false)
                        ->first();
                    
                    if ($season) {
                        $this->selectedSeason = $season->id;
                    }
                } elseif ($this->batch->season_id) {
                    // Fall back to batch season
                    $this->selectedSeason = $this->batch->season_id;
                }
            }
        }
        // If no fileId provided, show file list (currentFile remains null)
    }

    public function render()
    {
        return view('livewire.imports.commission-batch-review');
    }

    public function getSeasonsProperty()
    {
        return Seasons::where('locked', false)->orderBy('created_at', 'desc')->get();
    }

    public function getDepartmentsProperty()
    {
        return Departments::where('type', 'commission')->where('hidden', false)->get();
    }

    public function getPreviewDataProperty(): array
    {
        if (!$this->currentFile || !$this->currentFile->hasExtractedData()) {
            return [];
        }

        $data = $this->currentFile->extracted_data;
        $preview = [];
        
        foreach ($data['orders'] ?? [] as $orderIndex => $order) {
            foreach ($order['lines'] ?? [] as $lineIndex => $line) {
                $totalQty = array_sum(array_column($line['quantities'] ?? [], 'qty'));
                
                // Apply price overrides
                $quantities = $line['quantities'] ?? [];
                foreach ($quantities as $sizeIndex => &$qty) {
                    if (isset($this->priceOverrides[$orderIndex][$lineIndex][$sizeIndex])) {
                        $qty['price'] = $this->priceOverrides[$orderIndex][$lineIndex][$sizeIndex];
                    }
                }
                
                $avgPrice = $totalQty > 0 
                    ? array_sum(array_map(fn($q) => ($q['qty'] ?? 0) * ($q['price'] ?? 0), $quantities)) / $totalQty
                    : 0;

                // Apply ex-factory date override
                $exftyDate = $line['customer_exfty_date'] ?? '';
                if (isset($this->exftyDateOverrides[$orderIndex][$lineIndex])) {
                    $exftyDate = $this->exftyDateOverrides[$orderIndex][$lineIndex];
                }

                // Apply colour override
                $colourName = $line['colourway']['colour_name'] ?? '';
                if (isset($this->colourOverrides[$orderIndex][$lineIndex])) {
                    $colourName = $this->colourOverrides[$orderIndex][$lineIndex];
                }

                // Apply style number override
                $styleNumber = $line['colourway']['style_number'] ?? '';
                if (isset($this->styleNumberOverrides[$orderIndex][$lineIndex])) {
                    $styleNumber = $this->styleNumberOverrides[$orderIndex][$lineIndex];
                }

                $preview[] = [
                    'order_index' => $orderIndex,
                    'line_index' => $lineIndex,
                    'purchase_order' => $order['purchase_order_number'] ?? $data['purchase_order_number'] ?? '',
                    'order_date' => $order['order_date'] ?? '',
                    'style_number' => $styleNumber,
                    'colour_name' => $colourName,
                    'customer_exfty_date' => $exftyDate,
                    'total_qty' => $totalQty,
                    'avg_price' => number_format($avgPrice, 2),
                    'quantities' => $quantities,
                ];
            }
        }

        return $preview;
    }

    public function getCurrencySymbol(): string
    {
        $customer = Customer::where('name', $this->batch->customer_name)->first();
        return $customer?->currency ?? '£';
    }

    public function importFile()
    {
        if (!$this->currentFile || !$this->currentFile->isReadyForReview()) {
            session()->flash('error', 'File is not ready for import.');
            return;
        }

        if (empty($this->selectedSeason)) {
            session()->flash('error', 'Please select a season.');
            return;
        }

        try {
            DB::beginTransaction();

            $customer = Customer::where('name', $this->batch->customer_name)->firstOrFail();
            $season = Seasons::findOrFail($this->selectedSeason);
            $department = Departments::where('type', 'commission')->where('hidden', false)->first();

            if (!$department) {
                throw new \RuntimeException('No commission department found.');
            }

            // Apply price overrides to the extracted data
            $extractedData = $this->currentFile->extracted_data;
            foreach ($this->priceOverrides as $orderIndex => $orderOverrides) {
                foreach ($orderOverrides as $lineIndex => $lineOverrides) {
                    foreach ($lineOverrides as $sizeIndex => $newPrice) {
                        if (isset($extractedData['orders'][$orderIndex]['lines'][$lineIndex]['quantities'][$sizeIndex])) {
                            $extractedData['orders'][$orderIndex]['lines'][$lineIndex]['quantities'][$sizeIndex]['price'] = $newPrice;
                        }
                    }
                }
            }

            // Apply ex-factory date overrides
            foreach ($this->exftyDateOverrides as $orderIndex => $orderOverrides) {
                foreach ($orderOverrides as $lineIndex => $newDate) {
                    if (isset($extractedData['orders'][$orderIndex]['lines'][$lineIndex])) {
                        $extractedData['orders'][$orderIndex]['lines'][$lineIndex]['customer_exfty_date'] = $newDate;
                    }
                }
            }

            // Apply colour overrides
            foreach ($this->colourOverrides as $orderIndex => $orderOverrides) {
                foreach ($orderOverrides as $lineIndex => $newColour) {
                    if (isset($extractedData['orders'][$orderIndex]['lines'][$lineIndex]['colourway'])) {
                        $extractedData['orders'][$orderIndex]['lines'][$lineIndex]['colourway']['colour_name'] = $newColour;
                    }
                }
            }

            // Apply style number overrides
            foreach ($this->styleNumberOverrides as $orderIndex => $orderOverrides) {
                foreach ($orderOverrides as $lineIndex => $newStyleNumber) {
                    if (isset($extractedData['orders'][$orderIndex]['lines'][$lineIndex]['colourway'])) {
                        $extractedData['orders'][$orderIndex]['lines'][$lineIndex]['colourway']['style_number'] = $newStyleNumber;
                    }
                }
            }

            // Get the import service
            $importService = new CommissionImportService(new StyleMatchingService());

            // Import the data with overrides applied
            $results = $importService->processImport(
                $extractedData,
                $customer->id,
                $season->id,
                $department->id
            );

            // Mark file as imported
            $this->currentFile->markAsImported();

            // Update batch counters if not already set
            if (!$this->batch->season_id) {
                $this->batch->update([
                    'season_id' => $season->id,
                    'department_id' => $department->id,
                    'customer_id' => $customer->id,
                ]);
            }

            DB::commit();

            session()->flash('success', "File imported successfully! Created {$results['orders_created']} order(s) with {$results['lines_created']} line(s).");

            // Redirect back to file list in this batch
            return redirect()->route('commission.batch.review', ['batchId' => $this->batch->id]);

        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash('error', 'Import failed: ' . $e->getMessage());
        }
    }

    public function loadNextFile(): void
    {
        $this->currentFile = $this->batch->files()
            ->where('status', 'extracted')
            ->where('id', '>', $this->currentFile?->id ?? 0)
            ->first();

        $this->priceOverrides = [];
        $this->exftyDateOverrides = [];
        $this->colourOverrides = [];
        $this->styleNumberOverrides = [];
    }

    public function loadPreviousFile(): void
    {
        $this->currentFile = $this->batch->files()
            ->where('status', 'extracted')
            ->where('id', '<', $this->currentFile?->id ?? PHP_INT_MAX)
            ->orderBy('id', 'desc')
            ->first();

        $this->priceOverrides = [];
        $this->exftyDateOverrides = [];
        $this->colourOverrides = [];
        $this->styleNumberOverrides = [];
    }

    public function retryFile($fileId = null): void
    {
        // If fileId is provided, use that, otherwise use currentFile
        if ($fileId) {
            $file = CommissionImportBatchFile::where('batch_id', $this->batch->id)
                ->findOrFail($fileId);
        } else {
            $file = $this->currentFile;
        }

        if (!$file) {
            session()->flash('error', 'File not found.');
            return;
        }

        // Determine if this is a re-extraction or initial retry
        $wasImported = $file->status === 'imported';
        $wasExtracted = $file->status === 'extracted';
        $wasFailed = $file->status === 'failed';

        // For imported/extracted files: reset to pending and clear data
        // For failed files: just reset to pending
        if ($wasImported || $wasExtracted) {
            $file->update([
                'status' => 'pending',
                'extracted_data' => null,
                'error_message' => null,
                'total_orders' => 0,
                'total_lines' => 0,
                'total_quantity' => 0,
                'metadata' => array_merge($file->metadata ?? [], ['is_retry' => true, 'previous_status' => $file->status]),
            ]);
            
            // Note: We don't change the batch counters here because the file will be reprocessed
            // The job will handle counter updates based on is_retry flag
        } else {
            // For failed files, just reset status
            $file->update([
                'status' => 'pending',
                'error_message' => null,
                'metadata' => array_merge($file->metadata ?? [], ['is_retry' => true]),
            ]);
        }

        // Dispatch the job again
        \App\Jobs\ProcessCommissionFile::dispatch($file);

        session()->flash('success', 'File queued for re-extraction. The AI will process it again.');
        
        // Refresh the batch and file
        $this->batch->refresh();
        if ($this->currentFile && $this->currentFile->id === $file->id) {
            $this->currentFile->refresh();
        }
    }

    public function updatePrice(int $orderIndex, int $lineIndex, int $sizeIndex, $newPrice): void
    {
        $this->priceOverrides[$orderIndex][$lineIndex][$sizeIndex] = (float) $newPrice;
    }

    public function updateLinePrice(int $orderIndex, int $lineIndex, $newPrice): void
    {
        if (!$this->currentFile || !isset($this->currentFile->extracted_data['orders'][$orderIndex]['lines'][$lineIndex])) {
            return;
        }

        $line = $this->currentFile->extracted_data['orders'][$orderIndex]['lines'][$lineIndex];
        foreach ($line['quantities'] ?? [] as $sizeIndex => $qty) {
            $this->priceOverrides[$orderIndex][$lineIndex][$sizeIndex] = (float) $newPrice;
        }
    }

    public function updateExftyDate(int $orderIndex, int $lineIndex, $newDate): void
    {
        $this->exftyDateOverrides[$orderIndex][$lineIndex] = $newDate;
    }

    public function updateColour(int $orderIndex, int $lineIndex, $newColour): void
    {
        $this->colourOverrides[$orderIndex][$lineIndex] = $newColour;
    }

    public function updateStyleNumber(int $orderIndex, int $lineIndex, $newStyleNumber): void
    {
        $this->styleNumberOverrides[$orderIndex][$lineIndex] = $newStyleNumber;
    }

    public function skipFile($fileId = null, $reason = 'Not needed')
    {
        // If fileId is provided, use that, otherwise use currentFile
        if ($fileId) {
            $file = CommissionImportBatchFile::where('batch_id', $this->batch->id)
                ->findOrFail($fileId);
        } else {
            $file = $this->currentFile;
        }

        if (!$file) {
            session()->flash('error', 'File not found.');
            return;
        }

        $file->markAsSkipped($reason);
        
        session()->flash('success', 'File marked as skipped.');
        
        // Refresh the batch and file
        $this->batch->refresh();
        if ($this->currentFile && $this->currentFile->id === $file->id) {
            // Redirect back to file list if we're viewing this file
            return redirect()->route('commission.batch.review', ['batchId' => $this->batch->id]);
        }
    }

    public function downloadFile($fileId): StreamedResponse
    {
        $file = CommissionImportBatchFile::where('batch_id', $this->batch->id)
            ->findOrFail($fileId);

        if (!Storage::exists($file->file_path)) {
            session()->flash('error', 'File not found.');
            return response()->streamDownload(function() {}, '');
        }

        return Storage::download($file->file_path, $file->original_filename);
    }
}
