<?php

namespace App\Http\Livewire\Imports;

use App\Http\Livewire\BaseComponent;
use App\Models\CommissionImportBatch;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Livewire\WithPagination;

class CommissionBatchList extends BaseComponent
{
    use WithPagination;

    public string $search = '';
    public string $statusFilter = 'all';

    protected $queryString = ['search', 'statusFilter'];

    public function render()
    {
        $batches = CommissionImportBatch::query()
            ->where('user_id', Auth::id())
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('batch_name', 'like', "%{$this->search}%")
                      ->orWhere('customer_name', 'like', "%{$this->search}%");
                });
            })
            ->when($this->statusFilter !== 'all', function ($query) {
                $query->where('status', $this->statusFilter);
            })
            ->with(['files' => function ($query) {
                $query->selectRaw('batch_id, status, count(*) as count')
                      ->groupBy('batch_id', 'status');
            }])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('livewire.imports.commission-batch-list', [
            'batches' => $batches,
        ]);
    }

    public function deleteBatch($batchId)
    {
        $batch = CommissionImportBatch::where('user_id', Auth::id())
            ->findOrFail($batchId);

        // Delete all stored files
        foreach ($batch->files as $file) {
            if (Storage::exists($file->file_path)) {
                Storage::delete($file->file_path);
            }
        }

        $batch->delete();

        session()->flash('success', 'Batch deleted successfully.');
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingStatusFilter()
    {
        $this->resetPage();
    }
}
