<?php

namespace App\Http\Livewire\Imports;

use App\Http\Livewire\BaseComponent;
use App\Jobs\ProcessCommissionFile;
use App\Models\CommissionImportBatch;
use App\Models\CommissionImportBatchFile;
use App\Services\CommissionImports\CommissionExtractorFactory;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Livewire\WithFileUploads;
use Livewire\Attributes\Validate;

class CommissionBatchImport extends BaseComponent
{
    use WithFileUploads;

    #[Validate('required')]
    public string $selectedCustomer = '';

    #[Validate('nullable')]
    public string $batchName = '';

    #[Validate('required|array|min:1')]
    public array $files = [];

    public bool $isUploading = false;

    public function mount()
    {
        //
    }

    public function render()
    {
        return view('livewire.imports.commission-batch-import');
    }

    public function getAvailableCustomersProperty(): array
    {
        return CommissionExtractorFactory::getAvailableCustomers();
    }

    public function uploadBatch()
    {
        $this->validate();

        $this->isUploading = true;

        try {
            // Create the batch record
            $batch = CommissionImportBatch::create([
                'batch_name' => $this->batchName ?: 'Batch ' . now()->format('Y-m-d H:i'),
                'customer_name' => $this->selectedCustomer,
                'user_id' => Auth::id(),
                'total_files' => count($this->files),
                'status' => 'pending',
                'started_at' => now(),
            ]);

            // Store each file and create batch file records
            foreach ($this->files as $index => $file) {
                $originalFilename = $file->getClientOriginalName();
                $extension = $file->getClientOriginalExtension();
                
                // Store file in commission_imports directory
                $storedPath = $file->store("commission_imports/batch_{$batch->id}", 'local');

                // Create batch file record
                $batchFile = CommissionImportBatchFile::create([
                    'batch_id' => $batch->id,
                    'original_filename' => $originalFilename,
                    'stored_filename' => basename($storedPath),
                    'file_path' => $storedPath,
                    'status' => 'pending',
                ]);

                // Dispatch job to process this file
                ProcessCommissionFile::dispatch($batchFile);
            }

            session()->flash('success', "Batch created successfully! Processing {$batch->total_files} file(s) in the background.");
            
            // Redirect to batch list
            return redirect()->route('commission.batch.list');

        } catch (\Exception $e) {
            session()->flash('error', 'Failed to create batch: ' . $e->getMessage());
            $this->isUploading = false;
        }
    }
}
