<?php

namespace App\Http\Livewire\Imports;

use Gemini;
use OpenAI;
use OpenAI\Client;
use Gemini\Data\Blob;
use Livewire\Component;
use App\Models\Customer;
use Spatie\PdfToText\Pdf;
use Gemini\Enums\MimeType;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithFileUploads;
use App\Models\CustomerOrders;
use Illuminate\Support\Carbon;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Http;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ImportExcelWithValues;

class AsosAnantaInvoice extends BaseComponent
{
    use WithFileUploads;
    public $packingList = '';
    public $answer = '';
    public $question = '';

    public $packingListFiles = [];

    public $customer = 6;

	#[Computed]
	public function customers(){
		return Customer::allCached();
	}

    public function mount()
    {
        // Set default values for date range
        $this->startDate = Carbon::now()->subDays(3)->toDateString();
        $this->endDate = Carbon::now()->addDay()->toDateString();
    }

    public function uploadPackingList()
    {
        if (empty($this->packingListFiles)) {
            session()->flash('message', 'No files uploaded.');
            return;
        }
        elseif($this->buttonDisabled()){
            session()->flash('message', 'Too many files uploaded. Max 10.');
            return;
        }
        elseif(!$this->customer){
            session()->flash('message', 'Select a customer.');
            return;
        }

        // Loop through each uploaded file
        foreach ($this->packingListFiles as $file) {
            $extension = $file->getClientOriginalExtension();
            $fileContent = null;

            if ($extension === 'xlsx') {
                $fileContent = $plain   = $this->excelToText($file);
            } else {
                session()->flash('message', 'Unsupported file type: ' . $file->getClientOriginalName());
                continue;
            }
            $extracted = $this->extractInfo($fileContent);
            $this->updateDrop($extracted);
        }

    }

    private function excelToText($file): string
    {
        $rows = Excel::toArray(null, $file)[0]; // first sheet
        $lines = [];

        foreach ($rows as $row) {
            // Escape any tabs that might appear in the cell just in case
            $cleanCells = array_map(function ($cell) {
                return str_replace("\t", ' ', trim((string) $cell));
            }, $row);

            $lines[] = implode("\t", $cleanCells);
        }

        return implode("\n", $lines);
    }

    private function updateDrop($data){
        $lines = $data['lines'];
        $invoiceNo = $data['invoice_no'];
        $invoiceDate = Carbon::createFromFormat('d/m/Y', $data['invoice_date'])->format('Y-m-d');

        foreach($lines as $line){
            // dd($data);
            $shipmentLine = ShipmentLine::with('customer_order_lines.colourways')
                                ->whereRelation('customer_order_lines.customer_orders', 'customer_po', $line['po_number'])->get();
            if($shipmentLine->count() > 1){
                throw new \Exception("Too many shipment lines matched for PO: " . $line['po_number']);
            }
            $shipmentLine = $shipmentLine->first();
            $shipmentLine->update([
                'factory_invoice' => $invoiceNo,
                'exfty' => $invoiceDate,
                'asn' => $line['asn_number'],
            ]);
            $shipmentLine->customer_order_lines->colourways->update([
                'composition' => $line['fabric_composition'],
            ]);

            // dd($shipmentLine);
        }

        session()->flash('message', 'Complete!');
		session()->flash('alert-class', 'alert-success');
    }

    private function extractInfo($text)
    {
        $lines = [];

        // 1. Get invoice number & first date
        if (preg_match('/6\. INVOICE NO:\s+([^\t]+)\s+DATE:\s+(\d+)/', $text, $m)) {
            $invoiceNo = trim($m[1]);
            $excelSerial = (int) $m[2];
            // Excel date 1900 epoch
            $invoiceDate = \Carbon\Carbon::createFromDate(1899, 12, 30)->addDays($excelSerial)->format('d/m/Y');
        } else {
            $invoiceNo = null;
            $invoiceDate = null;
        }

        // 2. Extract each invoice row (PO, ASN, Fabric Compositon)
        // This matches: PO number, ASN number, STYLE, Fabric Composition
        preg_match_all(
            '/\b(\d{11,12})\b\t(\d{14})\t([A-Z0-9-]+)\t([^\t]+?)\t([A-Z ]+)\t\d{10}\t/',
            $text,
            $matches,
            PREG_SET_ORDER
        );

        $collection = collect($matches)->map(fn($m) => [
            'po_number' => $m[1],
            'asn_number' => $m[2],
            'style_no' => $m[3],
            'fabric_composition' => trim($m[4]),
            'colour' => trim($m[5]),
        ]);
        // dd($collection, $text);
        return [
            'invoice_no'   => $invoiceNo,
            'invoice_date' => $invoiceDate,
            'lines'        => $collection,
        ];
    }

    private function convertPdfToText($filePath)
    {
        // Convert PDF to plain text
        return Pdf::getText($filePath);
    }

    public function render()
    {
        Gate::authorize('order:update');
        if($this->buttonDisabled()){
            session()->flash('message', 'Max 30 files.');
            session()->flash('alert-class', 'alert-warning');
        }
        // dd($this->getData());
        return view('livewire.imports.asos-ananta-invoice');
    }

    private function buttonDisabled(){
        return count($this->packingListFiles) > 30;
    }
}
