<?php

namespace App\Http\Livewire\Imports;

use App\Models\Gauge;
use App\Models\Price;
use App\Models\Sizes;
use App\Models\Design;
use App\Models\Styles;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Models\Departments;
use App\Models\ShipmentLine;
use App\Models\StyleVersions;
use App\Models\CommodityCodes;
use App\Models\CustomerOrders;
use Illuminate\Support\Carbon;
use App\Models\CustomerAddress;
use App\Models\ShipmentLineSizes;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use App\Models\CustomerOrderLineQuantities;
use Livewire\WithFileUploads;

class Asos extends Component
{
	use WithFileUploads;

	public $imported;
	public $csvFile;
    public function render()
    {
        // $this->importOrders();
        return view('livewire.imports.orders');
    }

    public function importOrders(){
        Gate::check('adminONLY');

		$this->validate([
			'csvFile' => 'required|file|mimes:csv,txt|max:10240', // 10MB max
		]);

		// Read and validate CSV format
		$csvData = file($this->csvFile->getRealPath(), FILE_SKIP_EMPTY_LINES);
		
		$validation = $this->validateCSVFormat($csvData);
		if (!$validation['valid']) {
			$this->addError('csvFile', $validation['message']);
			return;
		}

		$csv = array_map("str_getcsv", $csvData);
		$keys = array_shift($csv);
		foreach ($csv as $i=>$row) {
			$csv[$i] = (object) array_combine($keys, $row);
		}
		$csv = collect($csv);

		$this->importData($csv);
		$this->imported = TRUE;
    }

	private function validateCSVFormat($csvData)
	{
		// Required columns that must be present
		$requiredColumns = [
			'season',
			'customer_address',
			'description',
			'colourways_name',
			'customer_po',
			'exfty',
		];

		// Optional but commonly used columns
		$expectedColumns = [
			'design_id', 'date_of_issue', 'customer_ref', 'customer_sample_number',
			'customer_description', 'composition', 'colour_approval',
			'green_seal_approval_comments', 'gold_seal_status_comments',
			'photo_shoot_sample', 'accessories', 'testing_comments',
			'cust_ship_no', 'incomplete_flag', 'order_received',
			'sent_to_factory', 'factory_cust_date', 'colour_approved_date',
			'lab_dips_sent_date', 'wh_cust_date', 'start_knit',
			'bulk_yarn_order_date', 'bulk_yarn_order_comments', 'start_knit_comments',
			'XXS', 'XS', 'S', 'M', 'L', 'XL', 'XXL',
			'sku_XXS', 'sku_XS', 'sku_S', 'sku_M', 'sku_L', 'sku_XL', 'sku_XXL',
			'truck_plates', 'collection_date', 'delivery_date', 'complete',
			'sealer_deadline', 'notes', 'asn_transferred', 'asn',
			'cmt', 'cmt_status', 'quote', 'quote_status',
		];

		if (empty($csvData)) {
			return ['valid' => false, 'message' => 'CSV file is empty.'];
		}

		// Parse first row as header
		$headerRow = str_getcsv($csvData[0]);
		$headerRow = array_map('trim', $headerRow);

		// Check for empty header
		if (empty($headerRow) || (count($headerRow) === 1 && empty($headerRow[0]))) {
			return ['valid' => false, 'message' => 'CSV header row is empty or invalid.'];
		}

		// Check for required columns
		$missingColumns = [];
		foreach ($requiredColumns as $column) {
			if (!in_array($column, $headerRow)) {
				$missingColumns[] = $column;
			}
		}

		if (!empty($missingColumns)) {
			return [
				'valid' => false,
				'message' => 'Missing required columns: ' . implode(', ', $missingColumns)
			];
		}

		// Check data rows for issues
		for ($i = 1; $i < count($csvData); $i++) {
			$row = str_getcsv($csvData[$i]);
			$row = array_map('trim', $row);

			// Check for empty rows (all fields empty)
			$nonEmptyFields = array_filter($row, function($field) {
				return !empty($field);
			});

			if (empty($nonEmptyFields)) {
				return [
					'valid' => false,
					'message' => "Empty row found at line " . ($i + 1) . ". Please remove empty rows from the CSV."
				];
			}

			// Check if row looks like a duplicate header
			$matchingHeaderFields = 0;
			foreach ($headerRow as $header) {
				if (in_array($header, $row)) {
					$matchingHeaderFields++;
				}
			}

			// If more than 50% of fields match header, it's likely a duplicate header
			if ($matchingHeaderFields > (count($headerRow) * 0.5)) {
				return [
					'valid' => false,
					'message' => "Duplicate header row found at line " . ($i + 1) . ". Please remove duplicate headers from within the data."
				];
			}

			// Check if row has correct number of columns
			if (count($row) !== count($headerRow)) {
				return [
					'valid' => false,
					'message' => "Row at line " . ($i + 1) . " has " . count($row) . " columns, but header has " . count($headerRow) . " columns. Please check for missing or extra delimiters."
				];
			}
		}

		return ['valid' => true, 'message' => 'CSV format is valid.'];
	}

	private function importData($csv){
		foreach($csv as $l=>$line){ //IMPORT DATA
			foreach($line as $i=>$item){
				$line->$i = trim(strip_tags($line->$i));
			}

			$line->customer_description = str_replace("|", "\r\n", $line->customer_description);
            $line->composition = str_replace("|", "\r\n", $line->composition);
            $line->colour_approval = str_replace("|", "\r\n", $line->colour_approval);
            $line->green_seal_approval_comments = str_replace("|", "\r\n", $line->green_seal_approval_comments);
            $line->accessories = str_replace("|", "\r\n", $line->accessories);


		//GET RE-USED VALUES
		$factory = Suppliers::where('name', "ananta")->first();
		$customer = Customer::where('name', 'asos')->first();
            $customerAddress = CustomerAddress::firstOrCreate(['name' => $line->customer_address, 'customer_id' => $customer->id], ['line1' => "na", 'countries_id' => $customer->countries_id]);
		$season = Seasons::where('description', $line->season)->first();
			$department = Departments::where('description', "kwa")->first();

            // dd($customer, $season, $factory, $customerAddress, $department);


			//CREATE/FIND DESIGN
            if(($design = Design::find($line->design_id)) == NULL){
                $design = Design::firstOrCreate(
                    [
                        'factory_id' => $factory->id ?? $design->factory_id,
                        'customers_id' => $customer->id ?? $design->customers_id,
                        'description' => $line->description,
                    ],
                    [
                        'gauge_id' => 4,
                        'issued' => !empty($line->date_of_issue) ? Carbon::createFromFormat('d/m/Y', $line->date_of_issue)->format('Y-m-d') : NULL,
                        'yarn' => $line->composition,
                    ]
                );
            }

            // dd($design);

			//CREATE/FIND STYLE
			// Strip 'SAMP' prefix from customer_ref if present
			$customerRef = preg_replace('/^SAMP/', '', $line->customer_ref);
			
			$style = Styles::firstOrCreate(
				[
					'designs_id' => $design->id,
					'seasons_id' => $season->id,
					'departments_id' => $department->id,
					'customers_id' => $customer->id ?? $design->customers_id,
				],
				[
					'customer_ref' => $customerRef,
					'commodity_codes_id' => "6001100000",
					'category' => "ladies",
					'customer_samp_no' => $line->customer_sample_number,
					'green_seal_approval_comments' => $line->green_seal_approval_comments,
					'gold_seal_status_comments' => $line->gold_seal_status_comments,
					'photo_shoot_sample_comments' => $line->photo_shoot_sample,
				]
			);
            // dd($style);

			//CREATE/FIND STYLE VERSION
            if(($styleVersion = $style->style_versions->first()) == NULL)
			$styleVersion = StyleVersions::firstOrCreate(
				[
					'name' => "A",
					'styles_id' => $style->id,
					'factory_id' => $factory->id ?? $design->factory_id,
				],
				[
					'gauge_id' => $gauge->id ?? $design->gauge_id,
				]
			);

            // dd($styleVersion);

			//CREATE/FIND COLOURWAY VERSION
            if(($colourway = Colourways::where('style_versions_id', $styleVersion->id)->where('name', 'like', "%{$line->colourways_name}%")->first()) == NULL){

                $colourway = Colourways::firstOrCreate(
                    [
                        'name' => $line->colourways_name,
                        'style_versions_id' => $styleVersion->id,
                    ],
                    [
                        'colour_type' => "solid",
                        'customer_description' => $line->customer_description,
                        'composition' => $line->composition,
                        'colour_approval_comments' => $line->colour_approval,
                        'accessories' => $line->accessories,
                        'testing_comments' => $line->testing_comments,
                    ]
                );
            }

            // dd($colourway);


			//CREATE/FIND ORDER
			$order = CustomerOrders::firstOrCreate(
				[
					'customer_po' => $line->customer_po,
					'seasons_id' => $season->id,
					'departments_id' => $department->id,
					'customers_id' => $customer->id ?? $design->customers_id,
				],
				[
					// 'order_date' => Carbon::createFromFormat('d/m/Y', $line->order_date)->format('Y-m-d'),
					'incoterms' => "FOB",
					'cust_ship_no' => $line->cust_ship_no ?? NULL,
					'customer_addresses_id' => $customerAddress->id ?? NULL,
					'incomplete_flag' => !empty($line->incomplete_flag) ? $line->incomplete_flag : 0,

                    'order_received' => $line->order_received,
				]
			);

            // dd($order);

            // dd(Carbon::createFromFormat('d/m/Y', $line->sent_to_factory)->format('Y-m-d'));

			//CREATE ORDER LINE
			$orderLine = CustomerOrderLines::firstOrCreate(
				[
					'customer_orders_id' => $order->id,
					'colourways_id' => $colourway->id,
				],
				[
					'order_sent_factory' => !empty($line->sent_to_factory) ? Carbon::createFromFormat('d/m/Y', $line->sent_to_factory)->format('Y-m-d') : NULL,
					'factory_cust_date' => !empty($line->factory_cust_date) ? Carbon::createFromFormat('d/m/Y', $line->factory_cust_date)->format('Y-m-d') : NULL,
					'colour_approved_date' => !empty($line->colour_approved_date) ? Carbon::createFromFormat('d/m/Y', $line->colour_approved_date)->format('Y-m-d') : NULL,
					'lab_dips_sent_date' => !empty($line->lab_dips_sent_date) ? Carbon::createFromFormat('d/m/Y', $line->lab_dips_sent_date)->format('Y-m-d') : NULL,
					'colour_approval_comments' => $line->colour_approval,
					'wh_cust_date' => !empty($line->wh_cust_date) ? Carbon::createFromFormat('d/m/Y', $line->wh_cust_date)->format('Y-m-d') : NULL,
					'start_knit' => !empty($line->start_knit) ? Carbon::createFromFormat('d/m/Y', $line->start_knit)->format('Y-m-d') : NULL,

                    'bulk_yarn_order_date' => $line->bulk_yarn_order_date,
                    'bulk_yarn_order_comments' => $line->bulk_yarn_order_comments,
                    'start_knit_comments' => $line->start_knit_comments,
                ]
			);

            // dd($orderLine);

			foreach(['XXS', 'XS', 'S', 'M', 'L', 'XL', 'XXL'] as $qty){
                $sku = 'sku_' . $qty;

                if(!empty($line->$qty)){
                    $sizeID = Sizes::firstOrCreate(['name' => $qty])->id;

                    $orderLineSize = CustomerOrderLineQuantities::firstOrNew(
                        [
                            'sizes_id' => $sizeID,
                            'customer_order_lines_id' => $orderLine->id,
                            'SKU' => $line->$sku,
                        ],
                        ['qty' => !empty($line->$qty) ? $line->$qty : 0,]
                    );
                    $orderLineSize->save();
                }
			}

            // dd($orderLineSize);
			$drop = ShipmentLine::firstOrCreate(
				[
					'customer_order_lines_id' => $orderLine->id,
					'exfty' => Carbon::createFromFormat('d/m/Y', $line->exfty)->format('Y-m-d'),
				],
				[
					'truck_plates' => $line->truck_plates ?? "",
					'collection_date' => !empty($line->collection_date) ? Carbon::createFromFormat('d/m/Y', $line->collection_date)->format('Y-m-d') : NULL,
					'delivery_date' => !empty($line->delivery_date) ? Carbon::createFromFormat('d/m/Y', $line->delivery_date)->format('Y-m-d') : NULL,
					'complete' => !empty($line->complete) ? $line->complete : 0,
					'sealer_deadline' => !empty($line->sealer_deadline) ? Carbon::createFromFormat('d/m/Y', $line->sealer_deadline)->format('Y-m-d') : NULL,
					'notes' => $line->notes ?? "",
					'asn_transferred_comments' => $line->asn_transferred,
					'asn' => $line->asn,
				]
			);
            // $drop->asn_transferred_comments = $line->asn_transferred;
			$drop->save();

            // dd($drop);

            foreach(['XXS', 'XS', 'S', 'M', 'L', 'XL', 'XXL'] as $qty){
                if(!empty($line->$qty)){
                    $sizeID = Sizes::firstOrCreate(['name' => $qty])->id;
                    $shipmentLineSize = ShipmentLineSizes::firstOrCreate(
                        [
                            'sizes_id' => $sizeID,
                            'shipment_line_id' => $drop->id,
                        ],
                        ['qty' => !empty($line->$qty) ? $line->$qty : 0,]
                    );
                    $shipmentLineSize->save();
                }
			}

            // dd($shipmentLineSize);

            if(!empty($line->cmt) || !empty($line->quote)){
                $price = Price::firstOrCreate([
                    'style_versions_id' => $styleVersion->id,
                ],
                [
                    'model' => "full_factored",
                    'cmt' => $line->cmt,
                    'cmt_status' => $line->cmt_status ?? "confirmed",
                    'quote' => $line->quote,
                    'quote_status' => $line->quote_status ?? "confirmed",
                ]);
            }


		}
	}
}
