<?php

namespace App\Http\Livewire\Finance;

use App\Helper\Conversions;
use App\Http\Livewire\FilterableComponent;
use App\Http\Livewire\Traits\Filterable;
use App\Models\Customer;
use App\Models\Departments;
use App\Models\Samples;
use App\Models\Seasons;
use App\Models\Sizes;
use App\Models\Suppliers;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Gate;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;
use Livewire\Component;
use Livewire\WithPagination;

class SamplesInvoicingNew extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideFilters = false;

    public $selected = [];
    public $selectAllCheckbox = false;
    public $currentPageSampleIds = [];

    public $season,
        $from,
        $to,
        $customer,
        $factory,
        $coo,
        $department,
        $category,
        $fn_complete = 0,
        $search,
        $paginate = 200;

    protected function filterKey(): string
    {
        return 'financesamplesfilters:new';
    }

    protected function columnsVersion(): float
    {
        return 1;
    }

    protected function filters(): array
    {
        return ['season', 'from', 'to', 'customer', 'factory', 'coo', 'department', 'category', 'fn_complete', 'search', 'paginate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    public function mount()
    {
        $this->loadFilterSettings();
        // Ensure paginate has a default value if not set
        if (empty($this->paginate)) {
            $this->paginate = 200;
        }
        
        // Clear any stale selection on mount
        $this->selected = [];
        $this->selectAllCheckbox = false;
    }

    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            if (in_array($propertyName, ['season', 'from', 'to', 'customer', 'factory', 'coo', 'department', 'category', 'fn_complete', 'search', 'paginate'])) {
                $this->resetPage();
                $this->clearSelection();
            }
            $this->saveFilterSettings();
        }
    }

    public function updatedPaginate()
    {
        $this->resetPage();
        $this->clearSelection();
    }

    public function rules(): array
    {
        return [
            'season' => 'nullable|integer',
            'from' => 'nullable|date',
            'to' => 'nullable|date|after_or_equal:from',
            'customer' => 'nullable|integer',
            'factory' => 'nullable|integer',
            'coo' => 'nullable|integer',
            'department' => 'nullable|integer',
            'category' => 'nullable|string|in:mens,ladies,childrens,accessories',
            'fn_complete' => 'nullable|boolean',
            'search' => 'nullable|string|max:200',
            'paginate' => 'nullable|integer|in:10,50,100,200,500,2000',
        ];
    }

    public function updatedSelectAllCheckbox()
    {
        if ($this->selectAllCheckbox) {
            $this->selectAll();
        } else {
            $this->deselectAll();
        }
    }

    public function toggleSelectAll()
    {
        if ($this->selectAllCheckbox) {
            $this->selectAll();
        } else {
            $this->deselectAll();
        }
    }

    public function selectAll()
    {
        $this->selected = [];
        if ($this->currentPageSampleIds) {
            foreach ($this->currentPageSampleIds as $id) {
                $this->selected[$id] = true;
            }
        }
        $this->selectAllCheckbox = true;
        $this->updatedSelected();
    }

    public function deselectAll()
    {
        $this->clearSelection();
    }

    public function updatedSelected()
    {
        $this->dispatch('load-selected', selected: $this->selected);
        $this->captureSelectedSamplesData();
    }

    #[On('close-modal')]
    public function reload()
    {
        $this->clearSelection();
        $this->render();
    }

    public function openInvoiceGenerator(): void
    {
        try {
            // Prune selection to current page and verify
            $currentIds = collect($this->currentPageSampleIds);
            $selectedIds = collect($this->selected)->filter()->keys()->filter(fn($id) => $currentIds->contains($id))->values();


            if ($selectedIds->isEmpty()) {
                $totalSelected = collect($this->selected)->filter()->count();
                if ($totalSelected > 0) {
                    session()->flash('message', "You have {$totalSelected} samples selected, but none are on the current page. Please navigate to the page containing your selected samples, or select samples from the current page.");
                    session()->flash('alert-class', 'alert-info');
                } else {
                    session()->flash('message', 'No rows selected. Please tick at least one row.');
                    session()->flash('alert-class', 'alert-warning');
                }
                return;
            }

            // Ensure the generator receives the latest selection payload right before opening
            $this->captureSelectedSamplesData();

            // Open the modal
            $this->dispatch('open-modal', name: 'invoice-generator');
        } catch (\Throwable $e) {
            \Log::error('openInvoiceGenerator failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            session()->flash('message', 'Failed to open invoice generator: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        }
    }

    /**
     * Clear all selection-related state and notify dependent components
     */
    protected function clearSelection(): void
    {
        $this->selected = [];
        $this->selectAllCheckbox = false;
        $this->currentPageSampleIds = [];
        $this->dispatch('load-selected', selected: []);
        $this->dispatch('load-selected-samples', selected: []);
    }

    #[Computed(cache: true, key: 'factories-with-countries')]
    public function factories()
    {
        return Suppliers::with('countries')->where('type', 'factory')->get();
    }

    #[Computed]
    public function sizes()
    {
        return Sizes::get();
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::allCached();
    }

    #[Computed]
    public function customers()
    {
        return Customer::allCached();
    }

    #[Computed]
    public function departments()
    {
        return Departments::allCached();
    }

    public function render()
    {
        Gate::authorize('finance:read');

        $query = Samples::select([
            'samples.id as id',
            'seasons.description as season',
            'customers.name as customer',
            'designs.id as rt_no',
            'styles.customer_ref as customer_ref',
            'designs.description as description',
            'sizes.name as size',
            'colourways.name as colourway',
            'samples.po as po',
            'sample_types.name as sample_type',
            'sample_types.id as sample_type_id',
            'suppliers.name as factory',
            'styles.carryover as carryover',
            'samples.qty as qty',
            'samples.date_expected as date_expected',
            'samples.date_received as date_received',
            'samples.date_sent as date_sent',
            'samples.fn_notes as fn_notes',
            'samples.rt_invoice as rt_invoice',
            'samples.fty_invoice as fty_invoice',
            'samples.finance_comments as finance_comments',
            'samples.do_not_charge as do_not_charge',
            'customers.settings as cust_settings',
            'colourways.id as colourways_id',
            'colourways.colour_type as colour_type',
            'customers.currency as cust_currency',
            'suppliers.currency as fact_currency',
            'styles.id as styles_id',
            'samples.fn_complete as fn_complete',
            'colourways.cancelled as cw_cancelled',
            'styles.cancelled as s_cancelled',
            'style_versions.name as style_version',
            DB::raw("(SELECT
                CONCAT(
                    '[',
                    GROUP_CONCAT(DISTINCT JSON_OBJECT('id', co.id, 'po', co.customer_po) SEPARATOR ','),
                    ']'
                ) AS orders
            FROM customer_order_lines col
            JOIN customer_orders co ON co.id = col.customer_orders_id
            JOIN colourways c ON c.id = col.colourways_id
            JOIN style_versions sv ON sv.id = c.style_versions_id
            JOIN styles s ON s.id = sv.styles_id
            WHERE s.id = styles.id) as orders"),
            'seasons.id as seasons_id',
            'suppliers.id as factory_id',
            'suppliers.countries_id as coo',
            'styles.departments_id as departments_id',
            'styles.category as category',
            'customers.id as customers_id',
            'samples.sizes_id as sizes_id',
        ])
            ->leftJoin('colourways', 'colourways.id', '=', 'samples.colourways_id')
            ->leftJoin('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->leftJoin('suppliers', 'suppliers.id', '=', 'style_versions.factory_id')
            ->leftJoin('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->leftJoin('customers', 'customers.id', '=', 'styles.customers_id')
            ->leftJoin('seasons', 'seasons.id', '=', 'styles.seasons_id')
            ->leftJoin('designs', 'designs.id', '=', 'styles.designs_id')
            ->leftJoin('sizes', 'sizes.id', '=', 'samples.sizes_id')
            ->leftJoin('sample_types', 'sample_types.id', '=', 'samples.sample_types_id')
            ->whereNotNull('samples.date_sent')
            ->whereNull('samples.deleted_at');

        // Eager for price() relationships
        $query->with(['colourways', 'colourways.style_versions.styles.customers']);

        // Filters
        $query = $query->search($this->search);
        if (!empty($this->season))
            $query->where('styles.seasons_id', '=', $this->season);
        if (!empty($this->factory))
            $query->where('style_versions.factory_id', '=', $this->factory);
        if (!empty($this->department))
            $query->where('styles.departments_id', '=', $this->department);
        if (!empty($this->coo))
            $query->where('suppliers.countries_id', '=', $this->coo);
        if (!empty($this->category))
            $query->where('styles.category', '=', $this->category);
        if (!empty($this->customer))
            $query->where('styles.customers_id', '=', $this->customer);
        if ($this->fn_complete == FALSE)
            $query->where('fn_complete', '!=', true);
        if (!empty($this->from) && !empty($this->to))
            $query->whereBetween('samples.date_sent', [$this->from, $this->to]);

        // Sort by sent date, then size order
        $query->orderBy('samples.date_sent')->orderBy('sizes.order');

        $paginator = $query->paginate($this->paginate);

        // Attach pricing and derived fields onto paginator items
        $this->augmentSamples($paginator->getCollection());

        $this->currentPageSampleIds = $paginator->getCollection()->pluck('id')->toArray();
        // Prune selection to ids visible on the current page to prevent leakage across pages/filters
        $this->selected = collect($this->selected)
            ->only($this->currentPageSampleIds)
            ->toArray();

        // Totals (page and selected)
        $pageTotals = $this->calculateTotals($paginator->getCollection());
        $selectedIds = collect($this->selected)->filter()->keys()->all();
        $selectedTotals = $this->calculateTotals($paginator->getCollection()->whereIn('id', $selectedIds));

        return view('livewire.finance.samples-invoicing-new', [
            'samples' => $paginator,
            'pageTotals' => $pageTotals,
            'selectedTotals' => $selectedTotals,
        ]);
    }

    /**
     * Prepare and dispatch detailed selected sample data to the invoice generator component.
     */
    public function captureSelectedSamplesData(): void
    {
        if (empty($this->selected)) {
            Log::info('SamplesInvoice: No rows currently selected; dispatching empty selection');
            $this->dispatch('load-selected-samples', selected: []);
            return;
        }

        // Only consider selections from the current page to avoid lingering ids
        $currentIds = collect($this->currentPageSampleIds);
        $selectedIds = collect($this->selected)
            ->filter()
            ->keys()
            ->filter(fn($id) => $currentIds->contains($id))
            ->values()
            ->toArray();
            
        // If no valid selections found, clear the selection to prevent confusion
        if (empty($selectedIds) && !empty($this->selected)) {
            $this->selected = [];
            $this->selectAllCheckbox = false;
        }
        Log::info('SamplesInvoice: Building selected sample payload', [
            'selected_count' => count($selectedIds),
            'selected_ids' => $selectedIds,
            'filters' => [
                'season' => $this->season,
                'from' => $this->from,
                'to' => $this->to,
                'customer' => $this->customer,
                'factory' => $this->factory,
                'department' => $this->department,
                'category' => $this->category,
                'coo' => $this->coo,
                'fn_complete' => $this->fn_complete,
            ],
        ]);
        if (empty($selectedIds)) {
            $this->dispatch('load-selected-samples', selected: []);
            return;
        }

        // Load only currently filtered subset to avoid stale selections from previous filters
        $samples = $this->getSamplesForIds($selectedIds);
        $samples = $this->addPricing($samples);

        $selectedSamplesData = $samples->map(function ($sample) {
            $settings = $this->resolveSampleSettings($sample->cust_settings, (int)$sample->sample_type_id);
            return [
                'id' => $sample->id,
                'customer_ref' => $sample->customer_ref,
                'description' => $sample->description,
                'colourway' => $sample->colourway,
                'size' => $sample->size,
                'qty' => $sample->qty,
                'sample_type' => $sample->sample_type,
                'date_sent' => $sample->date_sent,
                'price' => (float)($sample->quote ?? 0),
                'discount' => $settings->discount ?? 0,
                'surcharge' => $settings->surcharge ?? 0,
                'development' => $settings->development ?? 0,
                'freight' => $settings->freight ?? false,
                'quote_total' => (float) (($sample->quote ?? 0) * ($sample->qty ?? 0)),
                'customer' => $sample->customer,
                'customer_id' => $sample->customers_id,
                'season' => $sample->season,
                'do_not_charge' => $sample->do_not_charge,
                'fn_complete' => $sample->fn_complete,
                'rt_invoice' => $sample->rt_invoice,
                'currency' => $sample->cust_currency ?? '£',
                'category' => $sample->category ?? '',
            ];
        })->values()->toArray();

        Log::info('SamplesInvoice: Dispatching selected sample payload', [
            'payload_count' => count($selectedSamplesData),
        ]);
        $this->dispatch('load-selected-samples', selected: $selectedSamplesData);
    }

    /**
     * Fetch samples by IDs with required joins/fields for invoicing payload.
     */
    protected function getSamplesForIds(array $ids)
    {
        $query = Samples::select([
            'samples.id as id',
            'seasons.description as season',
            'customers.name as customer',
            'designs.id as rt_no',
            'styles.customer_ref as customer_ref',
            'designs.description as description',
            'sizes.name as size',
            'colourways.name as colourway',
            'samples.po as po',
            'sample_types.name as sample_type',
            'sample_types.id as sample_type_id',
            'suppliers.name as factory',
            'styles.carryover as carryover',
            'samples.qty as qty',
            'samples.date_expected as date_expected',
            'samples.date_received as date_received',
            'samples.date_sent as date_sent',
            'samples.fn_notes as fn_notes',
            'samples.rt_invoice as rt_invoice',
            'samples.fty_invoice as fty_invoice',
            'samples.finance_comments as finance_comments',
            'samples.do_not_charge as do_not_charge',
            'customers.settings as cust_settings',
            'colourways.id as colourways_id',
            'colourways.colour_type as colour_type',
            'customers.currency as cust_currency',
            'suppliers.currency as fact_currency',
            'styles.id as styles_id',
            'samples.fn_complete as fn_complete',
            'colourways.cancelled as cw_cancelled',
            'styles.cancelled as s_cancelled',
            'style_versions.name as style_version',
            'seasons.id as seasons_id',
            'suppliers.id as factory_id',
            'suppliers.countries_id as coo',
            'styles.departments_id as departments_id',
            'styles.category as category',
            'customers.id as customers_id',
            'samples.sizes_id as sizes_id',
        ])
            ->leftJoin('colourways', 'colourways.id', '=', 'samples.colourways_id')
            ->leftJoin('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->leftJoin('suppliers', 'suppliers.id', '=', 'style_versions.factory_id')
            ->leftJoin('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->leftJoin('customers', 'customers.id', '=', 'styles.customers_id')
            ->leftJoin('seasons', 'seasons.id', '=', 'styles.seasons_id')
            ->leftJoin('designs', 'designs.id', '=', 'styles.designs_id')
            ->leftJoin('sizes', 'sizes.id', '=', 'samples.sizes_id')
            ->leftJoin('sample_types', 'sample_types.id', '=', 'samples.sample_types_id')
            ->whereIn('samples.id', $ids)
            ->whereNull('samples.deleted_at')
            ->with(['colourways', 'colourways.style_versions.styles.customers']);

        return $query->get();
    }

    /**
     * Lightweight pricing adder for an iterable of Sample models
     */
    protected function addPricing($samples)
    {
        foreach ($samples as $s => $sample) {
            if ($sample->colourways) {
                $priceResult = $sample->colourways->price($sample->sizes_id);
                if ($priceResult && $priceResult['price'] && isset($priceResult['price']->quote)) {
                    $samples[$s]->quote = $priceResult['price']->quote;
                    $samples[$s]->quote_status = $priceResult['price']->quote_status;
                }
            }
        }
        return $samples;
    }

    /**
     * Augment each sample with:
     * - quote (per item) and quote_status via Colourways->price($sizeId)
     * - sample_settings resolved from customer settings by sample type
     * - quote_base (per item, GBP) and quote_base_total (qty * quote_base)
     */
    protected function augmentSamples($samples)
    {
        foreach ($samples as $idx => $sample) {
            // Quote per item from Colourways price()
            if ($sample->colourways) {
                $priceResult = $sample->colourways->price($sample->sizes_id);
                if ($priceResult && $priceResult['price'] && isset($priceResult['price']->quote)) {
                    $samples[$idx]->quote = $priceResult['price']->quote; // per item
                    $samples[$idx]->quote_status = $priceResult['price']->quote_status;
                }
            }

            // Resolve sample settings (discount, surcharge, development, freight)
            $samples[$idx]->sample_settings = $this->resolveSampleSettings(
                $sample->cust_settings,
                (int)($sample->sample_type_id)
            );

            // Compute base quote (per item, GBP)
            $seasonModel = $this->seasons->where('id', $sample->seasons_id)->first();
            $quotePerItem = (float)($samples[$idx]->quote ?? 0);
            $quoteBase = Conversions::convertCurrency($sample->cust_currency ?? '£', '£', $quotePerItem, $seasonModel);
            $samples[$idx]->quote_base = (float)$quoteBase;
            $samples[$idx]->quote_base_total = (float)($samples[$idx]->qty * $samples[$idx]->quote_base);
        }
        return $samples;
    }

    protected function resolveSampleSettings($customerSettings, int $sampleTypeId): object
    {
        $settings = is_array($customerSettings) ? $customerSettings : (json_decode($customerSettings ?? '[]', true) ?: []);
        $perType = $settings['samples-charge'][$sampleTypeId] ?? [];
        $fallback = [
            'discount' => $settings['samples-charge-discount'] ?? 0,
            'surcharge' => $settings['samples-charge-surcharge'] ?? 0,
            'development' => $settings['samples-charge-development'] ?? 0,
            'freight' => $settings['samples-charge-freight'] ?? false,
        ];
        $resolved = array_merge($fallback, $perType);
        // Normalize types
        $resolved['discount'] = (float)($resolved['discount'] ?? 0);
        $resolved['surcharge'] = (float)($resolved['surcharge'] ?? 0);
        $resolved['development'] = (float)($resolved['development'] ?? 0);
        $resolved['freight'] = (bool)($resolved['freight'] ?? false);
        return (object)$resolved;
    }

    protected function calculateTotals(Collection $items): array
    {
        $qty = (int)$items->sum('qty');
        $quoteBaseTotal = (float)$items->sum('quote_base_total');
        return [
            'qty' => $qty,
            'quote_base_total' => round($quoteBaseTotal, 2),
        ];
    }

    public function updateInline(int $id, string $field, $value): void
    {
        Gate::authorize('finance:update');
        $allowed = ['rt_invoice', 'fty_invoice', 'finance_comments', 'fn_complete'];
        if (!in_array($field, $allowed)) return;

        $payload = [$field => $field === 'fn_complete' ? (empty($value) ? 0 : (bool)$value) : (empty($value) ? NULL : $value)];
        Samples::where('id', $id)->update($payload);
        // No emit; Alpine keeps local state to avoid flicker
    }

    public function markDoNotChargeComplete()
    {
        try {
            Gate::authorize('finance:update');
            
            Log::info('Mark DNC Complete - Starting', [
                'filters' => [
                    'season' => $this->season,
                    'customer' => $this->customer,
                    'factory' => $this->factory,
                    'department' => $this->department,
                    'category' => $this->category,
                    'from' => $this->from,
                    'to' => $this->to,
                    'search' => $this->search,
                    'fn_complete' => $this->fn_complete,
                ]
            ]);
            
            // Use the same query structure as the main render method
            $query = Samples::select(['samples.id'])
                ->leftJoin('colourways', 'colourways.id', '=', 'samples.colourways_id')
                ->leftJoin('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
                ->leftJoin('styles', 'styles.id', '=', 'style_versions.styles_id')
                ->leftJoin('customers', 'customers.id', '=', 'styles.customers_id')
                ->leftJoin('seasons', 'seasons.id', '=', 'styles.seasons_id')
                ->leftJoin('designs', 'designs.id', '=', 'styles.designs_id')
                ->leftJoin('sizes', 'sizes.id', '=', 'samples.sizes_id')
                ->leftJoin('sample_types', 'sample_types.id', '=', 'samples.sample_types_id')
                ->leftJoin('suppliers', 'suppliers.id', '=', 'style_versions.factory_id')
                ->where('samples.do_not_charge', true)
                ->whereNotNull('samples.date_sent')
                ->whereNull('samples.deleted_at');
            
            // Apply filters
            if ($this->season) {
                $query->where('seasons.id', $this->season);
            }
            if ($this->customer) {
                $query->where('customers.id', $this->customer);
            }
            if ($this->factory) {
                $query->where('suppliers.id', $this->factory);
            }
            if ($this->coo) {
                $query->where('suppliers.countries_id', $this->coo);
            }
            if ($this->department) {
                $query->where('styles.departments_id', $this->department);
            }
            if ($this->category) {
                $query->where('styles.category', $this->category);
            }
            // Only look for incomplete DO NOT CHARGE samples (we want to mark them as complete)
            $query->where('samples.fn_complete', '!=', true);
            if ($this->from && $this->to) {
                $query->whereBetween('samples.date_sent', [$this->from, $this->to]);
            }
            if ($this->search) {
                $query->where(function ($q) {
                    $q->where('samples.id', 'like', '%' . $this->search . '%')
                      ->orWhere('styles.customer_ref', 'like', '%' . $this->search . '%')
                      ->orWhere('designs.description', 'like', '%' . $this->search . '%')
                      ->orWhere('customers.name', 'like', '%' . $this->search . '%');
                });
            }
            
            $doNotChargeSamples = $query->get();

            $count = $doNotChargeSamples->count();
            
            Log::info('Mark DNC Complete - Query Results', [
                'count' => $count,
                'sample_ids' => $doNotChargeSamples->pluck('id')->toArray()
            ]);
            
            if ($count > 0) {
                // Update all DO NOT CHARGE samples to complete
                $updatedCount = Samples::whereIn('id', $doNotChargeSamples->pluck('id'))
                    ->update(['fn_complete' => true]);

                session()->flash('message', "Successfully marked {$updatedCount} DO NOT CHARGE sample(s) as complete.");
                session()->flash('alert-class', 'alert-success');
            } else {
                session()->flash('message', 'No DO NOT CHARGE samples found with current filters.');
                session()->flash('alert-class', 'alert-info');
            }
        } catch (\Exception $e) {
            session()->flash('message', 'Error marking DO NOT CHARGE samples as complete: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        }
    }
}


