<?php

namespace App\Http\Livewire\Finance;

use Exception;
use App\Models\Samples;
use Livewire\Component;
use Livewire\Attributes\On;
use Livewire\Attributes\Modelable;
use Illuminate\Support\Facades\Log;
use Livewire\Mechanisms\HandleComponents\HandleComponents;

class SamplesInvoicingBulk extends Component
{
    #[Modelable]
    public $selected = [];
    public $bulk = [];
    public $nullify = [
        'rt_invoice' => false,
        'fty_invoice' => false,
        'comments' => false,
    ];
    public $completionStatus = null; // 'complete', 'not_complete', or null

    #[On('refresh')]
    public function render()
    {
        return view('livewire.finance.samples-invoicing-bulk');
    }

    #[On('load-selected')]
    public function loadSelected($selected){
        $this->selected = $selected;
    }

    private function getParentComponentInstance()
    {
        return app(HandleComponents::class)::$componentStack[0];
    }

    public function bulkSave()
    {
        try {
            // Validate input
            $validated = $this->validate([
                'bulk.rt_invoice' => 'nullable|string|max:35',
                'bulk.fty_invoice' => 'nullable|string|max:35',
                'bulk.comments' => 'nullable|string|max:200',
                'nullify.rt_invoice' => 'boolean',
                'nullify.fty_invoice' => 'boolean',
                'nullify.comments' => 'boolean',
                'completionStatus' => 'nullable|string|in:complete,not_complete',
                'selected.*' => 'required|boolean',
            ]);

            // Build updates array
            $updates = [];

            // Handle regular bulk updates
            foreach ($this->bulk as $field => $value) {
                if (!is_null($value) && $value !== "") {
                    $updates[$field] = is_string($value) ? strip_tags(trim($value)) : $value;
                }
            }

            // Handle nullify options
            foreach ($this->nullify as $field => $shouldNullify) {
                if ($shouldNullify) {
                    $updates[$field] = null;
                }
            }

            // Handle completion status
            if ($this->completionStatus === 'complete') {
                $updates['fn_complete'] = true;
            } elseif ($this->completionStatus === 'not_complete') {
                $updates['fn_complete'] = false;
            }

            if (empty($updates)) {
                session()->flash('message', 'No valid data provided for update.');
                session()->flash('alert-class', 'alert-warning');
                return;
            }

            // Filter `selected` to remove false values and get the keys (IDs)
            $selectedIds = collect($this->selected)
                ->filter() // Removes false values
                ->keys()   // Gets the list of valid IDs
                ->toArray();

            if (empty($selectedIds)) {
                session()->flash('message', 'No records selected for updating.');
                session()->flash('alert-class', 'alert-warning');
                return;
            }

            // Fetch all Sample models where ID is in the list
            $samples = Samples::whereIn('id', $selectedIds)->get();

            if ($samples->isEmpty()) {
                session()->flash('message', 'No matching records found in the database.');
                session()->flash('alert-class', 'alert-warning');
                return;
            }

            // Iterate over each sample and update
            foreach ($samples as $sample) {
                $sample->update($updates);
            }

            // Success message
            session()->flash('message', count($samples) . ' records successfully updated.');
            session()->flash('alert-class', 'alert-success');

            // Close modal & reset fields
            $this->reset('selected', 'bulk', 'nullify', 'completionStatus');
            $this->dispatch('close-modal');

        } catch (Exception $e) {
            // Log error
            Log::error('Bulk update failed: ' . $e->getMessage(), ['exception' => $e]);

            // Show user-friendly error message
            session()->flash('message', 'An error occurred while updating records. Please try again.');
            session()->flash('alert-class', 'alert-danger');
        }
    }

}
