<?php

namespace App\Http\Livewire\Finance;

use App\Models\Sizes;
use App\Models\Samples;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Helper\Functions;
use App\Models\Suppliers;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;
use App\Models\ViewFinanceSamples;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;
use App\Livewire\Forms\FinanceSamplesForm;

class SamplesInvoicing extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideFilters = false;

    public $selected = [];
    public $selectAllCheckbox = false;
    public $currentPageSampleIds = [];

    public function updatedSelected(){
        $this->dispatch('load-selected', selected: $this->selected);
        $this->captureSelectedSamplesData();
    }

    public function captureSelectedSamplesData()
    {
        if (empty($this->selected)) {
            $this->dispatch('load-selected-samples', selected: []);
            return;
        }

        // Get the selected sample IDs
        $selectedIds = collect($this->selected)
            ->filter()
            ->keys()
            ->toArray();

        if (empty($selectedIds)) {
            $this->dispatch('load-selected-samples', selected: []);
            return;
        }

        // Get the current samples data from the render method
        $samples = $this->getSamplesData();
        
        // Filter to only selected samples and add pricing
        $selectedSamples = $samples->whereIn('id', $selectedIds);
        $selectedSamples = $this->addPricing($selectedSamples);

        // dd($selectedSamples);

        // Convert to array format for the invoice generator
        $selectedSamplesData = $selectedSamples->map(function ($sample) {
            return [
                'id' => $sample->id,
                'customer_ref' => $sample->customer_ref,
                'description' => $sample->description,
                'colourway' => $sample->colourway,
                'size' => $sample->size,
                'qty' => $sample->qty,
                'sample_type' => $sample->sample_type,
                'date_sent' => $sample->date_sent,
                'price' => (float) str_replace(',', '', $sample->quote ?? 0),
                'discount' => $sample->sample_settings->discount ?? 0,
                'surcharge' => $sample->sample_settings->surcharge ?? 0,
                'development' => $sample->sample_settings->development ?? 0,
                'freight' => $sample->sample_settings->freight ?? false,
                'quote_total' => (float) str_replace(',', '', $sample->quote_total ?? 0),
                'customer' => $sample->customer,
                'customer_id' => $sample->customers_id,
                'season' => $sample->season,
                'do_not_charge' => $sample->do_not_charge,
                'fn_complete' => $sample->fn_complete,
                'rt_invoice' => $sample->rt_invoice,
                'currency' => $sample->cust_currency ?? '£',
            ];
        })->values()->toArray();

        // Dispatch the actual sample data to the invoice generator
        $this->dispatch('load-selected-samples', selected: $selectedSamplesData);
    }

    public function getSamplesData()
    {
        // Get the current samples data using the same query as the main table
        $query = Samples::select([
            'samples.id as id',
            'seasons.description as season',
            'customers.name as customer',
            'designs.id as rt_no',
            'styles.customer_ref as customer_ref',
            'designs.description as description',
            'sizes.name as size',
            'colourways.name as colourway',
            'samples.po as po',
            'sample_types.name as sample_type',
            'sample_types.id as sample_type_id',
            'suppliers.name as factory',
            'styles.carryover as carryover',
            'samples.qty as qty',
            'samples.date_expected as date_expected',
            'samples.date_received as date_received',
            'samples.date_sent as date_sent',
            'samples.fn_notes as fn_notes',
            'samples.rt_invoice as rt_invoice',
            'samples.fty_invoice as fty_invoice',
            'samples.finance_comments as finance_comments',
            'samples.do_not_charge as do_not_charge',
            'customers.settings as cust_settings',
            'colourways.id as colourways_id',
            'colourways.colour_type as colour_type',
            'customers.currency as cust_currency',
            'suppliers.currency as fact_currency',
            'styles.id as styles_id',
            'samples.fn_complete as fn_complete',
            'colourways.cancelled as cw_cancelled',
            'styles.cancelled as s_cancelled',
            'style_versions.name as style_version',
            DB::raw("(SELECT
                CONCAT(
                    '[',
                    GROUP_CONCAT(DISTINCT JSON_OBJECT('id', co.id, 'po', co.customer_po) SEPARATOR ','),
                    ']'
                ) AS orders
            FROM customer_order_lines col
            JOIN customer_orders co ON co.id = col.customer_orders_id
            JOIN colourways c ON c.id = col.colourways_id
            JOIN style_versions sv ON sv.id = c.style_versions_id
            JOIN styles s ON s.id = sv.styles_id
            WHERE s.id = styles.id) as orders"),
                        'seasons.id as seasons_id',
                        'suppliers.id as factory_id',
                        'suppliers.countries_id as coo',
                        'styles.departments_id as departments_id',
                        'styles.category as category',
                        'customers.id as customers_id',
        ])
        ->leftJoin('colourways', 'colourways.id', '=', 'samples.colourways_id')
        ->leftJoin('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->leftJoin('suppliers', 'suppliers.id', '=', 'style_versions.factory_id')
        ->leftJoin('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->leftJoin('customers', 'customers.id', '=', 'styles.customers_id')
        ->leftJoin('seasons', 'seasons.id', '=', 'styles.seasons_id')
        ->leftJoin('designs', 'designs.id', '=', 'styles.designs_id')
        ->leftJoin('sizes', 'sizes.id', '=', 'samples.sizes_id')
        ->leftJoin('sample_types', 'sample_types.id', '=', 'samples.sample_types_id')
        ->whereNull('samples.deleted_at');

        // Load the colourways relationship with price model for pricing
        $query->with(['colourways', 'colourways.style_versions.styles.customers']);

        $query = $query->search($this->search);
        $query->whereNotNull('date_sent');
        if(!empty($this->season))
            $query->where('styles.seasons_id', '=', $this->season);
        if(!empty($this->factory))
            $query->where('style_versions.factory_id', '=', $this->factory);
        if(!empty($this->department))
            $query->where('styles.departments_id', '=', $this->department);
        if(!empty($this->coo))
            $query->where('suppliers.countries_id', '=', $this->coo);
        if(!empty($this->category))
            $query->where('styles.category', '=', $this->category);
        if(!empty($this->customer))
            $query->where('styles.customers_id', '=', $this->customer);
        if($this->fn_complete == FALSE)
            $query->where('fn_complete', '!=', true);
        if(!empty($this->from) && !empty($this->to))
            $query->whereBetween('date_sent', [$this->from, $this->to]);

        return $query->get();
    }

    public function selectAll()
    {
        $this->selected = [];
        
        // Use the already loaded sample IDs
        if ($this->currentPageSampleIds) {
            foreach($this->currentPageSampleIds as $id) {
                $this->selected[$id] = true;
            }
        }
        $this->selectAllCheckbox = true;
        $this->updatedSelected();
    }

    public function deselectAll()
    {
        $this->selected = [];
        $this->selectAllCheckbox = false;
        $this->updatedSelected();
    }

    public function toggleSelectAll()
    {
        if ($this->selectAllCheckbox) {
            $this->selectAll();
        } else {
            $this->deselectAll();
        }
    }

    #[On('close-modal')]
    public function reload(){
        $this->reset('selected');
        $this->render();
    }

    public  $season,
            $from,
            $to,
            $customer,
            $factory,
            $coo,
            $department,
            $category,
            $fn_complete = 0,
            $search,
            $paginate;

    protected function filterKey(): string
    {
        return 'financesamplesfilters';
    }

    protected function columnsVersion(): float
    {
        return 1;
    }

    protected function filters(): array
    {
        return ['season', 'from', 'to', 'customer', 'factory', 'coo', 'department', 'category', 'fn_complete', 'search', 'paginate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }


    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['season', 'from', 'to', 'customer', 'factory', 'coo', 'department', 'category', 'fn_complete', 'search', 'paginate'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }
    public function mount()
    {
        $this->loadFilterSettings();
    }


    public FinanceSamplesForm $form;

    #[Computed(cache: true, key: 'factories-with-countries')]
    public function factories(){
        return Suppliers::with('countries')->where('type', 'factory')->get();
    }
    #[Computed]
    public function sizes(){
        return Sizes::get();
    }

    #[Computed]
    public function seasons(){
        return Seasons::allCached();
    }

    #[Computed]
    public function customers(){
        return Customer::allCached();
    }

    #[Computed]
    public function departments(){
        return Departments::allCached();
    }

    public function render()
    {
        Gate::authorize('finance:read');

        $query = Samples::select([
            'samples.id as id',
            'seasons.description as season',
            'customers.name as customer',
            'designs.id as rt_no',
            'styles.customer_ref as customer_ref',
            'designs.description as description',
            'sizes.name as size',
            'colourways.name as colourway',
            'samples.po as po',
            'sample_types.name as sample_type',
            'sample_types.id as sample_type_id',
            'suppliers.name as factory',
            'styles.carryover as carryover',
            'samples.qty as qty',
            'samples.date_expected as date_expected',
            'samples.date_received as date_received',
            'samples.date_sent as date_sent',
            'samples.fn_notes as fn_notes',
            'samples.rt_invoice as rt_invoice',
            'samples.fty_invoice as fty_invoice',
            'samples.finance_comments as finance_comments',
            'samples.do_not_charge as do_not_charge',
            'customers.settings as cust_settings',
            'colourways.id as colourways_id',
            'colourways.colour_type as colour_type',
            'customers.currency as cust_currency',
            'suppliers.currency as fact_currency',
            'styles.id as styles_id',
            'samples.fn_complete as fn_complete',
            'colourways.cancelled as cw_cancelled',
            'styles.cancelled as s_cancelled',
            'style_versions.name as style_version',
            DB::raw("(SELECT
                CONCAT(
                    '[',
                    GROUP_CONCAT(DISTINCT JSON_OBJECT('id', co.id, 'po', co.customer_po) SEPARATOR ','),
                    ']'
                ) AS orders
            FROM customer_order_lines col
            JOIN customer_orders co ON co.id = col.customer_orders_id
            JOIN colourways c ON c.id = col.colourways_id
            JOIN style_versions sv ON sv.id = c.style_versions_id
            JOIN styles s ON s.id = sv.styles_id
            WHERE s.id = styles.id) as orders"),
                        'seasons.id as seasons_id',
                        'suppliers.id as factory_id',
                        'suppliers.countries_id as coo',
                        'styles.departments_id as departments_id',
                        'styles.category as category',
                        'customers.id as customers_id',
        ])
        ->leftJoin('colourways', 'colourways.id', '=', 'samples.colourways_id')
        ->leftJoin('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->leftJoin('suppliers', 'suppliers.id', '=', 'style_versions.factory_id')
        ->leftJoin('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->leftJoin('customers', 'customers.id', '=', 'styles.customers_id')
        ->leftJoin('seasons', 'seasons.id', '=', 'styles.seasons_id')
        ->leftJoin('designs', 'designs.id', '=', 'styles.designs_id')
        ->leftJoin('sizes', 'sizes.id', '=', 'samples.sizes_id')
        ->leftJoin('sample_types', 'sample_types.id', '=', 'samples.sample_types_id')
        ->whereNull('samples.deleted_at');

        // Load the colourways relationship with price model for pricing
        $query->with(['colourways', 'colourways.style_versions.styles.customers']);

        $query = $query->search($this->search);
        $query->whereNotNull('date_sent');
        if(!empty($this->season))
            $query->where('styles.seasons_id', '=', $this->season);
        if(!empty($this->factory))
            $query->where('style_versions.factory_id', '=', $this->factory);
        if(!empty($this->department))
            $query->where('styles.departments_id', '=', $this->department);
        if(!empty($this->coo))
            $query->where('suppliers.countries_id', '=', $this->coo);
        if(!empty($this->category))
            $query->where('styles.category', '=', $this->category);
        if(!empty($this->customer))
            $query->where('styles.customers_id', '=', $this->customer);
        if($this->fn_complete == FALSE)
            $query->where('fn_complete', '!=', true);
        if(!empty($this->from) && !empty($this->to))
            $query->whereBetween('date_sent', [$this->from, $this->to]);


        $samples = $query->paginate($this->paginate);
        $samples = $this->addPricing($samples);

        $this->currentPageSampleIds = $samples->pluck('id')->toArray(); // Store only the IDs
        $this->form->set($samples);


        return view('livewire.finance.samples-invoicing', [
            'samples' => $samples,
        ]);
    }

    public function updatedForm($val, $var){
        Gate::authorize('finance:update');
        $this->form->update($val, $var);
    }

    public function addPricing($samples){
		foreach($samples as $s=>$sample){
			// Get quote price from colourways price method with size
			if ($sample->colourways) {
				$priceResult = $sample->colourways->price($sample->sizes_id);
				if ($priceResult && $priceResult['price'] && isset($priceResult['price']->quote)) {
					$samples[$s]->quote = $priceResult['price']->quote;
					$samples[$s]->quote_status = $priceResult['price']->quote_status;
				}
			}
		}
        return $samples;
	}


}

