<?php

namespace App\Http\Livewire\Finance;

use App\Models\Samples;
use App\Models\Customer;
use Livewire\Component;
use App\Services\ZohoService;
use Livewire\Attributes\On;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\HandlesZohoOAuth;

class SamplesInvoiceGenerator extends Component
{
    use HandlesZohoOAuth;

    public $selectedSamples = [];
    public $customer;
    public $date;
    public $invoiceData = [];
    public $showInvoiceModal = false;
    public $zohoConnected = false;
    public $createdInvoices = [];
    public $totalInvoices = 0;
    public $completionMessage = '';
    public $showCompletionModal = false;

    protected ?int $lastZohoCustomerFetchCount = null;
    protected array $lastZohoSuggestions = [];

    // Zoho configuration
    public $org = '20079258157';
    // UI selection for account; prefer code 40500 by default
    public $accountCode = '40500';
    public $accountPreference = 'Sales';
    public array $accountOptions = [];
    public $reportingTagDepartment = 'Knitted Wholesale Tom';
    public $reportingTagCustomer;
    public $reportingTagCommissionsCustomer;
    public $vat;
    public $departmentOptions = [];

    private $zoho;

    public function mount()
    {
        $this->date = now()->format('Y-m-d');
        $this->zoho = app(ZohoService::class);
        $this->departmentOptions = $this->getDepartmentOptions();
    }

    public function hydrate()
    {
        $this->zoho = app(ZohoService::class);

        if ($this->zoho->isConnected()) {
            $this->zohoConnected = true;
            // Preload account options for dropdown (code and name)
            try {
                $accs = $this->zoho->getChartOfAccounts($this->org);
                $this->accountOptions = $accs->map(function($a){
                    return [
                        'id' => $a['account_id'] ?? '',
                        'code' => (string)($a['account_code'] ?? ''),
                        'name' => (string)($a['account_name'] ?? ''),
                        'label' => trim(((string)($a['account_code'] ?? '')).' - '.((string)($a['account_name'] ?? ''))),
                    ];
                })->filter(fn($a)=>$a['code']!=='' || $a['name']!=='')->sortBy('label')->values()->toArray();
            } catch (\Throwable $e) {
                // swallow; dropdown will be empty
            }
            
            // Reload department options when Zoho is connected
            try {
                $this->departmentOptions = $this->getDepartmentOptions();
            } catch (\Exception $e) {
                Log::error('Failed to load department options', ['error' => $e->getMessage()]);
            }
        } else {
            $this->authUrl = $this->zoho->getAuthUrl();
            $this->zohoConnected = false;
        }
    }

    #[On('load-selected-samples')]
    public function loadSelectedSamples($selected)
    {
        // Initialize selectedSamples with proper structure for quantity editing
        $this->selectedSamples = [];
        foreach ($selected as $sample) {
            $this->selectedSamples[$sample['id']] = $sample;
        }
        $this->prepareInvoiceData();
    }

    public function prepareInvoiceData()
    {
        if (empty($this->selectedSamples)) {
            $this->invoiceData = [];
            return;
        }

        // Filter out samples that shouldn't be invoiced
        $filteredSamples = collect($this->selectedSamples)->filter(function ($sample) {
            // Skip if already invoiced or complete
            if (!empty($sample['rt_invoice']) || $sample['fn_complete']) {
                return false;
            }

            // Skip if do not charge
            if ($sample['do_not_charge']) {
                return false;
            }

            return true;
        });

        // Group by customer and shipment date for invoice generation
        $this->invoiceData = $filteredSamples->groupBy(function ($sample) {
            return $sample['customer_id'] . '|' . $sample['date_sent'];
        })->map(function ($samples, $key) {
            $parts = explode('|', $key);
            $customerId = $parts[0];
            $shipmentDate = $parts[1];
            
            // Get customer info for reporting tags
            $customer = Customer::find($customerId);
            
            return [
                'customer_id' => $customerId,
                'customer_name' => $customer->name ?? 'Unknown',
                'shipment_date' => $shipmentDate,
                'samples' => $samples->values()->toArray(),
                'reporting_tag_customer' => $customer->zoho_customer_tag ?? $customer->zoho_customer ?? $customer->name,
                'reporting_tag_factored' => $customer->zoho_factored ?? 'No',
            ];
        })->values()->toArray();

        // Set default customer if only one customer
        if (count($this->invoiceData) === 1) {
            $this->customer = $this->invoiceData[0]['customer_id'];
        }
    }


    public function generateInvoices()
    {
        Gate::authorize('finance:create');

        if (!$this->zohoConnected) {
            session()->flash('message', 'Zoho is not connected. Please connect and try again.');
            session()->flash('alert-class', 'alert-danger');
            return;
        }

        if (empty($this->customer)) {
            session()->flash('message', 'Please select a customer before generating invoices.');
            session()->flash('alert-class', 'alert-warning');
            return;
        }

        if (empty($this->invoiceData)) {
            session()->flash('message', 'No samples selected for invoicing.');
            session()->flash('alert-class', 'alert-warning');
            return;
        }

        $this->createdInvoices = [];
        $errors = [];

        try {
            foreach ($this->invoiceData as $group) {
                $invoiceResult = $this->createCustomerInvoice($group);
                
                if ($invoiceResult['success']) {
                    $this->createdInvoices[] = $invoiceResult['invoice_number'];
                    $this->updateSampleInvoices($group['samples'], $invoiceResult['invoice_number']);
                } else {
                    \Log::warning('SamplesInvoiceGenerator: Invoice creation failed', [
                        'customer_id' => $group['customer_id'],
                        'error' => $invoiceResult['error'] ?? 'unknown',
                    ]);
                    $errors[] = $invoiceResult['error'];
                }
            }

            if (!empty($errors)) {
                session()->flash('message', 'Some invoices failed to create: ' . implode(', ', $errors));
                session()->flash('alert-class', 'alert-warning');
            } else {
                $this->completionMessage = 'Successfully created ' . count($this->createdInvoices) . ' invoice(s): ' . implode(', ', $this->createdInvoices);
                $this->showCompletionModal = true;
            }

        } catch (\Exception $e) {
            Log::error('Invoice generation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            session()->flash('message', 'Invoice generation failed: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        }
    }

    private function createCustomerInvoice($group)
    {
        try {
            $customerId = $group['customer_id'];
            $samples = $group['samples'];
            $shipmentDate = $group['shipment_date'];
            $reportingTagCustomer = $group['reporting_tag_customer'];
            $reportingTagFactored = $group['reporting_tag_factored'];
            // Build reporting tags once per group for reuse on each line item
            $reportingTags = $this->buildReportingTags($reportingTagCustomer, $reportingTagFactored);
            
            // Get customer information
            $customer = Customer::find($customerId);
            if (!$customer) {
                return ['success' => false, 'error' => 'Customer not found'];
            }

            // Resolve account_id for this org
            $accountId = $this->resolveAccountId($this->accountCode ?: $this->accountPreference);
            if (!$accountId) {
                return [
                    'success' => false,
                    'error' => 'Zoho account not found for: ' . (($this->accountCode ?: $this->accountPreference) ?: 'default'),
                ];
            }

            // Prepare line items
            $totalAmount = 0;

            foreach ($samples as $sample) {
                if ($sample['do_not_charge']) {
                    continue;
                }

                $basePrice = $sample['price'];
                $qty = $this->selectedSamples[$sample['id']]['qty'] ?? $sample['qty'];
                $discount = $sample['discount'] ?? 0;
                $surcharge = $sample['surcharge'] ?? 0;
                $development = $sample['development'] ?? 0;
                $category = $sample['category'] ?? '';
                
                // Calculate VAT based on category only
                // All samples are sent to UK, so VAT applies regardless of customer currency
                // Only children's samples are exempt from VAT
                $isChildrens = strtolower(trim($category)) === 'childrens';
                $tax = !$isChildrens;
                
                // Get tax ID - use VAT tax for taxable items, zero tax for children's items
                $taxId = $tax ? $this->getVatTaxId() : $this->getZeroTaxId();
                
                // Calculate discount and surcharge amounts
                $discountAmount = $discount > 0 ? ($basePrice * $qty) * ($discount / 100) : 0;
                $surchargeAmount = $surcharge > 0 ? ($basePrice * $qty) * ($surcharge / 100) : 0;
                
                // Calculate final line total
                $lineTotal = ($basePrice * $qty) - $discountAmount + $surchargeAmount + $development;

                // Base sample line with discount applied
                $sampleLineItem = [
                    'name' => 'Sample',
                    'description' => $sample['customer_ref'] . ' - ' . $sample['description'] . ' - ' . $sample['colourway'] . ' - ' . $sample['size'],
                    'rate' => number_format($basePrice, 2, '.', ''),
                    'quantity' => $qty,
                    'discount_amount' => $discountAmount > 0 ? number_format($discountAmount, 2, '.', '') : null,
                    'account_id' => $accountId,
                    'header_name' => 'Samples shipped on ' . \Carbon\Carbon::parse($shipmentDate)->format('d/m/Y'),
                    'tags' => $reportingTags,
                ];
                
                // Always add tax_id (either VAT tax or zero tax)
                if ($taxId) {
                    $sampleLineItem['tax_id'] = $taxId;
                }
                
                $lineItems[] = $sampleLineItem;
                $totalAmount += $lineTotal;

                // Add surcharge as separate line if present (since it's an addition, not a discount)
                if ($surcharge > 0) {
                    // Calculate surcharge per item (not total)
                    $surchargePerItem = $basePrice * ($surcharge / 100);
                    
                    $surchargeLineItem = [
                        'name' => 'Sample Surcharge',
                        'description' => $sample['customer_ref'] . ' - ' . $surcharge . '% surcharge',
                        'rate' => number_format($surchargePerItem, 2, '.', ''),
                        'quantity' => $qty,
                        'account_id' => $accountId,
                        'header_name' => 'Samples shipped on ' . \Carbon\Carbon::parse($shipmentDate)->format('d/m/Y'),
                        'tags' => $reportingTags,
                    ];
                    
                    // Always add tax_id (either VAT tax or zero tax)
                    if ($taxId) {
                        $surchargeLineItem['tax_id'] = $taxId;
                    }
                    
                    $lineItems[] = $surchargeLineItem;
                }

                // Add development cost as separate line if present
                // Development fee is a one-time charge per style version, not per sample quantity
                if ($development > 0) {
                    $developmentLineItem = [
                        'name' => 'Development Cost',
                        'description' => $sample['customer_ref'] . ' - Development',
                        'rate' => number_format($development, 2, '.', ''),
                        'quantity' => 1,
                        'account_id' => $accountId,
                        'header_name' => 'Samples shipped on ' . \Carbon\Carbon::parse($shipmentDate)->format('d/m/Y'),
                        'tags' => $reportingTags,
                    ];
                    
                    // Always add tax_id (either VAT tax or zero tax)
                    if ($taxId) {
                        $developmentLineItem['tax_id'] = $taxId;
                    }
                    
                    $lineItems[] = $developmentLineItem;
                }
            }
            if (empty($lineItems)) {
                return ['success' => false, 'error' => 'No chargeable samples found'];
            }

            // Resolve Zoho customer (must already exist in Zoho; do not create)
            $zohoContactName = ($customer->zoho_customer ?? $customer->zoho_customer_tag ?? $customer->name);
            $zohoContactId = $this->resolveZohoContactIdByName($zohoContactName);

            if (!$zohoContactId) {
                if (($this->lastZohoCustomerFetchCount ?? 0) === 0) {
                    Log::warning('Zoho customers fetch returned 0 records');
                    return [
                        'success' => false,
                        'error' => 'Unable to fetch customers from Zoho (0 returned). Please connect Zoho and refresh.',
                    ];
                }
                Log::warning('Zoho customer name not matched', ['needle' => $zohoContactName, 'count' => $this->lastZohoCustomerFetchCount, 'suggestions' => $this->lastZohoSuggestions]);
                return [
                    'success' => false,
                    'error' => 'Zoho customer not found: ' . $zohoContactName . (!empty($this->lastZohoSuggestions) ? ' | Suggestions: ' . implode(', ', $this->lastZohoSuggestions) : ''),
                ];
            }

            // Create invoice payload (use existing Zoho contact id)
            $invoicePayload = [
                'customer_id' => $zohoContactId,
                'reference_number' => 'SAMPLES-' . now()->format('Ymd'),
                'date' => $this->date,
                'line_items' => $lineItems,
                'status' => 'draft',
                'custom_fields' => $this->buildSamplesCustomFields(),
            ];

            // Attempt to set a UK shipping address on the invoice to satisfy Zoho VAT rules
            try {
                $addresses = $this->zoho->getAddresses($this->org, $zohoContactId);
                // Prefer a shipping address in United Kingdom; otherwise any address with country = GB/UK
                $uk = $addresses->first(function ($addr) {
                    $country = mb_strtolower((string)($addr['country'] ?? ''));
                    $code    = mb_strtolower((string)($addr['country_code'] ?? ''));
                    return $country === 'united kingdom' || $country === 'uk' || $code === 'gb' || $code === 'uk';
                }) ?? $addresses->firstWhere('country_code', 'GB');

                if (!empty($uk) && !empty($uk['address_id'])) {
                    $invoicePayload['shipping_address_id'] = $uk['address_id'];
                }
            } catch (\Throwable $e) {
                Log::error('Failed to fetch/set Zoho shipping address', [
                    'contact_id' => $zohoContactId,
                    'error' => $e->getMessage(),
                ]);
            }

            // Do not add invoice-level reporting tags; tags are per line item

            // Create invoice in Zoho
            $zohoInvoice = $this->zoho->createInvoice($invoicePayload, $this->org);

            if (!$zohoInvoice || !isset($zohoInvoice['invoice_number'])) {
                Log::error('Zoho invoice creation returned unexpected response', ['response' => $zohoInvoice]);
                return ['success' => false, 'error' => 'Zoho returned an invalid response (no invoice number)'];
            }

            return [
                'success' => true,
                'invoice_number' => $zohoInvoice['invoice_number']
            ];

        } catch (\Exception $e) {
            Log::error('Customer invoice creation failed', [
                'customer_id' => $customerId,
                'error' => $e->getMessage()
            ]);
            
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    private function resolveZohoContactIdByName(string $name): ?string
    {
        $normalize = function ($value) {
            $v = mb_strtolower((string)$value);
            $v = trim($v);
            // Remove common corporate suffixes and punctuation/spaces
            $v = preg_replace('/\b(limited|ltd|plc|llc|co|company|inc|the)\b/i', '', $v);
            $v = preg_replace('/[^a-z0-9]+/i', '', $v);
            return $v;
        };

        $needle = $normalize($name);

        // First pass: cached customers
        $customers = $this->zoho->getCustomers($this->org);
        $this->lastZohoCustomerFetchCount = $customers->count();

        // 0) Exact match on customer_name/contact_name
        $exact = $customers->first(function ($c) use ($name) {
            $cn = trim((string)($c['customer_name'] ?? ''));
            $ct = trim((string)($c['contact_name'] ?? ''));
            $target = trim($name);
            return $cn === $target || $ct === $target;
        });
        if ($exact) {
            return $exact['contact_id'] ?? ($exact['customer_id'] ?? null);
        }

        // 1) Case-insensitive exact match
        $ci = $customers->first(function ($c) use ($name) {
            $cn = mb_strtolower(trim((string)($c['customer_name'] ?? '')));
            $ct = mb_strtolower(trim((string)($c['contact_name'] ?? '')));
            $target = mb_strtolower(trim($name));
            return $cn === $target || $ct === $target;
        });
        if ($ci) {
            return $ci['contact_id'] ?? ($ci['customer_id'] ?? null);
        }

        // 2) Normalized contains match across multiple fields
        $match = $customers->first(function ($c) use ($needle, $normalize) {
            $fields = [
                $c['customer_name'] ?? null,
                $c['contact_name'] ?? null,
                $c['company_name'] ?? null,
            ];
            foreach ($fields as $f) {
                if ($f === null) continue;
                $n = $normalize($f);
                if ($n === $needle || str_contains($n, $needle) || str_contains($needle, $n)) {
                    return true;
                }
            }
            return false;
        });
        if ($match) {
            return $match['contact_id'] ?? ($match['customer_id'] ?? null);
        }

        // Cache might be stale – clear and re-fetch once
        try { $this->zoho->clearCache(); } catch (\Throwable $e) {}
        $customers = $this->zoho->getCustomers($this->org);
        $this->lastZohoCustomerFetchCount = $customers->count();
        // Retry with normalized contains
        $match = $customers->first(function ($c) use ($needle, $normalize) {
            $fields = [
                $c['customer_name'] ?? null,
                $c['contact_name'] ?? null,
                $c['company_name'] ?? null,
            ];
            foreach ($fields as $f) {
                if ($f === null) continue;
                $n = $normalize($f);
                if ($n === $needle || str_contains($n, $needle) || str_contains($needle, $n)) {
                    return true;
                }
            }
            return false;
        });
        if (!$match && $this->lastZohoCustomerFetchCount) {
            // Build suggestions list most likely to match (contains token from input)
            $token = mb_substr($needle, 0, 6); // short token to broaden match
            $this->lastZohoSuggestions = $customers
                ->map(fn($c) => $c['customer_name'] ?? ($c['contact_name'] ?? ($c['company_name'] ?? '')))
                ->filter()
                ->unique()
                ->filter(function ($n) use ($normalize, $token) {
                    $nn = $normalize($n);
                    return str_contains($nn, $token);
                })
                ->take(8)
                ->values()
                ->toArray();
            \Log::info('Zoho customer matching failed', [
                'needle' => $name,
                'normalized' => $needle,
                'suggestions' => $this->lastZohoSuggestions,
                'total' => $this->lastZohoCustomerFetchCount,
            ]);
        }
        return $match['contact_id'] ?? ($match['customer_id'] ?? null);
    }

    // No customer creation or lookup required; we pass the mapped name directly

    private function getInvoiceCustomFieldId($name)
    {
        try {
            $fields = $this->zoho->getCustomFields($this->org);
            $id = collect($fields['invoice'])->firstWhere('label', $name)['customfield_id'] ?? "";
            return $id;
        } catch (\Exception $e) {
            Log::error('Failed to get custom field ID', ['name' => $name, 'error' => $e->getMessage()]);
            return '';
        }
    }

    /**
     * Build custom fields for samples invoices.
     * Explicitly clears the "Sample/Style Version" field to prevent it from appearing on the invoice.
     */
    private function buildSamplesCustomFields(): array
    {
        $customFields = [];
        
        // Explicitly clear the Sample/Style Version field
        $styleVersionFieldId = $this->getInvoiceCustomFieldId('Sample/Style Version');
        if (!empty($styleVersionFieldId)) {
            $customFields[] = [
                'customfield_id' => $styleVersionFieldId,
                'value' => '',
            ];
        }
        
        return $customFields;
    }

    /**
     * Resolve a Zoho account_id for this org from an account code or name.
     * Falls back to the first Income/Sales account if no explicit preference matches.
     */
    private function resolveAccountId(?string $preference): ?string
    {
        try {
            $accounts = $this->zoho->getChartOfAccounts($this->org);
            if ($accounts->isEmpty()) {
                Log::error('Zoho accounts fetch returned empty list');
                return null;
            }

            $pref = trim((string)($preference ?? ''));
            if ($pref !== '') {
                // Try by id
                $acc = $accounts->firstWhere('account_id', $pref);
                if ($acc) return $acc['account_id'];
                // Try by code
                $acc = $accounts->firstWhere('account_code', $pref);
                if ($acc) return $acc['account_id'];
                // Try by exact name
                $acc = $accounts->first(function ($a) use ($pref) {
                    return trim((string)($a['account_name'] ?? '')) === $pref;
                });
                if ($acc) return $acc['account_id'];
                // Try case-insensitive name
                $lp = mb_strtolower($pref);
                $acc = $accounts->first(function ($a) use ($lp) {
                    return mb_strtolower(trim((string)($a['account_name'] ?? ''))) === $lp;
                });
                if ($acc) return $acc['account_id'];
            }

            // Fallback: find an income/sales account
            $acc = $accounts->first(function ($a) {
                $type = mb_strtolower((string)($a['account_type'] ?? ''));
                $name = mb_strtolower((string)($a['account_name'] ?? ''));
                return str_contains($type, 'income') || str_contains($name, 'sales');
            });
            if ($acc) return $acc['account_id'];

            // Last resort: first account in list
            return $accounts->first()['account_id'] ?? null;
        } catch (\Throwable $e) {
            Log::error('resolveAccountId failed', ['error' => $e->getMessage()]);
            return null;
        }
    }

    private function updateSampleInvoices($samples, $invoiceNumber)
    {
        $sampleIds = collect($samples)->pluck('id')->toArray();
        
        Samples::whereIn('id', $sampleIds)->update([
            'rt_invoice' => $invoiceNumber,
            'fn_complete' => true,
            'updated_at' => now()
        ]);
    }

    public function getCustomers()
    {
        return Customer::all()->sortBy('name');
    }

    public function refresh()
    {
        $this->reset(['completionMessage', 'showCompletionModal', 'createdInvoices']);
        
        // Reload department options when refreshing
        if ($this->zohoConnected) {
            try {
                $this->departmentOptions = $this->getDepartmentOptions();
            } catch (\Exception $e) {
                Log::error('Failed to reload department options on refresh', ['error' => $e->getMessage()]);
            }
        }
        
        $this->dispatch('refresh');
    }

    public function closeAndRefresh()
    {
        $this->showCompletionModal = false;
        // Ask the page to refresh and close the outer modal
        $this->dispatch('refresh');
        $this->dispatch('close-modal');
    }

    // Uses HandlesZohoOAuth trait's initiateZohoOAuth() method for consistency

    public function getDepartmentOptions()
    {
        try {
            // Get the Department tag ID first
            $departmentTagId = $this->getReportingTagId('Department');
            if (!$departmentTagId) {
                Log::warning('Department tag not found, using default');
                return ['Knitted Wholesale Tom' => 'Knitted Wholesale Tom'];
            }
            
            $departments = $this->zoho->getReportingTagOptions($this->org, $departmentTagId);
            if (empty($departments)) {
                Log::warning('Zoho returned no Department tag options; falling back to default');
                return ['Knitted Wholesale Tom' => 'Knitted Wholesale Tom'];
            }
            return collect($departments)->pluck('tag_option_name', 'tag_option_id')->toArray();
        } catch (\Exception $e) {
            Log::error('Failed to fetch department options', ['error' => $e->getMessage()]);
            return ['Knitted Wholesale Tom' => 'Knitted Wholesale Tom'];
        }
    }

    private function buildReportingTags($customerTag, $factoredTag)
    {
        $tags = [];
        
        try {
            // Department tag
            $departmentTagId = $this->getReportingTagId('Department');
            if ($departmentTagId) {
                $departmentTagOptions = $this->zoho->getReportingTagOptions($this->org, $departmentTagId);
                $departmentTagOption = collect($departmentTagOptions)->where('tag_option_name', $this->reportingTagDepartment)->first();
                if ($departmentTagOption) {
                    $tags[] = [
                        'tag_id' => $departmentTagId,
                        'tag_option_id' => $departmentTagOption['tag_option_id'],
                    ];
                }
            }

            // Customer tag
            $customerTagId = $this->getReportingTagId('Customer');
            if ($customerTagId) {
                $customerTagOptions = $this->zoho->getReportingTagOptions($this->org, $customerTagId);
                $customerTagOption = collect($customerTagOptions)->where('tag_option_name', $customerTag)->first();
                if ($customerTagOption) {
                    $tags[] = [
                        'tag_id' => $customerTagId,
                        'tag_option_id' => $customerTagOption['tag_option_id'],
                    ];
                }
            }

            // Factored tag
            $factoredTagId = $this->getReportingTagId('Factored');
            if ($factoredTagId) {
                $factoredTagOptions = $this->zoho->getReportingTagOptions($this->org, $factoredTagId);
                $factoredTagOption = collect($factoredTagOptions)->where('tag_option_name', $factoredTag)->first();
                if ($factoredTagOption) {
                    $tags[] = [
                        'tag_id' => $factoredTagId,
                        'tag_option_id' => $factoredTagOption['tag_option_id'],
                    ];
                }
            }
        } catch (\Exception $e) {
            Log::error('Failed to build reporting tags', ['error' => $e->getMessage()]);
        }
        
        return $tags;
    }

    private function getReportingTagId($tagName)
    {
        try {
            $tags = $this->zoho->getReportingTags($this->org);
            $tag = collect($tags)->where('tag_name', $tagName)->first();
            return $tag['tag_id'] ?? null;
        } catch (\Exception $e) {
            Log::error('Failed to get reporting tag ID', ['tag_name' => $tagName, 'error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Get the VAT tax ID from Zoho for line items
     */
    private function getVatTaxId(): ?string
    {
        try {
            // Get tax rates from Zoho (using same org as ZohoInvoiceManager)
            $taxes = $this->zoho->getTaxes('20079258157');
            
            if (empty($taxes)) {
                Log::warning('No tax rates found in Zoho');
                return null;
            }
            
            // Look for 'Standard Rate' tax (same as ZohoInvoiceManager)
            $tax = collect($taxes)->where('tax_name', 'Standard Rate')->first();
            if ($tax) {
                return $tax['tax_id'];
            }
            
            // Fallback to first tax if Standard Rate not found
            return $taxes[0]['tax_id'] ?? null;
            
        } catch (\Exception $e) {
            Log::error('Failed to get VAT tax ID', ['error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Get the zero tax ID from Zoho for line items that should have no tax
     */
    private function getZeroTaxId(): ?string
    {
        try {
            // Get tax rates from Zoho
            $taxes = $this->zoho->getTaxes('20079258157');
            
            if (empty($taxes)) {
                Log::warning('No tax rates found in Zoho for zero tax');
                return null;
            }
            
            // Look for zero tax options
            $zeroTaxOptions = [
                'Zero Tax',
                'No Tax', 
                '0%',
                'Zero Rate',
                'Exempt'
            ];
            
            foreach ($zeroTaxOptions as $option) {
                $tax = collect($taxes)->where('tax_name', $option)->first();
                if ($tax) {
                    Log::info('Found zero tax option', ['tax_name' => $option, 'tax_id' => $tax['tax_id']]);
                    return $tax['tax_id'];
                }
            }
            
            // If no zero tax found, return null (no tax_id field)
            Log::info('No zero tax option found, will omit tax_id field');
            return null;
            
        } catch (\Exception $e) {
            Log::error('Failed to get zero tax ID', ['error' => $e->getMessage()]);
            return null;
        }
    }

    public function render()
    {
        return view('livewire.finance.samples-invoice-generator', [
            'customers' => $this->getCustomers(),
        ]);
    }
}


