<?php

namespace App\Http\Livewire\Finance\Cashflow;

use App\Models\Seasons;
use Livewire\Component;
use App\Models\Cashflow;
use App\Models\Customer;
use App\Models\Countries;
use App\Models\Suppliers;
use App\Helper\Conversions;
use App\Models\Departments;
use App\Models\CashflowLine;
use App\Models\ShipmentLine;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;

class WhCashflowNew extends Component
{
    public $name;
    public $exfty_date_to, $customer_id, $factory_id, $country_id, $department_id, $category_id;

    protected $rules = [
        'exfty_date_to' => 'required|date',
        'customer_id' => 'nullable|exists:customers,id',
        'factory_id' => 'nullable|exists:suppliers,id',
        'country_id' => 'nullable|exists:countries,id',
        'department_id' => 'nullable|exists:departments,id',
        'category_id' => 'nullable|in:ladies,mens,childrens,accessories',
    ];

    #[Computed]
    public function customers()
    {
        return Customer::allCached();
    }

    #[Computed]
    public function factories()
    {
        return Suppliers::allCached();
    }

    #[Computed]
    public function countries()
    {
        return Countries::all();
    }

    #[Computed]
    public function departments()
    {
        return Departments::allCached();
    }
    #[Computed]
    public function seasons()
    {
        return Seasons::allCached();
    }
    // public function render()
    // {
    //     return view('livewire.finance.cashflow.wh-cashflow-new');
    // }

    public function refreshData(){
        session()->flash('message', 'Validation Error');
		session()->flash('alert-class', 'alert-warning');
        $this->validate();
        $results = $this->getResults();
        $this->extractResults($results);
        $this->dispatch('close-modal');
        $this->dispatch('refresh-page');
        session()->flash('message', 'Created');
		session()->flash('alert-class', 'alert-success');
    }

    public function extractResults($results){
        $cashflow = Cashflow::create([
            'name' => $this->name,
            'data' => json_encode([
                'exfty_date_to' => $this->exfty_date_to,
                'customer' => Customer::find($this->customer_id)?->name,
                'factory' => Suppliers::find($this->factory_id)?->name,
                'country' => Countries::find($this->country_id)?->country,
                'department' => Departments::find($this->department_id)?->description,
                'category' => $this->category_id,
            ]),
        ]);
        $lines = [];
        foreach($results as $result){
            $line = new CashflowLine;
            $line->cashflows_id = $cashflow->id;
            $line->shipment_lines_id = $result->id;
            $line->cmt = $result->total_cmt;
            $line->cmt_terms = $result->fact_terms;
            $line->cmt_currency = $result->fact_currency;
            $line->total_cmt_base = $result->total_cmt_base;
            $line->yarn = $result->total_yarn;
            $line->yarn_terms = $result->yarn_terms;
            $line->yarn_currency = '£';
            $line->total_yarn_base = $result->total_yarn_base ?? 0;
            $line->sale = $result->total_quote;
            $line->sale_terms = $result->customer_terms;
            $line->sale_currency = $result->cust_currency;
            $line->total_quote_base = $result->total_quote_base;
            $vatAmounts = $this->checkVAT($result);
            $line->vat = $vatAmounts['vat'];
            $line->vat_base = $vatAmounts['vat_base'];
            $line->factored = $result->customer_factored ?? FALSE;
            $line->seasons_id = $result->seasons_id;
            $line->suppliers_id = $result->factory_id;
            $line->customers_id = $result->customer_id;
            $line->designs_id = $result->designs_id;
            $line->customer_po = $result->customer_po;
            $line->customer_ref = $result->customer_ref;
            $line->description = $result->description;
            $line->colour = $result->colour;
            $line->incoterms = $result->incoterms;
            $line->exfty = $result->exfty;
            $line->qty = $result->total_qty;
            $line->is_full_factored = $result->is_full_factored;
            $line->is_complete = $result->complete;
            $line->total_factoring = $result->total_factoring;
            
            // Note: yarn_spinner_details are calculated but not stored to database
            // The cashflow pivot can access this data by re-running the yarn calculation if needed
            
            $lines[] = $line;
        }
        $cashflow->cashflow_lines()->saveMany($lines);
    }

    public function checkVAT($result){
        if($result->incoterms == 'CIF'
            || $result->incoterms == 'DAP'
            || $result->incoterms == 'DDP'
            || $result->incoterms == 'DPU'
            || $result->incoterms == 'CFR'
            || $result->incoterms == 'CIP'
            || $result->incoterms == 'CPT')
        {
            return [
                'vat' => $result->total_quote * 0.2,
                'vat_base' => $result->total_quote_base * 0.2
            ];
        }
        else
        {
            return [
                'vat' => 0,
                'vat_base' => 0
            ];
        }
    }

    public function getResults(){
        $query = ShipmentLine::select(
            "shipment_lines.id AS id",
            "shipment_lines.customer_order_lines_id",
            "seasons.id AS seasons_id",
            "factory.id AS factory_id",
            "customers.id AS customer_id",
            "styles.designs_id",
            "customer_orders.customer_po AS customer_po",
            "styles.customer_ref",
            "designs.description AS description",
            "colourways.name AS colour",
            "customer_orders.incoterms AS incoterms",

            "shipment_lines.complete AS complete",

            "factory.payment_terms AS fact_terms",
            "factory.currency AS fact_currency",

            "customers.currency AS cust_currency",

            "customers.factored AS customer_factored",



        )
        ->selectRaw("
                (
                    SELECT
                        JSON_ARRAYAGG(
                            JSON_OBJECT(
                                'percentage', percentage,
                                'days', days
                            )
                        )
                    FROM
                        customer_payment_terms
                    WHERE
                        customer_payment_terms.customer_id = customers.id
                ) AS customer_terms
        ")
        ->selectRaw("
            CASE
                WHEN factory_cust_date > exfty
                THEN factory_cust_date
                ELSE exfty
            END AS exfty")
        ->selectRaw("customer_order_lines.factory_cust_date")
        ->selectRaw("
            (select
                (
                    case when (
                    min(`sl`.`collection_date`) is not null
                    ) then min(`sl`.`collection_date`) else min(`sl`.`exfty`) end
                )
                from
                `shipment_lines` `sl`
                where
                (
                    `sl`.`shipment_id` = `shipment_lines`.`shipment_id`
                )
            ) AS `truck_first_collection`
        ")
        ->selectRaw("(select s.payment_terms from colourway_yarns cy join yarn_colours yc on yc.id = cy.yarn_colours_id join yarns y on y.id = yc.yarn_id join suppliers s on s.id = y.suppliers_id where cy.colourways_id = colourways.id order by cy.percentage desc limit 1) as yarn_terms")
        ->selectRaw("tc.cached_data as cached_totals")

        ->Join('customer_order_lines', function($join)
        {
            $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id');
            $join->on('customer_order_lines.cancelled', '=', DB::raw(0));
        })
        ->Join('customer_orders', function($join)
        {
            $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
            $join->on('customer_orders.order_type', '=', DB::raw("'wholesale'"));
            $join->on('customer_orders.cancelled', '=', DB::raw(0));
        })
        ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
        ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
        ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
        ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
        ->join('designs', 'designs.id', '=', 'styles.designs_id')
        ->leftJoin('shipments', 'shipments.id', '=', 'shipment_lines.shipment_id')
        ->leftJoin('suppliers as transporters', 'transporters.id', '=', 'shipments.transporter_id')
        ->leftJoin('total_cache as tc', function($join) {
            $join->on('tc.entity_id', '=', 'shipment_lines.id')
                 ->where('tc.entity_type', '=', 'shipment_line')
                 ->where('tc.cache_key', '=', 'prices')
                 ->whereNotNull('tc.fresh_at');
        })
        ->groupBy('shipment_lines.id');
        

        $query->where('customer_orders.order_type', 'wholesale');
        $query->where('customer_order_lines.cancelled', FALSE);
        $query->where('customer_orders.customers_id', '!=', 9);

        $query->where(function($q) {
            $q->whereNull('shipment_lines.rt_invoice')
            ->orWhere('shipment_lines.rt_invoice', '');
        });

        $query->where('exfty', '<=', $this->exfty_date_to);

        if(!empty($this->customer_id)){
            $query->where('customer_orders.customers_id', $this->customer_id);
        }

        if(!empty($this->factory_id)){
            $query->whereHas('customer_order_lines.colourways.style_versions', function($q){
                $q->where('factory_id', $this->factory_id);
            });
        }

        if(!empty($this->country_id)){
            $query->whereHas('customer_order_lines.colourways.style_versions.factories', function($q){
                $q->where('countries_id', $this->country_id);
            });
        }

        if(!empty($this->department_id)){
            $query->whereHas('customer_order_lines.customer_orders', function($q){
                $q->where('departments_id', $this->department_id);
            });
        }

        if(!empty($this->category_id)){
            $query->whereHas('customer_order_lines.colourways.style_versions.styles', function($q){
                $q->where('category', $this->category_id);
            });
        }

        


        $query->where('customer_orders.cancelled', 0)
        ->where('customer_order_lines.cancelled', 0);


        $query->orderByRaw("
            -(shipment_lines.complete),
            -(truck_first_collection) desc,
            shipment_lines.shipment_id,
            -(shipment_lines.exfty) desc,
            factory.countries_id,
            -(factory.name),
            -(customers.name),
            -(
            customer_orders.customer_po
            ),
            -(styles.id),
            -(colourways.name)
        ");

        $results = $query->get();


        foreach($results as $l=>$line){
            // Parse cached totals from SQL
            $cachedTotals = json_decode($line->cached_totals ?? '{}', true);
            
            // Use cached totals from SQL
            $results[$l]->total_quote = $cachedTotals['quote'] ?? 0;
            $results[$l]->total_quote_base = $cachedTotals['quote_base'] ?? 0;
            $results[$l]->total_cmt = $cachedTotals['cmt'] ?? 0;
            $results[$l]->total_cmt_base = $cachedTotals['cmt_base'] ?? 
                Conversions::convertCurrency($line->fact_currency, '£', $cachedTotals['cmt'] ?? 0, $line->seasons_id);
            
            // Calculate yarn costs properly - can't use cache as prices can be made up from several spinners
            $totalYarnBase = 0;
            $totalFactoringCost = 0;
            $isFullFactored = FALSE;
            $yarnSpinnerDetails = []; // Store details for each spinner for cashflow pivot
            
            // Get shipment line sizes and calculate yarn costs
            $shipmentLineSizes = \DB::table('shipment_line_sizes as sls')
                ->select('sls.*', 's.name as size_name')
                ->join('sizes as s', 's.id', '=', 'sls.sizes_id')
                ->where('sls.shipment_line_id', $line->id)
                ->get();
                
            foreach($shipmentLineSizes as $size) {
                // Get the colourway_id from customer_order_lines
                $colourwayId = \DB::table('customer_order_lines as col')
                    ->where('col.id', $line->customer_order_lines_id)
                    ->value('col.colourways_id');
                
                // Get the style_versions_id from colourways
                $styleVersionId = \DB::table('colourways')
                    ->where('id', $colourwayId)
                    ->value('style_versions_id');
                
                // Use the existing price resolution system instead of trying to recreate it
                $orderLineQty = $line->customer_order_lines->customer_order_line_quantities
                    ->where('sizes_id', $size->sizes_id)
                    ->first();
                    
                if ($orderLineQty) {
                    $priceModel = $orderLineQty->price_model;
                    
                    // Get the model from the prices table directly
                    $model = \DB::table('prices')
                        ->where('style_versions_id', $styleVersionId)
                        ->where('sizes_id', $size->sizes_id)
                        ->value('model') ?? 'standard';
                    
                    if ($priceModel && is_array($priceModel)) {
                        $qty = $size->shipped_qty > 0 ? $size->shipped_qty : $size->qty;
                        
                        // Handle different price models
                        if ($model == 'full_factored') {
                        // Full factored: no yarn costs
                        $isFullFactored = TRUE;
                        // Don't add any yarn costs
                    } elseif ($model == 'manual') {
                        // Manual: use manually entered yarn value and get terms from biggest spinner
                        $yarnValueBase = $priceModel['yarn_cost_kg'] ?? 0;
                        $totalYarnBase += $qty * $yarnValueBase;
                        
                        // Get terms from biggest spinner in colourway, or default to 30 days
                        $biggestSpinnerTerms = $this->getBiggestSpinnerTerms($colourwayId, $styleVersionId);
                        $yarnSpinnerDetails['manual'] = [
                            'spinner_id' => 'manual',
                            'spinner_name' => 'Manual Entry',
                            'payment_terms' => $biggestSpinnerTerms,
                            'currency' => '£',
                            'total_cost' => $qty * $yarnValueBase,
                            'total_cost_base' => $qty * $yarnValueBase,
                            'percentage' => 100,
                        ];
                    } else {
                        // Standard: use the actual yarn cost from price model (already calculated correctly)
                        $yarnCostEur = $priceModel['yarn_cost_kg'] ?? 0;
                        $yarnCostForThisSize = $yarnCostEur * $qty;
                        $yarnCostBase = Conversions::convertCurrency('€', '£', $yarnCostForThisSize, $line->seasons_id ?? 1);
                        $totalYarnBase += $yarnCostBase;
                        
                        // Get spinner breakdown for standard model - split the actual yarn cost proportionally
                        $spinnerBreakdown = $this->getSpinnerBreakdownForStandard($colourwayId, $styleVersionId, $qty, $yarnCostBase);
                        foreach ($spinnerBreakdown as $spinnerKey => $spinnerData) {
                            if (!isset($yarnSpinnerDetails[$spinnerKey])) {
                                $yarnSpinnerDetails[$spinnerKey] = $spinnerData;
                            } else {
                                $yarnSpinnerDetails[$spinnerKey]['total_cost'] += $spinnerData['total_cost'];
                                $yarnSpinnerDetails[$spinnerKey]['total_cost_base'] += $spinnerData['total_cost_base'];
                            }
                        }
                    }
                    
                    // Calculate factoring using the helper function if customer is factored
                    if ($line->customer_factored) {
                        $priceArray = [
                            'quote' => $priceModel['quote'] ?? 0,
                            'cust_terms' => $line->customer_terms ?? '[]'
                        ];
                        $factoringPerPiece = calculateFactoring($priceArray);
                        $totalFactoringCost += $qty * $factoringPerPiece;
                    }
                    }
                }
            }
            
            $results[$l]->total_yarn = $totalYarnBase; // Use base currency value for compatibility
            $results[$l]->total_yarn_base = $totalYarnBase;
            $results[$l]->total_factoring = $totalFactoringCost;
            $results[$l]->is_full_factored = $isFullFactored;
            $results[$l]->yarn_spinner_details = $yarnSpinnerDetails; // For cashflow pivot to split by spinner
            
            // Clean up unused data
            unset($results[$l]->cached_totals);
            unset($results[$l]->truck_first_collection);
        }

        // dd($results->sum('total_quote'), $results->sum('total_quote_base'), $results->sum('total_cmt'), $results->sum('total_cmt_base'));

        return $results;
    }

    /**
     * Calculate yarn value base (in GBP) for a given price record
     * This mimics the logic from Price model's getYarnValueBaseAttribute
     */
    private function calculateYarnValueBase($price, $seasonId)
    {
        if (!$price || !$price->yarn_value) {
            return 0;
        }

        // Get the style version to find factory currency
        $styleVersion = \DB::table('style_versions as sv')
            ->select('sv.*', 'f.currency as factory_currency')
            ->join('suppliers as f', 'f.id', '=', 'sv.factory_id')
            ->where('sv.id', $price->style_versions_id)
            ->first();

        $factCurrency = $styleVersion->factory_currency ?? 'USD';
        
        // Convert yarn value from factory currency to GBP
        return Conversions::convertCurrency($factCurrency, '£', $price->yarn_value, $seasonId);
    }

    /**
     * Get yarn spinner breakdown for a shipment line (for cashflow pivot usage)
     * Returns array of spinner details with costs and payment terms
     */
    public static function getYarnSpinnerBreakdown($shipmentLineId)
    {
        $shipmentLine = ShipmentLine::with(['customer_order_lines.customer_order_line_quantities', 'shipment_line_sizes'])->find($shipmentLineId);
        if (!$shipmentLine) {
            return [];
        }

        $yarnSpinnerDetails = [];
        $colourwayId = $shipmentLine->customer_order_lines->colourways_id;
        $styleVersionId = $shipmentLine->customer_order_lines->colourways->style_versions_id;

        foreach($shipmentLine->shipment_line_sizes as $size) {
            // Use the existing price resolution system
            $orderLineQty = $shipmentLine->customer_order_lines->customer_order_line_quantities
                ->where('sizes_id', $size->sizes_id)
                ->first();
                
            if ($orderLineQty) {
                $priceModel = $orderLineQty->price_model;
                
                // Get the model from the prices table directly
                $model = \DB::table('prices')
                    ->where('style_versions_id', $styleVersionId)
                    ->where('sizes_id', $size->sizes_id)
                    ->value('model') ?? 'standard';
                
                if ($priceModel && is_array($priceModel)) {
                    $qty = $size->shipped_qty > 0 ? $size->shipped_qty : $size->qty;
                    
                    // Handle different price models
                    if ($model == 'full_factored') {
                    // Full factored: no yarn costs
                    continue;
                } elseif ($model == 'manual') {
                    // Manual: use manually entered yarn value and get terms from biggest spinner
                    $yarnValueBase = $priceModel['yarn_cost_kg'] ?? 0;
                    $biggestSpinnerTerms = self::getBiggestSpinnerTermsStatic($colourwayId, $styleVersionId);
                    
                    $yarnSpinnerDetails['manual'] = [
                        'spinner_id' => 'manual',
                        'spinner_name' => 'Manual Entry',
                        'payment_terms' => $biggestSpinnerTerms,
                        'currency' => '£',
                        'total_cost' => ($yarnSpinnerDetails['manual']['total_cost'] ?? 0) + ($qty * $yarnValueBase),
                        'percentage' => 100,
                    ];
                } else {
                    // Standard: use the actual yarn cost from price model (already calculated correctly)
                    $yarnCostEur = $priceModel['yarn_cost_kg'] ?? 0;
                    $yarnCostForThisSize = $yarnCostEur * $qty;
                    // Get season_id from the shipment line
                    $seasonId = $shipmentLine->customer_order_lines->customer_orders->seasons_id ?? 1;
                    $yarnCostBase = Conversions::convertCurrency('€', '£', $yarnCostForThisSize, $seasonId);
                    
                    // Get spinner breakdown for standard model - split the actual yarn cost proportionally
                    $spinnerBreakdown = self::getSpinnerBreakdownForStandardStatic($colourwayId, $styleVersionId, $qty, $yarnCostBase);
                    
                    foreach ($spinnerBreakdown as $spinnerKey => $spinnerData) {
                        if (!isset($yarnSpinnerDetails[$spinnerKey])) {
                            $yarnSpinnerDetails[$spinnerKey] = $spinnerData;
                        } else {
                            $yarnSpinnerDetails[$spinnerKey]['total_cost'] += $spinnerData['total_cost'];
                        }
                    }
                }
                }
            }
        }

        return array_values($yarnSpinnerDetails);
    }

    /**
     * Get payment terms from the biggest spinner in a colourway, or default to 30 days
     */
    private function getBiggestSpinnerTerms($colourwayId, $styleVersionId)
    {
        // First try the current colourway
        $biggestSpinner = \DB::table('colourway_yarns as cy')
            ->join('yarn_colours as yc', 'yc.id', '=', 'cy.yarn_colours_id')
            ->join('yarns as y', 'y.id', '=', 'yc.yarn_id')
            ->join('suppliers as s', 's.id', '=', 'y.suppliers_id')
            ->where('cy.colourways_id', $colourwayId)
            ->orderBy('cy.percentage', 'desc')
            ->value('s.payment_terms');
            
        if ($biggestSpinner) {
            return $biggestSpinner;
        }
        
        // Fallback to first colourway with yarn data in same style_version
        $fallbackTerms = \DB::table('colourways as c')
            ->join('colourway_yarns as cy', 'cy.colourways_id', '=', 'c.id')
            ->join('yarn_colours as yc', 'yc.id', '=', 'cy.yarn_colours_id')
            ->join('yarns as y', 'y.id', '=', 'yc.yarn_id')
            ->join('suppliers as s', 's.id', '=', 'y.suppliers_id')
            ->where('c.style_versions_id', $styleVersionId)
            ->where('cy.percentage', '>', 0)
            ->orderBy('c.id')
            ->orderBy('cy.percentage', 'desc')
            ->value('s.payment_terms');
            
        return $fallbackTerms ?: 30; // Default to 30 days if no spinner data found
    }

    /**
     * Get spinner breakdown for standard price model
     */
    private function getSpinnerBreakdownForStandard($colourwayId, $styleVersionId, $qty, $totalYarnValueBase)
    {
        // Get yarn suppliers for this colourway
        $colourwayYarns = \DB::table('colourway_yarns as cy')
            ->select('cy.percentage', 'cy.price', 's.payment_terms', 's.currency', 's.name as spinner_name', 's.id as spinner_id')
            ->join('yarn_colours as yc', 'yc.id', '=', 'cy.yarn_colours_id')
            ->join('yarns as y', 'y.id', '=', 'yc.yarn_id')
            ->join('suppliers as s', 's.id', '=', 'y.suppliers_id')
            ->where('cy.colourways_id', $colourwayId)
            ->get();
        
        // If no yarn data for this colourway OR all percentages are 0, find first colourway in same style_version with yarn data
        $needsFallback = $colourwayYarns->isEmpty() || $colourwayYarns->sum('percentage') == 0;
        
        if ($needsFallback) {
            $fallbackColourwayId = \DB::table('colourways as c')
                ->join('colourway_yarns as cy', 'cy.colourways_id', '=', 'c.id')
                ->where('c.style_versions_id', $styleVersionId)
                ->where('cy.percentage', '>', 0)
                ->orderBy('c.id')
                ->value('c.id');
                
            if ($fallbackColourwayId) {
                $colourwayYarns = \DB::table('colourway_yarns as cy')
                    ->select('cy.percentage', 'cy.price', 's.payment_terms', 's.currency', 's.name as spinner_name', 's.id as spinner_id')
                    ->join('yarn_colours as yc', 'yc.id', '=', 'cy.yarn_colours_id')
                    ->join('yarns as y', 'y.id', '=', 'yc.yarn_id')
                    ->join('suppliers as s', 's.id', '=', 'y.suppliers_id')
                    ->where('cy.colourways_id', $fallbackColourwayId)
                    ->get();
            }
        }
        
        $spinnerDetails = [];
        $totalPercentage = $colourwayYarns->sum('percentage');
        
        foreach($colourwayYarns as $yarnSpinner) {
            if ($totalPercentage > 0) {
                // Calculate proportional cost based on percentage
                $proportionalCost = ($yarnSpinner->percentage / $totalPercentage) * $totalYarnValueBase;
                $spinnerKey = $yarnSpinner->spinner_id;
                
                if (!isset($spinnerDetails[$spinnerKey])) {
                    $spinnerDetails[$spinnerKey] = [
                        'spinner_id' => $yarnSpinner->spinner_id,
                        'spinner_name' => $yarnSpinner->spinner_name,
                        'payment_terms' => $yarnSpinner->payment_terms,
                        'currency' => '£', // Already converted to base
                        'total_cost' => 0,
                        'total_cost_base' => 0,
                        'percentage' => 0,
                    ];
                }
                
                // Combine multiple entries for the same spinner
                $spinnerDetails[$spinnerKey]['total_cost'] += $proportionalCost;
                $spinnerDetails[$spinnerKey]['total_cost_base'] += $proportionalCost;
                $spinnerDetails[$spinnerKey]['percentage'] += $yarnSpinner->percentage;
            }
        }
        
        return $spinnerDetails;
    }

    /**
     * Static version of getBiggestSpinnerTerms for use in static context
     */
    private static function getBiggestSpinnerTermsStatic($colourwayId, $styleVersionId)
    {
        // First try the current colourway
        $biggestSpinner = \DB::table('colourway_yarns as cy')
            ->join('yarn_colours as yc', 'yc.id', '=', 'cy.yarn_colours_id')
            ->join('yarns as y', 'y.id', '=', 'yc.yarn_id')
            ->join('suppliers as s', 's.id', '=', 'y.suppliers_id')
            ->where('cy.colourways_id', $colourwayId)
            ->orderBy('cy.percentage', 'desc')
            ->value('s.payment_terms');
            
        if ($biggestSpinner) {
            return $biggestSpinner;
        }
        
        // Fallback to first colourway with yarn data in same style_version
        $fallbackTerms = \DB::table('colourways as c')
            ->join('colourway_yarns as cy', 'cy.colourways_id', '=', 'c.id')
            ->join('yarn_colours as yc', 'yc.id', '=', 'cy.yarn_colours_id')
            ->join('yarns as y', 'y.id', '=', 'yc.yarn_id')
            ->join('suppliers as s', 's.id', '=', 'y.suppliers_id')
            ->where('c.style_versions_id', $styleVersionId)
            ->where('cy.percentage', '>', 0)
            ->orderBy('c.id')
            ->orderBy('cy.percentage', 'desc')
            ->value('s.payment_terms');
            
        return $fallbackTerms ?: 30; // Default to 30 days if no spinner data found
    }

    /**
     * Static version of getSpinnerBreakdownForStandard for use in static context
     */
    public static function getSpinnerBreakdownForStandardStatic($colourwayId, $styleVersionId, $qty, $totalYarnValueBase)
    {
        // Get yarn suppliers for this colourway
        $colourwayYarns = \DB::table('colourway_yarns as cy')
            ->select('cy.percentage', 'cy.price', 's.payment_terms', 's.currency', 's.name as spinner_name', 's.id as spinner_id')
            ->join('yarn_colours as yc', 'yc.id', '=', 'cy.yarn_colours_id')
            ->join('yarns as y', 'y.id', '=', 'yc.yarn_id')
            ->join('suppliers as s', 's.id', '=', 'y.suppliers_id')
            ->where('cy.colourways_id', $colourwayId)
            ->get();
        
        // If no yarn data for this colourway OR all percentages are 0, find first colourway in same style_version with yarn data
        $needsFallback = $colourwayYarns->isEmpty() || $colourwayYarns->sum('percentage') == 0;
        
        if ($needsFallback) {
            $fallbackColourwayId = \DB::table('colourways as c')
                ->join('colourway_yarns as cy', 'cy.colourways_id', '=', 'c.id')
                ->where('c.style_versions_id', $styleVersionId)
                ->where('cy.percentage', '>', 0)
                ->orderBy('c.id')
                ->value('c.id');
                
            if ($fallbackColourwayId) {
                $colourwayYarns = \DB::table('colourway_yarns as cy')
                    ->select('cy.percentage', 'cy.price', 's.payment_terms', 's.currency', 's.name as spinner_name', 's.id as spinner_id')
                    ->join('yarn_colours as yc', 'yc.id', '=', 'cy.yarn_colours_id')
                    ->join('yarns as y', 'y.id', '=', 'yc.yarn_id')
                    ->join('suppliers as s', 's.id', '=', 'y.suppliers_id')
                    ->where('cy.colourways_id', $fallbackColourwayId)
                    ->get();
            }
        }
        
        $spinnerDetails = [];
        $totalPercentage = $colourwayYarns->sum('percentage');
        
        foreach($colourwayYarns as $yarnSpinner) {
            if ($totalPercentage > 0) {
                // Calculate proportional cost based on percentage
                $proportionalCost = ($yarnSpinner->percentage / $totalPercentage) * $totalYarnValueBase;
                $spinnerKey = $yarnSpinner->spinner_id;
                
                if (!isset($spinnerDetails[$spinnerKey])) {
                    $spinnerDetails[$spinnerKey] = [
                        'spinner_id' => $yarnSpinner->spinner_id,
                        'spinner_name' => $yarnSpinner->spinner_name,
                        'payment_terms' => $yarnSpinner->payment_terms,
                        'currency' => '£', // Already converted to base
                        'total_cost' => 0,
                        'percentage' => 0,
                    ];
                }
                
                // Combine multiple entries for the same spinner
                $spinnerDetails[$spinnerKey]['total_cost'] += $proportionalCost;
                $spinnerDetails[$spinnerKey]['percentage'] += $yarnSpinner->percentage;
            }
        }
        
        return $spinnerDetails;
    }

}