<?php

namespace App\Http\Livewire\Finance;

use Livewire\Component;
use App\Services\ZohoService;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\Http\Livewire\Traits\HandlesZohoOAuth;

class BankFileGenerator extends Component
{
    use HandlesZohoOAuth;
    public $fromDate;
    public $toDate;
    public $organizationId;
    public $debitAccount;  // T010 - Your company's account number with sort code
    public $debitSortCode;  // Part of T010
    public $isProcessing = false;
    public $errorMessage = '';
    public $successMessage = '';
    public $zohoConnected = false;

    protected $rules = [
        'fromDate' => 'required|date',
        'toDate' => 'required|date|after_or_equal:fromDate',
        'debitAccount' => 'required',
        'debitSortCode' => 'required',
    ];

    public function mount()
    {
        // Default to today
        $this->fromDate = Carbon::now()->format('Y-m-d');
        $this->toDate = Carbon::now()->format('Y-m-d');
        
        // Use organization ID from environment
        $this->organizationId = env('ZOHO_ORG_ID', '20079258157');
        
        // Default debit account details (your company's bank account)
        $this->debitSortCode = env('NATWEST_DEBIT_SORT_CODE', '');
        $this->debitAccount = env('NATWEST_DEBIT_ACCOUNT', '');
        
        // Check Zoho connection status
        $zohoService = app(ZohoService::class);
        $this->zohoConnected = $zohoService->isConnected();
    }

    public function generateBankFile()
    {
        $this->validate();
        
        $this->isProcessing = true;
        $this->errorMessage = '';
        $this->successMessage = '';

        try {
            $zohoService = app(ZohoService::class);
            
            // Check if connected to Zoho
            if (!$zohoService->isConnected()) {
                throw new \Exception('Not connected to Zoho Books. Please authenticate first.');
            }

            // Fetch vendor payments
            $payments = $zohoService->getVendorPayments(
                $this->organizationId,
                $this->fromDate,
                $this->toDate
            );

            // Log payment modes to see what's available
            Log::info('All payments fetched', [
                'count' => $payments->count(),
                'payment_modes' => $payments->pluck('payment_mode')->unique()->toArray()
            ]);

            // Filter to only include payments with payment method "Bank File" (if the field exists and matches)
            $filteredPayments = $payments->filter(function($payment) {
                $paymentMode = strtolower($payment['payment_mode'] ?? '');
                // Only filter if payment_mode exists and contains bank/file
                if (empty($paymentMode)) {
                    return true; // Include if no payment_mode field
                }
                return str_contains($paymentMode, 'bank') || str_contains($paymentMode, 'file');
            });

            // Use filtered payments if we have any, otherwise use all
            if ($filteredPayments->isNotEmpty()) {
                $payments = $filteredPayments;
                Log::info('Filtered to Bank File payments', ['count' => $payments->count()]);
            } else {
                Log::warning('No payments matched Bank File filter, using all payments');
            }

            if ($payments->isEmpty()) {
                $this->errorMessage = 'No vendor payments found for the selected date range.';
                $this->isProcessing = false;
                return;
            }

            // Generate file content
            $fileContent = $this->buildBankFileContent($payments, $zohoService);

            // Return file as download
            $fileName = 'NatWest_Payments_' . date('Ymd_His') . '.csv';
            
            $this->successMessage = 'Bank file generated successfully. Download should start automatically.';
            
            // Use Livewire's download feature
            return Response::streamDownload(function () use ($fileContent) {
                echo $fileContent;
            }, $fileName, [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
            ]);

        } catch (\Exception $e) {
            Log::error('Bank file generation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            $this->errorMessage = 'Error: ' . $e->getMessage();
        } finally {
            $this->isProcessing = false;
        }
    }

    /**
     * Build the NatWest bank file content from vendor payments in CSV format
     * Format: NatWest Standard Domestic Payment Template
     * Field mapping: H001-H003(empty), T001(01), T002(T), T003(template ref), T010(debit account), 
     * T013(GBP), T022(sort code), T028(account number), T030(beneficiary name), T034(reference)
     */
    protected function buildBankFileContent($payments, ZohoService $zohoService): string
    {
        $fileLines = [];

        foreach ($payments as $payment) {
            $amount = $payment['amount'] ?? 0;

            // Try multiple possible date field names from Zoho API
            $paymentDate = $payment['date'];
            $referenceNumber = $payment['reference_number'] ?? '';
            $vendorId = $payment['vendor_id'] ?? null;
            $paymentId = $payment['payment_id'] ?? '';
            
            // Log the FULL payment data to debug date issues
            Log::info('Processing payment - FULL DATA', [
                'full_payment_array' => $payment,
                'payment_date_used' => $paymentDate,
                'has_payment_date' => isset($payment['payment_date']),
                'has_date' => isset($payment['date']),
                'has_payment_expected_date' => isset($payment['payment_expected_date']),
            ]);

            // Skip if no vendor
            if (!$vendorId) {
                Log::info('Skipping payment with no vendor_id', ['payment' => $payment]);
                continue;
            }

            // Fetch vendor details
            $vendor = $zohoService->getVendorById($this->organizationId, $vendorId);
            
            if (!$vendor) {
                Log::warning('Could not fetch vendor details', ['vendor_id' => $vendorId]);
                continue;
            }

            $vendorName = $vendor['contact_name'] ?? 'Unknown';
            
            // Extract custom fields (Sort Code and Account Number)
            $sortCode = '';
            $accountNumber = '';
            
            $customFields = $vendor['custom_fields'] ?? [];
            foreach ($customFields as $field) {
                $label = trim($field['label'] ?? '');
                $value = trim($field['value'] ?? '');
                
                // Case-insensitive matching for flexibility
                if (strcasecmp($label, 'Sort Code') === 0) {
                    $sortCode = $value;
                }
                if (strcasecmp($label, 'Account Number') === 0) {
                    $accountNumber = $value;
                }
            }
            
            // Log if custom fields are missing
            if (empty($sortCode)) {
                Log::warning('Missing Sort Code for vendor', [
                    'vendor_id' => $vendorId,
                    'vendor_name' => $vendorName,
                    'custom_fields' => $customFields
                ]);
            }
            if (empty($accountNumber)) {
                Log::warning('Missing Account Number for vendor', [
                    'vendor_id' => $vendorId,
                    'vendor_name' => $vendorName,
                    'custom_fields' => $customFields
                ]);
            }

            // Format amount as decimal (e.g., 166.42)
            $amountDecimal = number_format((float)$amount, 2, '.', '');

            // Format debit account: sortcode + account number (no dashes)
            $debitAccountFull = str_replace(['-', ' '], '', $this->debitSortCode) . $this->debitAccount;
            
            // Format sort code (remove dashes)
            $sortCodeFormatted = str_replace(['-', ' '], '', $sortCode);
            
            // Format payment date as DDMMYYYY (from Zoho payment_date field)
            if ($paymentDate) {
                try {
                    $payDate = Carbon::parse($paymentDate);
                    $payDateFormatted = $payDate->format('dmY'); // DDMMYYYY
                    Log::info('Using Zoho payment date', [
                        'vendor' => $vendorName,
                        'zoho_date' => $paymentDate,
                        'formatted' => $payDateFormatted
                    ]);
                } catch (\Exception $e) {
                    $payDateFormatted = Carbon::now()->format('dmY');
                    Log::warning('Failed to parse Zoho payment date, using today', [
                        'vendor' => $vendorName,
                        'zoho_date' => $paymentDate,
                        'error' => $e->getMessage()
                    ]);
                }
            } else {
                $payDateFormatted = Carbon::now()->format('dmY');
                Log::warning('No payment date in Zoho, using today', [
                    'vendor' => $vendorName,
                    'payment_id' => $paymentId
                ]);
            }

            // Use payment reference - default to "Robert Todd" if not provided or is "Bank"
            $paymentReference = 'Robert Todd';
            if (!empty($referenceNumber) && strtolower($referenceNumber) !== 'bank') {
                $paymentReference = $referenceNumber;
            }

            // Build row matching NatWest Standard Domestic Payment format (Section 4.4)
            // Example: ,,,01,,,,,60100087654321,,,166.42,,16012025,,,,,,,,,151000,,,,,12345678,MR JOHN SMITH,,,,INVOICE 1234
            $row = [
                '',                         // H001
                '',                         // H002
                '',                         // H003
                '01',                       // T001 - Record type (01 = standard domestic payment)
                '',                         // T002
                '',                         // T003
                '',                         // T004
                '',                         // T005
                '',                         // T006
                '',                         // T007
                '',                         // T008
                '',                         // T009
                $debitAccountFull,          // T010 - Debit account identifier (your company account)
                '',                         // T011
                '',                         // T012
                '',                         // T013 - Payment currency (optional, will auto-default to GBP)
                $amountDecimal,             // T014 - Payment amount (e.g., 166.42)
                '',                         // T015
                $payDateFormatted,          // T016 - Date payment to Arrive/Credit Date (DDMMYYYY)
                '',                         // T017
                '',                         // T018
                '',                         // T019
                '',                         // T020
                '',                         // T021
                $sortCodeFormatted,         // T022 - Account with Bank identifier (beneficiary sort code)
                '',                         // T023
                '',                         // T024
                '',                         // T025
                '',                         // T026
                '',                         // T027
                $accountNumber,             // T028 - Beneficiary account Number
                '',                         // T029
                $vendorName,                // T030 - Beneficiary name
                '',                         // T031 - Beneficiary address line 2
                '',                         // T032 - Beneficiary address line 3
                '',                         // T033 - Beneficiary address line 4
                $paymentReference,          // T034 - Beneficiary reference
            ];
            
            // Add empty fields up to T082 (total 82 fields after H003)
            // We have 32 fields so far (T001-T034), need 48 more to reach T082
            for ($i = 0; $i < 48; $i++) {
                $row[] = '';
            }

            // Build CSV line manually without quotes (NatWest requires alphanumeric only, no quotes)
            $fileLines[] = implode(',', $row);
        }

        return implode("\n", $fileLines);
    }

    public function refresh()
    {
        // Refresh Zoho connection status
        $zohoService = app(ZohoService::class);
        $this->zohoConnected = $zohoService->isConnected();
    }

    public function render()
    {
        return view('livewire.finance.bank-file-generator');
    }
}

