<?php

namespace App\Http\Livewire\Development\Styles;

use App\Models\Gauge;
use App\Models\Styles;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Livewire\Attributes\Title;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\CacheFilterableComponent;

class StylesList extends CacheFilterableComponent
{
	use WithPagination;

    public $hideFilters = false;

    protected function filterKey(): string
    {
        return 'style_filters';
    }

    public $defaultFilters = [
        'season' => null,
        'department' => null,
        'category' => null,
        'gauge' => null,
        'customer' => null,
        'factory' => null,
        'search' => '',
        'paginate' => 10,
    ];

    public function mount()
    {
        $this->loadFilters();
    }

    #[Validate([
        'filters.season'     => ['nullable', 'exists:seasons,id'],
        'filters.department' => ['nullable', 'exists:departments,id'],
        'filters.category'   => ['nullable', 'in:ladies,mens,childrens,accessories'],
        'filters.gauge'      => ['nullable', 'exists:gauges,id'],
        'filters.customer'   => ['nullable', 'exists:customers,id'],
        'filters.factory'    => ['nullable', 'exists:suppliers,id'],
        'filters.search'     => ['nullable', 'string', 'max:30'],
        'filters.paginate'   => ['required', 'numeric', 'min:10', 'max:1000'],
    ])]

	#[Computed()]
	public function customers(){
		return Customer::allCached();
	}
	#[Computed()]
	public function factories(){
		return Suppliers::whereHas('style_versions')->get();
	}
	#[Computed()]
	public function seasons(){
		return Seasons::allCached();
	}
	#[Computed()]
	public function departments(){
		return Departments::allCached();
	}
	#[Computed()]
	public function gauges(){
		return Gauge::allCached();
	}

	public $editID = 0;

	#[On('refreshData')]
    public function refreshData()
    {
        // Refresh to show updated style data
        $this->resetPage();
    }

	#[On('refresh-styles')]
    public function render()
    {
        Gate::authorize('style:read');

        $styles = $this->getStylesQuery()
            ->paginate($this->filters['paginate'] ?? 10);

        $styleSummaries = $styles->getCollection()->mapWithKeys(function ($style) {
            $colourways = $style->style_versions
                ->flatMap(fn($sv) => $sv->colourways->map(fn($cw) => [
                    'id' => $cw->id,
                    'name' => $cw->name,
                    'version' => $sv->name,
                ]));

            $visible = $colourways->take(5);
            $remaining = max(0, $colourways->count() - 5);

            return [$style->id => [
                'visible' => $visible,
                'remaining' => $remaining,
            ]];
        });

        return view('livewire.development.styles.styles-list', [
            'styles' => $styles,
            'styleSummaries' => $styleSummaries,
        ]);
    }

    public function delete($id){
        Styles::find($id)->delete();
    }

	#[On('close-modal')]
	public function close(){
		$this->editID = 0;
	}

    private function getStylesQuery()
    {
        return Styles::with([
            'firstImageColourway',
            'style_versions:id,styles_id,name',
            'style_versions.colourways:id,style_versions_id,name',
            'customers:id,name',
            'seasons:id,description',
            'designs:id,description,gauge_id,factory_id'
        ])
        ->select([
            'styles.id',
            'styles.customer_ref',
            'styles.seasons_id',
            'styles.category',
            'styles.customers_id',
            'styles.departments_id',
            'styles.designs_id',
            'styles.created_at',
        ])
        ->when($this->filters['search'], fn ($query) => $query->search($this->filters['search']))
        ->when($this->filters['season'], fn ($query) => $query->where('styles.seasons_id', $this->filters['season']))
        ->when($this->filters['department'], fn ($query) => $query->where('styles.departments_id', $this->filters['department']))
        ->when($this->filters['category'], fn ($query) => $query->where('styles.category', $this->filters['category']))
        ->when($this->filters['customer'], fn ($query) => $query->where('styles.customers_id', $this->filters['customer']))
        ->when($this->filters['gauge'], fn ($query) => $query->whereHas('style_versions', fn ($q) => $q->where('gauge_id', $this->filters['gauge'])))
        ->when($this->filters['factory'], fn ($query) => $query->whereHas('style_versions', fn ($q) => $q->where('factory_id', $this->filters['factory'])))
        ->orderBy('styles.created_at', 'desc');
    }

}
