<?php

namespace App\Http\Livewire\Development\Styles;

use App\Models\Yarn;
use App\Models\Gauge;
use App\Models\Design;
use App\Models\Intake;
use App\Models\Styles;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Departments;
use App\Models\YarnColours;
use Livewire\WithFileUploads;
use Livewire\Attributes\On;
use App\Models\CommodityCodes;
use Livewire\Attributes\Computed;
use App\Models\CustomerOrderLines;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Gate;
use App\Livewire\Forms\StyleEditFormNew;

class StyleEditNew extends BaseComponent
{
    use WithFileUploads;

    public StyleEditFormNew $form;
    public $viewOnly = false;
    public $selectedVersion = 0;
    public $selectedColourway = 0;
    public $designsPicker = 0;
    public $commodityPicker = 0;
    public $commodity_search = "";
    public $new = false;
    public $commissionView;

    public function updated($propertyName, $value)
    {
        if (preg_match('/^form\.style_versions\.\d+\.colourways\.\d+\.colourway_yarns\.\d+\.percentage$/', $propertyName)) {
            if ($value === '' || is_null($value)) {
                data_set($this, $propertyName, 0);
            }
        }

        if (preg_match('/^form\.style_versions\.(\d+)\.colourways\.(\d+)\.colourway_yarns\.(\d+)\.yarn_id$/', $propertyName, $matches)) {
            [$_, $svIndex, $cwIndex, $cyIndex] = $matches;

            // Blank the corresponding yarn_id
            $this->form->style_versions[$svIndex]['colourways'][$cwIndex]['colourway_yarns'][$cyIndex]['yarn_colours_id'] = null;
        }
    }

    public function mount($styleId = null)
    {
        // Instantiate the form object.
        $this->form = new StyleEditFormNew($this, []);

        // If styleId is provided (standalone page), load the style
        if ($styleId) {
            $this->loadStyle($styleId);
        }
    }

    #[On('edit-style')]
    public function editStyle($style)
    {
        $this->loadStyle($style);
    }

    protected function loadStyle($styleId)
    {
        // Reset state
        $this->selectedVersion = 0;
        $this->selectedColourway = 0;
        $this->viewOnly = false;

        // Load the style record (with relationships) from the database.
        $style = Styles::with([
            'designs',
            'seasons',
            'customers',
            'style_versions.colourways.colourway_yarns.yarn_colours.yarn.suppliers',
            'style_versions.factories',
            'style_versions.gauges',
        ])->find($styleId);

        if ($style){
            $style->description = $style->designs->description;
            foreach($style->style_versions as $sv=>$styleVersion){
                foreach($styleVersion->colourways as $c=>$colourway){
                    foreach($colourway->colourway_yarns as $cy=>$colourway_yarn){
                        $style->style_versions[$sv]->colourways[$c]->colourway_yarns[$cy]->suppliers_id = $colourway_yarn->yarn_colours->yarn->suppliers->id;
                        $style->style_versions[$sv]->colourways[$c]->colourway_yarns[$cy]->yarn_id = $colourway_yarn->yarn_colours->yarn->id;
                    }
                }
            }
        } else{
            return;
        }

        // Fill the form object with the style data.
        $this->form->fill($style->toArray());

        if ($style->seasons->locked && Gate::check('style:update')) {
            $this->viewOnly = true;
        }
    }

    // Computed properties for dropdowns
    #[Computed]
    public function departments()
    {
        return Departments::where('hidden', false)->get();
    }

    #[Computed]
    public function commodityCodes()
    {
        $codes = CommodityCodes::allCached();
        
        if (!empty($this->commodity_search)) {
            $keywords = preg_split('/\s+/', $this->commodity_search, -1, PREG_SPLIT_NO_EMPTY);

            $codes = $codes->filter(function ($code) use ($keywords) {
                // Combine searchable fields into one string for matching
                $searchable = implode(' ', [
                    (string) $code->id,
                    (string) $code->Gender,
                    (string) $code->Type,
                    (string) $code->Composition,
                ]);
                
                // All keywords must appear in the combined searchable text
                foreach ($keywords as $term) {
                    if (stripos($searchable, $term) === false) {
                        return false;
                    }
                }
                return true;
            });
        }
        
        // Sort by order column, then by description
        return $codes->sortBy([
            ['order', 'asc'],
            ['description', 'asc'],
        ]);
    }

    #[Computed]
    public function designs()
    {
        return Design::allCached();
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::where('locked', false)->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::allCached();
    }

    #[Computed]
    public function gauges()
    {
        return Gauge::allCached();
    }

    #[Computed]
    public function intakes()
    {
        return Intake::get();
    }

    #[Computed]
    public function factories()
    {
        return Suppliers::allCached();
    }

    #[Computed]
    public function spinners()
    {
        return Suppliers::whereHas('yarns')->get();
    }

    #[Computed]
    public function yarns()
    {
        $supplierIds = collect();
        foreach ($this->form->style_versions as $version) {
            foreach ($version['colourways'] as $cw) {
                foreach ($cw['colourway_yarns'] as $yarn) {
                    $supplierIds->push($yarn['suppliers_id']);
                }
            }
        }
        $supplierIds = $supplierIds->unique()->toArray();

        return Yarn::select('yarns.id', 'yarns.suppliers_id', 'yarns.description', 'counts.count')
            ->whereIn('suppliers_id', $supplierIds)
            ->join('counts', 'counts.id', '=', 'yarns.counts_id')
            ->get();
    }

    #[Computed]
    public function colours()
    {
        $yarnIds = collect();
        foreach ($this->form->style_versions as $version) {
            foreach ($version['colourways'] as $cw) {
                foreach ($cw['colourway_yarns'] as $yarn) {
                    $yarnIds->push($yarn['yarn_id']);
                }
            }
        }
        $yarnIds = $yarnIds->unique()->toArray();

        return YarnColours::whereIn('yarn_id', $yarnIds)->get();
    }

    public function save()
    {
        if (!$this->viewOnly) {
            Gate::authorize('style:update');
            $this->form->update();
        }
        $this->dispatch('saved');
    }

    public function newStyle()
    {
        Gate::authorize('style:update');
        $this->new = true;
        $this->form = new StyleEditFormNew();
    }

    public function getYarnPercentageTotal($colourway)
    {
        return collect($colourway['colourway_yarns'] ?? [])->sum('percentage');
    }

    public function addVersion()
    {
        $this->form->addVersion();
    }

    public function removeVersion($index)
    {
        $colourways = $this->form->style_versions[$index]['colourways'] ?? [];

        foreach ($colourways as $colourway) {
            if (!empty($colourway['id'])) {
                $hasOrders = CustomerOrderLines::where('colourways_id', $colourway['id'])->exists();

                if ($hasOrders) {
                    session()->flash('message', 'One or more colourways in this version has customer orders attached and cannot be removed.');
                    session()->flash('alert-class', 'alert-danger');
                    return;
                }
            }
        }

        $this->form->removeVersion($index);
        $this->selectedVersion = 0;
        $this->selectedColourway = 0;
    }

    public function cloneVersion($index)
    {
        $this->form->cloneVersion($index);
    }

    public function addColourway($versionIndex)
    {
        $this->form->addColourway($versionIndex);
    }

    public function removeColourway($versionIndex, $colourwayIndex)
    {
        $colourway = $this->form->style_versions[$versionIndex]['colourways'][$colourwayIndex] ?? [];

        if (!empty($colourway['id'])) {
            $hasOrders = CustomerOrderLines::where('colourways_id', $colourway['id'])->exists();

            if ($hasOrders) {
                session()->flash('message', 'This colourway has customer orders attached and cannot be removed.');
                session()->flash('alert-class', 'alert-danger');
                return;
            }
        }

        $this->form->removeColourway($versionIndex, $colourwayIndex);
        if($colourwayIndex == $this->selectedColourway){
            $this->selectedColourway = 0;
        }
    }

    public function cloneColourway($versionIndex, $colourwayIndex)
    {
        $this->form->cloneColourway($versionIndex, $colourwayIndex);
    }

    public function addYarn($versionIndex, $colourwayIndex)
    {
        $this->form->addYarn($versionIndex, $colourwayIndex);
    }

    public function removeYarn($versionIndex, $colourwayIndex, $yarnIndex)
    {
        $this->form->removeYarn($versionIndex, $colourwayIndex, $yarnIndex);
    }

    public function cloneYarn($versionIndex, $colourwayIndex, $yarnIndex)
    {
        $this->form->cloneYarn($versionIndex, $colourwayIndex, $yarnIndex);
    }

    public function setCommodityCode($code){
        $this->form->setCommodityCode($code);
        $this->commodityPicker = false;
    }

    public function setDesigns($design){
        $this->form->setDesigns($design);
        $this->designsPicker = false;
    }

    public function render()
    {
        Gate::authorize('style:read');
        return view('livewire.development.styles.style-edit-new')->layout('layouts.nosidebar');
    }
}
