<?php

namespace App\Http\Livewire\Development\Styles;

use App\Models\Styles;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\QueryException;

class CloneStyle extends Component
{

    public $style = NULL;

    public $clone = [];

    public $existingStyle = NULL;

    public $showWarning = false;


    #[Computed()]
	public function departments(){
		return Departments::where('hidden', false)->get();
	}
	#[Computed()]
	public function seasons(){
		return Seasons::where('locked', FALSE)->get();
	}
	#[Computed()]
	public function customers(){
		return Customer::allCached();
	}

    public function mount(){
        $this->style = NULL;
    }

    public function render()
    {
        return view('livewire.development.styles.clone-style');
    }

    #[On('clone-style')]
    public function open($style){
        $this->style = Styles::find($style);
        $this->clone = [
            'departments_id' => $this->style->departments_id,
            'seasons_id' => "",
            'customers_id' => $this->style->customers_id,
        ];
        $this->existingStyle = NULL;
        $this->showWarning = false;
    }

    /**
     * Check if a style with the same design, season, department, and customer already exists.
     */
    public function checkForExistingStyle(): ?Styles
    {
        if (empty($this->clone['seasons_id']) || empty($this->clone['departments_id']) || empty($this->clone['customers_id'])) {
            return null;
        }

        return Styles::where('designs_id', $this->style->designs_id)
            ->where('seasons_id', $this->clone['seasons_id'])
            ->where('departments_id', $this->clone['departments_id'])
            ->where('customers_id', $this->clone['customers_id'])
            ->first();
    }

    public function save(){
        $validated = $this->validate([
            'clone.departments_id' => 'required|numeric|min:0',
            'clone.seasons_id' => 'required|numeric|min:0',
            'clone.customers_id' => 'required|numeric|min:0',
        ]);
        
        foreach($this->clone as $c=>$clone){
            $this->clone[$c] = trim(strip_tags($clone));
        }
        
        // Check if no changes were made (same values as original)
        if( $this->clone['departments_id'] == $this->style->departments_id &&
            $this->clone['seasons_id'] == $this->style->seasons_id &&
            $this->clone['customers_id'] == $this->style->customers_id){
            session()->flash('message', 'Cannot clone to the same location. Please select a different season, department, or customer.');
            session()->flash('alert-class', 'alert-danger');
            return;
        }

        // Check for existing style at the target location
        $this->existingStyle = $this->checkForExistingStyle();
        
        if ($this->existingStyle) {
            $seasonName = $this->seasons->firstWhere('id', $this->clone['seasons_id'])?->description ?? 'Unknown';
            $customerName = $this->customers->firstWhere('id', $this->clone['customers_id'])?->name ?? 'Unknown';
            $departmentName = $this->departments->firstWhere('id', $this->clone['departments_id'])?->description ?? 'Unknown';
            
            session()->flash('message', "A style with this design already exists for {$customerName} in {$seasonName} ({$departmentName}). Style reference: {$this->existingStyle->customer_ref}");
            session()->flash('alert-class', 'alert-warning');
            $this->showWarning = true;
            return;
        }

        $this->performClone();
    }

    protected function performClone()
    {
        try {
            $this->style->fill($this->clone);
            $this->style->duplicate();
            session()->flash('message', 'Style cloned successfully.');
            session()->flash('alert-class', 'alert-success');
            $this->showWarning = false;
            $this->existingStyle = NULL;
            $this->dispatch('close-modal');
        } catch (QueryException $e) {
            // Handle database errors (e.g., unique constraint violations)
            if (str_contains($e->getMessage(), 'Duplicate entry') || str_contains($e->getMessage(), 'UNIQUE constraint')) {
                session()->flash('message', 'A style with this combination already exists. The clone could not be created.');
                session()->flash('alert-class', 'alert-danger');
            } else {
                session()->flash('message', 'An error occurred while cloning the style. Please try again.');
                session()->flash('alert-class', 'alert-danger');
                \Log::error('Style clone error: ' . $e->getMessage());
            }
        } catch (\Exception $e) {
            session()->flash('message', 'An unexpected error occurred. Please try again.');
            session()->flash('alert-class', 'alert-danger');
            \Log::error('Style clone error: ' . $e->getMessage());
        }
    }

    #[On('close-modal')]
    public function close(){
        $this->reset(['style', 'clone', 'existingStyle', 'showWarning']);
    }
}
